#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
/*
 * S3BlobStore.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/S3BlobStore.h"

#include "fdbclient/ClientKnobs.h"
#include "fdbclient/Knobs.h"
#include "flow/FastRef.h"
#include "flow/IConnection.h"
#include "flow/Trace.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "md5/md5.h"
#include "libb64/encode.h"
#include "fdbclient/sha1/SHA1.h"
#include <climits>
#include <iostream>
#include <time.h>
#include <iomanip>
#include <openssl/sha.h>
#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/hex.hpp>
#include "flow/IAsyncFile.h"
#include "flow/Hostname.h"
#include "flow/UnitTest.h"
#include "rapidxml/rapidxml.hpp"
#ifdef WITH_AWS_BACKUP
#include "fdbclient/FDBAWSCredentialsProvider.h"
#endif

#include "flow/actorcompiler.h" // has to be last include

using namespace rapidxml;

json_spirit::mObject S3BlobStoreEndpoint::Stats::getJSON() {
	json_spirit::mObject o;

	o["requests_failed"] = requests_failed;
	o["requests_successful"] = requests_successful;
	o["bytes_sent"] = bytes_sent;

	return o;
}

S3BlobStoreEndpoint::Stats S3BlobStoreEndpoint::Stats::operator-(const Stats& rhs) {
	Stats r;
	r.requests_failed = requests_failed - rhs.requests_failed;
	r.requests_successful = requests_successful - rhs.requests_successful;
	r.bytes_sent = bytes_sent - rhs.bytes_sent;
	return r;
}

S3BlobStoreEndpoint::Stats S3BlobStoreEndpoint::s_stats;
std::unique_ptr<S3BlobStoreEndpoint::BlobStats> S3BlobStoreEndpoint::blobStats;
Future<Void> S3BlobStoreEndpoint::statsLogger = Never();

std::unordered_map<BlobStoreConnectionPoolKey, Reference<S3BlobStoreEndpoint::ConnectionPoolData>>
    S3BlobStoreEndpoint::globalConnectionPool;

S3BlobStoreEndpoint::BlobKnobs::BlobKnobs() {
	secure_connection = 1;
	connect_tries = CLIENT_KNOBS->BLOBSTORE_CONNECT_TRIES;
	connect_timeout = CLIENT_KNOBS->BLOBSTORE_CONNECT_TIMEOUT;
	max_connection_life = CLIENT_KNOBS->BLOBSTORE_MAX_CONNECTION_LIFE;
	request_tries = CLIENT_KNOBS->BLOBSTORE_REQUEST_TRIES;
	request_timeout_min = CLIENT_KNOBS->BLOBSTORE_REQUEST_TIMEOUT_MIN;
	requests_per_second = CLIENT_KNOBS->BLOBSTORE_REQUESTS_PER_SECOND;
	concurrent_requests = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_REQUESTS;
	list_requests_per_second = CLIENT_KNOBS->BLOBSTORE_LIST_REQUESTS_PER_SECOND;
	write_requests_per_second = CLIENT_KNOBS->BLOBSTORE_WRITE_REQUESTS_PER_SECOND;
	read_requests_per_second = CLIENT_KNOBS->BLOBSTORE_READ_REQUESTS_PER_SECOND;
	delete_requests_per_second = CLIENT_KNOBS->BLOBSTORE_DELETE_REQUESTS_PER_SECOND;
	multipart_max_part_size = CLIENT_KNOBS->BLOBSTORE_MULTIPART_MAX_PART_SIZE;
	multipart_min_part_size = CLIENT_KNOBS->BLOBSTORE_MULTIPART_MIN_PART_SIZE;
	concurrent_uploads = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_UPLOADS;
	concurrent_lists = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_LISTS;
	concurrent_reads_per_file = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_READS_PER_FILE;
	concurrent_writes_per_file = CLIENT_KNOBS->BLOBSTORE_CONCURRENT_WRITES_PER_FILE;
	enable_read_cache = CLIENT_KNOBS->BLOBSTORE_ENABLE_READ_CACHE;
	read_block_size = CLIENT_KNOBS->BLOBSTORE_READ_BLOCK_SIZE;
	read_ahead_blocks = CLIENT_KNOBS->BLOBSTORE_READ_AHEAD_BLOCKS;
	read_cache_blocks_per_file = CLIENT_KNOBS->BLOBSTORE_READ_CACHE_BLOCKS_PER_FILE;
	max_send_bytes_per_second = CLIENT_KNOBS->BLOBSTORE_MAX_SEND_BYTES_PER_SECOND;
	max_recv_bytes_per_second = CLIENT_KNOBS->BLOBSTORE_MAX_RECV_BYTES_PER_SECOND;
	max_delay_retryable_error = CLIENT_KNOBS->BLOBSTORE_MAX_DELAY_RETRYABLE_ERROR;
	max_delay_connection_failed = CLIENT_KNOBS->BLOBSTORE_MAX_DELAY_CONNECTION_FAILED;
	sdk_auth = false;
	enable_object_integrity_check = CLIENT_KNOBS->BLOBSTORE_ENABLE_OBJECT_INTEGRITY_CHECK;
	global_connection_pool = CLIENT_KNOBS->BLOBSTORE_GLOBAL_CONNECTION_POOL;
}

bool S3BlobStoreEndpoint::BlobKnobs::set(StringRef name, int value) {
#define TRY_PARAM(n, sn)                                                                                               \
	if (name == #n || name == #sn) {                                                                                   \
		n = value;                                                                                                     \
		return true;                                                                                                   \
	}
	TRY_PARAM(secure_connection, sc)
	TRY_PARAM(connect_tries, ct);
	TRY_PARAM(connect_timeout, cto);
	TRY_PARAM(max_connection_life, mcl);
	TRY_PARAM(request_tries, rt);
	TRY_PARAM(request_timeout_min, rtom);
	// TODO: For backward compatibility because request_timeout was renamed to request_timeout_min
	if (name == "request_timeout"_sr || name == "rto"_sr) {
		request_timeout_min = value;
		return true;
	}
	TRY_PARAM(requests_per_second, rps);
	TRY_PARAM(list_requests_per_second, lrps);
	TRY_PARAM(write_requests_per_second, wrps);
	TRY_PARAM(read_requests_per_second, rrps);
	TRY_PARAM(delete_requests_per_second, drps);
	TRY_PARAM(concurrent_requests, cr);
	TRY_PARAM(multipart_max_part_size, maxps);
	TRY_PARAM(multipart_min_part_size, minps);
	TRY_PARAM(multipart_retry_delay_ms, mrd);
	TRY_PARAM(concurrent_uploads, cu);
	TRY_PARAM(concurrent_lists, cl);
	TRY_PARAM(concurrent_reads_per_file, crpf);
	TRY_PARAM(concurrent_writes_per_file, cwpf);
	TRY_PARAM(enable_read_cache, erc);
	TRY_PARAM(read_block_size, rbs);
	TRY_PARAM(read_ahead_blocks, rab);
	TRY_PARAM(read_cache_blocks_per_file, rcb);
	TRY_PARAM(max_send_bytes_per_second, sbps);
	TRY_PARAM(max_recv_bytes_per_second, rbps);
	TRY_PARAM(max_delay_retryable_error, dre);
	TRY_PARAM(max_delay_connection_failed, dcf);
	TRY_PARAM(sdk_auth, sa);
	TRY_PARAM(enable_object_integrity_check, eoic);
	TRY_PARAM(global_connection_pool, gcp);
#undef TRY_PARAM
	return false;
}

// Returns an S3 Blob URL parameter string that specifies all of the non-default options for the endpoint using option
// short names.
std::string S3BlobStoreEndpoint::BlobKnobs::getURLParameters() const {
	static BlobKnobs defaults;
	std::string r;
#define _CHECK_PARAM(n, sn)                                                                                            \
	if (n != defaults.n) {                                                                                             \
		r += format("%s%s=%d", r.empty() ? "" : "&", #sn, n);                                                          \
	}
	_CHECK_PARAM(secure_connection, sc);
	_CHECK_PARAM(connect_tries, ct);
	_CHECK_PARAM(connect_timeout, cto);
	_CHECK_PARAM(max_connection_life, mcl);
	_CHECK_PARAM(request_tries, rt);
	_CHECK_PARAM(request_timeout_min, rto);
	_CHECK_PARAM(requests_per_second, rps);
	_CHECK_PARAM(list_requests_per_second, lrps);
	_CHECK_PARAM(write_requests_per_second, wrps);
	_CHECK_PARAM(read_requests_per_second, rrps);
	_CHECK_PARAM(delete_requests_per_second, drps);
	_CHECK_PARAM(concurrent_requests, cr);
	_CHECK_PARAM(multipart_max_part_size, maxps);
	_CHECK_PARAM(multipart_min_part_size, minps);
	_CHECK_PARAM(concurrent_uploads, cu);
	_CHECK_PARAM(concurrent_lists, cl);
	_CHECK_PARAM(concurrent_reads_per_file, crpf);
	_CHECK_PARAM(concurrent_writes_per_file, cwpf);
	_CHECK_PARAM(enable_read_cache, erc);
	_CHECK_PARAM(read_block_size, rbs);
	_CHECK_PARAM(read_ahead_blocks, rab);
	_CHECK_PARAM(read_cache_blocks_per_file, rcb);
	_CHECK_PARAM(max_send_bytes_per_second, sbps);
	_CHECK_PARAM(max_recv_bytes_per_second, rbps);
	_CHECK_PARAM(sdk_auth, sa);
	_CHECK_PARAM(enable_object_integrity_check, eoic);
	_CHECK_PARAM(global_connection_pool, gcp);
	_CHECK_PARAM(max_delay_retryable_error, dre);
	_CHECK_PARAM(max_delay_connection_failed, dcf);
#undef _CHECK_PARAM
	return r;
}

std::string guessRegionFromDomain(std::string domain) {
	static const std::vector<const char*> knownServices = { "s3.", "cos.", "oss-", "obs." };
	boost::algorithm::to_lower(domain);

	for (int i = 0; i < knownServices.size(); ++i) {
		const char* service = knownServices[i];

		std::size_t p = domain.find(service);
		if (p == std::string::npos || (p >= 1 && domain[p - 1] != '.')) {
			// eg. 127.0.0.1, example.com, s3-service.example.com, mys3.example.com
			continue;
		}

		StringRef h = StringRef(domain).substr(p);

		if (!h.startsWith("oss-"_sr)) {
			h.eat(service); // ignore s3 service
		}

		return h.eat(".").toString();
	}

	return "";
}

Reference<S3BlobStoreEndpoint> S3BlobStoreEndpoint::fromString(const std::string& url,
                                                               const Optional<std::string>& proxy,
                                                               std::string* resourceFromURL,
                                                               std::string* error,
                                                               ParametersT* ignored_parameters) {
	if (resourceFromURL)
		resourceFromURL->clear();

	try {
		// Replace HTML-encoded ampersands with raw ampersands
		std::string decoded_url = url;
		size_t pos = 0;
		while ((pos = decoded_url.find("&amp;", pos)) != std::string::npos) {
			decoded_url.replace(pos, 5, "&");
			pos += 1;
		}

		// Also handle double-encoded ampersands
		pos = 0;
		while ((pos = decoded_url.find("&amp;amp;", pos)) != std::string::npos) {
			decoded_url.replace(pos, 9, "&");
			pos += 1;
		}

		StringRef t(decoded_url);
		StringRef prefix = t.eat("://");
		if (prefix != "blobstore"_sr)
			throw format("Invalid blobstore URL prefix '%s'", prefix.toString().c_str());

		Optional<std::string> proxyHost, proxyPort;
		if (proxy.present()) {
			StringRef proxyRef(proxy.get());
			if (proxy.get().find("://") != std::string::npos) {
				StringRef proxyPrefix = proxyRef.eat("://");
				if (proxyPrefix != "http"_sr) {
					throw format("Invalid proxy URL prefix '%s'. Either don't use a prefix, or use http://",
					             proxyPrefix.toString().c_str());
				}
			}
			std::string proxyBody = proxyRef.eat().toString();
			if (!Hostname::isHostname(proxyBody) && !NetworkAddress::parseOptional(proxyBody).present()) {
				throw format("'%s' is not a valid value for proxy. Format should be either IP:port or host:port.",
				             proxyBody.c_str());
			}
			StringRef p(proxyBody);
			proxyHost = p.eat(":").toString();
			proxyPort = p.eat().toString();
		}

		Optional<StringRef> cred;
		if (url.find("@") != std::string::npos) {
			cred = t.eat("@");
		}
		uint8_t foundSeparator = 0;
		StringRef hostPort = t.eatAny("/?", &foundSeparator);
		StringRef resource;
		if (foundSeparator == '/') {
			resource = t.eat("?");
		}

		// hostPort is at least a host or IP address, optionally followed by :portNumber or :serviceName
		StringRef h(hostPort);
		StringRef host = h.eat(":");
		if (host.size() == 0)
			throw std::string("host cannot be empty");

		StringRef service = h.eat();

		std::string region = guessRegionFromDomain(host.toString());

		BlobKnobs knobs;
		HTTP::Headers extraHeaders;
		while (1) {
			StringRef name = t.eat("=");
			if (name.size() == 0)
				break;
			StringRef value = t.eat("&");

			// Special case for header
			if (name == "header"_sr) {
				StringRef originalValue = value;
				StringRef headerFieldName = value.eat(":");
				StringRef headerFieldValue = value;
				if (headerFieldName.size() == 0 || headerFieldValue.size() == 0) {
					throw format("'%s' is not a valid value for '%s' parameter.  Format is <FieldName>:<FieldValue> "
					             "where strings are not empty.",
					             originalValue.toString().c_str(),
					             name.toString().c_str());
				}
				std::string& fieldValue = extraHeaders[headerFieldName.toString()];
				// RFC 2616 section 4.2 says header field names can repeat but only if it is valid to concatenate their
				// values with comma separation
				if (!fieldValue.empty()) {
					fieldValue.append(",");
				}
				fieldValue.append(headerFieldValue.toString());
				continue;
			}

			// overwrite s3 region from parameter
			if (name == "region"_sr) {
				region = value.toString();
				continue;
			}

			// See if the parameter is a knob
			// First try setting a dummy value (all knobs are currently numeric) just to see if this parameter is known
			// to S3BlobStoreEndpoint. If it is, then we will set it to a good value or throw below, so the dummy set
			// has no bad side effects.
			bool known = knobs.set(name, 0);

			// If the parameter is not known to S3BlobStoreEndpoint then throw unless there is an ignored_parameters set
			// to add it to
			if (!known) {
				if (ignored_parameters == nullptr) {
					throw format("%s is not a valid parameter name", name.toString().c_str());
				}
				(*ignored_parameters)[name.toString()] = value.toString();
				continue;
			}

			// The parameter is known to S3BlobStoreEndpoint so it must be numeric and valid.
			char* valueEnd = nullptr;
			std::string s = value.toString();
			long int ivalue = strtol(s.c_str(), &valueEnd, 10);
			if (*valueEnd || (ivalue == 0 && s != "0") ||
			    (((ivalue == LONG_MAX) || (ivalue == LONG_MIN)) && errno == ERANGE))
				throw format("%s is not a valid value for %s", s.c_str(), name.toString().c_str());

			// It should not be possible for this set to fail now since the dummy set above had to have worked.
			ASSERT(knobs.set(name, ivalue));
		}

		if (resourceFromURL != nullptr)
			*resourceFromURL = resource.toString();

		Optional<S3BlobStoreEndpoint::Credentials> creds;
		if (cred.present()) {
			StringRef c(cred.get());
			StringRef key = c.eat(":");
			StringRef secret = c.eat(":");
			StringRef securityToken = c.eat();
			creds = S3BlobStoreEndpoint::Credentials{ key.toString(), secret.toString(), securityToken.toString() };
		}

		if (region.empty() && CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER) {
			throw std::string(
			    "Failed to get region from host or parameter in url, region is required for aws v4 signature");
		}

		return makeReference<S3BlobStoreEndpoint>(
		    host.toString(), service.toString(), region, proxyHost, proxyPort, creds, knobs, extraHeaders);

	} catch (std::string& err) {
		if (error != nullptr)
			*error = err;
		TraceEvent(SevWarnAlways, "S3BlobStoreEndpointBadURL")
		    .suppressFor(60)
		    .detail("Description", err)
		    .detail("Format", getURLFormat())
		    .detail("URL", url);
		throw backup_invalid_url();
	}
}

std::string S3BlobStoreEndpoint::getResourceURL(std::string resource, std::string params) const {
	std::string hostPort = host;
	if (!service.empty()) {
		hostPort.append(":");
		hostPort.append(service);
	}

	// If secret isn't being looked up from credentials files then it was passed explicitly in the URL so show it here.
	std::string credsString;
	if (credentials.present()) {
		if (!lookupKey) {
			credsString = credentials.get().key;
		}
		if (!lookupSecret) {
			credsString += ":" + credentials.get().secret;
			if (!credentials.get().securityToken.empty()) {
				credsString += ":" + credentials.get().securityToken;
			}
		}
		credsString += "@";
	}

	std::string r = format("blobstore://%s%s/%s", credsString.c_str(), hostPort.c_str(), resource.c_str());

	// Get params that are deviations from knob defaults
	std::string knobParams = knobs.getURLParameters();
	if (!knobParams.empty()) {
		if (!params.empty()) {
			params.append("&");
		}
		params.append(knobParams);
	}

	for (const auto& [k, v] : extraHeaders) {
		if (!params.empty()) {
			params.append("&");
		}
		params.append("header=");
		params.append(k);
		params.append(":");
		params.append(v);
	}

	if (!params.empty()) {
		r.append("?").append(params);
	}

	return r;
}

std::string S3BlobStoreEndpoint::constructResourcePath(const std::string& bucket, const std::string& object) const {
	std::string resource;

	if (host.find(bucket + ".") != 0) {
		resource += std::string("/") + bucket; // not virtual hosting mode
	}

	if (!object.empty()) {
		// Don't add a slash if the object starts with one
		if (!object.starts_with("/")) {
			resource += "/";
		}
		resource += object;
	}

	return resource;
}

															#line 461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via bucketExists_impl()
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class BucketExists_implActor>
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BucketExists_implActorState {
															#line 468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BucketExists_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket)
															#line 477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("bucketExists_impl", reinterpret_cast<unsigned long>(this));

	}
	~BucketExists_implActorState() 
	{
		fdb_probe_actor_destroy("bucketExists_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateRead->getAllowance(1);
															#line 460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BucketExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BucketExists_implActor*>(this)->actor_wait_state = 1;
															#line 460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BucketExists_implActor, 0, Void >*>(static_cast<BucketExists_implActor*>(this)));
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BucketExists_implActorState();
		static_cast<BucketExists_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, "");
															#line 463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BucketExists_implActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BucketExists_implActor*>(this)->actor_wait_state = 2;
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BucketExists_implActor*>(this)));
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, "");
															#line 463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BucketExists_implActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<BucketExists_implActor*>(this)->actor_wait_state = 2;
															#line 466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BucketExists_implActor*>(this)));
															#line 563 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BucketExists_implActor*>(this)->actor_wait_state > 0) static_cast<BucketExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<BucketExists_implActor*>(this)->ActorCallback< BucketExists_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BucketExists_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevError, "S3ClientBucketExistsError") .detail("Bucket", bucket) .detail("Host", b->host) .errorUnsuppressed(e);
															#line 473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<BucketExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~BucketExists_implActorState(); static_cast<BucketExists_implActor*>(this)->destroy(); return 0; }
															#line 670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<BucketExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~BucketExists_implActorState();
		static_cast<BucketExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<BucketExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~BucketExists_implActorState(); static_cast<BucketExists_implActor*>(this)->destroy(); return 0; }
															#line 682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<BucketExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~BucketExists_implActorState();
		static_cast<BucketExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont3(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BucketExists_implActor*>(this)->actor_wait_state > 0) static_cast<BucketExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<BucketExists_implActor*>(this)->ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 462 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via bucketExists_impl()
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BucketExists_implActor final : public Actor<bool>, public ActorCallback< BucketExists_implActor, 0, Void >, public ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<BucketExists_implActor>, public BucketExists_implActorState<BucketExists_implActor> {
															#line 778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<BucketExists_implActor>::operator new;
	using FastAllocated<BucketExists_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(400876317269903616UL, 2868354188595055104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BucketExists_implActor, 0, Void >;
friend struct ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BucketExists_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<bool>(),
		   BucketExists_implActorState<BucketExists_implActor>(b, bucket),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("bucketExists_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14807564591002974976UL, 18309392055811966464UL);
		ActorExecutionContextHelper __helper(static_cast<BucketExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("bucketExists_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("bucketExists_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BucketExists_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BucketExists_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<bool> bucketExists_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket ) {
															#line 459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<bool>(new BucketExists_implActor(b, bucket));
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<bool> S3BlobStoreEndpoint::bucketExists(std::string const& bucket) {
	return bucketExists_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket);
}

															#line 839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via objectExists_impl()
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ObjectExists_implActor>
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectExists_implActorState {
															#line 846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectExists_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("objectExists_impl", reinterpret_cast<unsigned long>(this));

	}
	~ObjectExists_implActorState() 
	{
		fdb_probe_actor_destroy("objectExists_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateRead->getAllowance(1);
															#line 482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 1;
															#line 482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ObjectExists_implActor, 0, Void >*>(static_cast<ObjectExists_implActor*>(this)));
															#line 879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ObjectExists_implActorState();
		static_cast<ObjectExists_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 2;
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectExists_implActor*>(this)));
															#line 913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 2;
															#line 487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectExists_implActor*>(this)));
															#line 933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectExists_implActor*>(this)->ActorCallback< ObjectExists_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ObjectExists_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~ObjectExists_implActorState(); static_cast<ObjectExists_implActor*>(this)->destroy(); return 0; }
															#line 1017 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~ObjectExists_implActorState();
		static_cast<ObjectExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectExists_implActor*>(this)->SAV<bool>::futures) { (void)(r->code == 200); this->~ObjectExists_implActorState(); static_cast<ObjectExists_implActor*>(this)->destroy(); return 0; }
															#line 1029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectExists_implActor*>(this)->SAV< bool >::value()) bool(r->code == 200);
		this->~ObjectExists_implActorState();
		static_cast<ObjectExists_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ObjectExists_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectExists_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectExists_implActor*>(this)->ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 485 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1122 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via objectExists_impl()
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectExists_implActor final : public Actor<bool>, public ActorCallback< ObjectExists_implActor, 0, Void >, public ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ObjectExists_implActor>, public ObjectExists_implActorState<ObjectExists_implActor> {
															#line 1127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ObjectExists_implActor>::operator new;
	using FastAllocated<ObjectExists_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14154593728276089856UL, 16485483446205967872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ObjectExists_implActor, 0, Void >;
friend struct ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectExists_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 1145 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<bool>(),
		   ObjectExists_implActorState<ObjectExists_implActor>(b, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("objectExists_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1440333905109885440UL, 8919820050798510080UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectExists_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("objectExists_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("objectExists_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ObjectExists_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ObjectExists_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<bool> objectExists_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& object ) {
															#line 481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<bool>(new ObjectExists_implActor(b, bucket, object));
															#line 1179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 490 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<bool> S3BlobStoreEndpoint::objectExists(std::string const& bucket, std::string const& object) {
	return objectExists_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via deleteObject_impl()
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class DeleteObject_implActor>
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteObject_implActorState {
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteObject_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 1206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteObject_impl", reinterpret_cast<unsigned long>(this));

	}
	~DeleteObject_implActorState() 
	{
		fdb_probe_actor_destroy("deleteObject_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateDelete->getAllowance(1);
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 1;
															#line 496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteObject_implActor, 0, Void >*>(static_cast<DeleteObject_implActor*>(this)));
															#line 1228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteObject_implActorState();
		static_cast<DeleteObject_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("DELETE", resource, headers, nullptr, 0, { 200, 204, 404 });
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 2;
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<DeleteObject_implActor*>(this)));
															#line 1262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("DELETE", resource, headers, nullptr, 0, { 200, 204, 404 });
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 2;
															#line 502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<DeleteObject_implActor*>(this)));
															#line 1282 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteObject_implActor*>(this)->ActorCallback< DeleteObject_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteObject_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 1366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarnAlways, "S3BlobStoreEndpointDeleteObjectMissing") .detail("Host", b->host) .detail("Bucket", bucket) .detail("Object", object);
															#line 1370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<DeleteObject_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteObject_implActorState(); static_cast<DeleteObject_implActor*>(this)->destroy(); return 0; }
															#line 1374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<DeleteObject_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteObject_implActorState();
		static_cast<DeleteObject_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 1386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarnAlways, "S3BlobStoreEndpointDeleteObjectMissing") .detail("Host", b->host) .detail("Bucket", bucket) .detail("Object", object);
															#line 1390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<DeleteObject_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteObject_implActorState(); static_cast<DeleteObject_implActor*>(this)->destroy(); return 0; }
															#line 1394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<DeleteObject_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteObject_implActorState();
		static_cast<DeleteObject_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteObject_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteObject_implActor*>(this)->ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via deleteObject_impl()
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteObject_implActor final : public Actor<Void>, public ActorCallback< DeleteObject_implActor, 0, Void >, public ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<DeleteObject_implActor>, public DeleteObject_implActorState<DeleteObject_implActor> {
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<DeleteObject_implActor>::operator new;
	using FastAllocated<DeleteObject_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10054888715282929408UL, 10359214054737948928UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteObject_implActor, 0, Void >;
friend struct ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteObject_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 1510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteObject_implActorState<DeleteObject_implActor>(b, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteObject_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14670167000419629568UL, 2229091376396350720UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteObject_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteObject_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteObject_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteObject_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> deleteObject_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& object ) {
															#line 495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new DeleteObject_implActor(b, bucket, object));
															#line 1544 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::deleteObject(std::string const& bucket, std::string const& object) {
	return deleteObject_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 1553 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via deleteRecursively_impl()
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class DeleteRecursively_implActor>
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteRecursively_implActorState {
															#line 1560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteRecursively_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& prefix,int* const& pNumDeleted,int64_t* const& pBytesDeleted) 
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   prefix(prefix),
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pNumDeleted(pNumDeleted),
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pBytesDeleted(pBytesDeleted),
															#line 525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resultStream(),
															#line 527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   done(b->listObjectsStream(bucket, resultStream, prefix, '/', std::numeric_limits<int>::max()))
															#line 1579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteRecursively_impl", reinterpret_cast<unsigned long>(this));

	}
	~DeleteRecursively_implActorState() 
	{
		fdb_probe_actor_destroy("deleteRecursively_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			done = map(done, [=](Void) mutable { resultStream.sendError(end_of_stream()); return Void(); });
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			deleteFutures = std::list<Future<Void>>();
															#line 1596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			try {
															#line 537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				;
															#line 1600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteRecursively_implActorState();
		static_cast<DeleteRecursively_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 1629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_end_of_stream)
															#line 1639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 567 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = done;
															#line 538 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 544 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		auto __when_expr_1 = resultStream.getFuture();
															#line 1672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<S3BlobStoreEndpoint::ListResult>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<S3BlobStoreEndpoint::ListResult>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
		static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 1;
															#line 540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteRecursively_implActor, 0, Void >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 544 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 1680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 1689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 1698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 1707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult const& list,int loopDepth) 
	{
															#line 545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& object : list.objects ) {
															#line 546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			deleteFutures.push_back(map(b->deleteObject(bucket, object.name), [=](Void) -> Void { if (pNumDeleted != nullptr) { ++*pNumDeleted; } if (pBytesDeleted != nullptr) { *pBytesDeleted += object.size; } return Void(); }));
															#line 1718 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult && list,int loopDepth) 
	{
															#line 545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& object : list.objects ) {
															#line 546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			deleteFutures.push_back(map(b->deleteObject(bucket, object.name), [=](Void) -> Void { if (pNumDeleted != nullptr) { ++*pNumDeleted; } if (pBytesDeleted != nullptr) { *pBytesDeleted += object.size; } return Void(); }));
															#line 1730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteRecursively_implActor*>(this)->ActorCallback< DeleteRecursively_implActor, 0, Void >::remove();
		static_cast<DeleteRecursively_implActor*>(this)->ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteRecursively_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(deleteFutures.size() > CLIENT_KNOBS->BLOBSTORE_CONCURRENT_REQUESTS))
															#line 1874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deleteFutures.front();
															#line 561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 2;
															#line 561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteRecursively_implActor, 2, Void >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 1887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 1909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 1918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteRecursively_implActor*>(this)->ActorCallback< DeleteRecursively_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeleteRecursively_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<DeleteRecursively_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteRecursively_implActorState(); static_cast<DeleteRecursively_implActor*>(this)->destroy(); return 0; }
															#line 2002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<DeleteRecursively_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteRecursively_implActorState();
		static_cast<DeleteRecursively_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(deleteFutures.size() > 0))
															#line 2021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = deleteFutures.front();
															#line 571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 3;
															#line 571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeleteRecursively_implActor, 3, Void >*>(static_cast<DeleteRecursively_implActor*>(this)));
															#line 2034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 572 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 2056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 572 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		deleteFutures.pop_front();
															#line 2065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state > 0) static_cast<DeleteRecursively_implActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteRecursively_implActor*>(this)->ActorCallback< DeleteRecursively_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DeleteRecursively_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DeleteRecursively_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string prefix;
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int* pNumDeleted;
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int64_t* pBytesDeleted;
															#line 525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PromiseStream<S3BlobStoreEndpoint::ListResult> resultStream;
															#line 527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Future<Void> done;
															#line 535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::list<Future<Void>> deleteFutures;
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via deleteRecursively_impl()
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DeleteRecursively_implActor final : public Actor<Void>, public ActorCallback< DeleteRecursively_implActor, 0, Void >, public ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >, public ActorCallback< DeleteRecursively_implActor, 2, Void >, public ActorCallback< DeleteRecursively_implActor, 3, Void >, public FastAllocated<DeleteRecursively_implActor>, public DeleteRecursively_implActorState<DeleteRecursively_implActor> {
															#line 2166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<DeleteRecursively_implActor>::operator new;
	using FastAllocated<DeleteRecursively_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17914909177932496384UL, 14294767750031632896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteRecursively_implActor, 0, Void >;
friend struct ActorSingleCallback< DeleteRecursively_implActor, 1, S3BlobStoreEndpoint::ListResult >;
friend struct ActorCallback< DeleteRecursively_implActor, 2, Void >;
friend struct ActorCallback< DeleteRecursively_implActor, 3, Void >;
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DeleteRecursively_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& prefix,int* const& pNumDeleted,int64_t* const& pBytesDeleted) 
															#line 2186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteRecursively_implActorState<DeleteRecursively_implActor>(b, bucket, prefix, pNumDeleted, pBytesDeleted),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14744632183853925888UL, 8951783522599013632UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteRecursively_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteRecursively_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteRecursively_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteRecursively_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteRecursively_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeleteRecursively_implActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> deleteRecursively_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& prefix, int* const& pNumDeleted, int64_t* const& pBytesDeleted ) {
															#line 520 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new DeleteRecursively_implActor(b, bucket, prefix, pNumDeleted, pBytesDeleted));
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::deleteRecursively(std::string const& bucket,
                                                    std::string prefix,
                                                    int* pNumDeleted,
                                                    int64_t* pBytesDeleted) {
	return deleteRecursively_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, prefix, pNumDeleted, pBytesDeleted);
}

															#line 2234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via createBucket_impl()
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class CreateBucket_implActor>
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class CreateBucket_implActorState {
															#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	CreateBucket_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   packets()
															#line 2252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("createBucket_impl", reinterpret_cast<unsigned long>(this));

	}
	~CreateBucket_implActorState() 
	{
		fdb_probe_actor_destroy("createBucket_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateWrite->getAllowance(1);
															#line 588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 1;
															#line 588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 0, Void >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CreateBucket_implActorState();
		static_cast<CreateBucket_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<bool> __when_expr_1 = b->bucketExists(bucket);
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 2;
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 1, bool >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<bool> __when_expr_1 = b->bucketExists(bucket);
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 2;
															#line 590 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 1, bool >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(bool const& exists,int loopDepth) 
	{
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!exists)
															#line 2404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource = b->constructResourcePath(bucket, "");
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers = HTTP::Headers();
															#line 595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string region = b->getRegion();
															#line 596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (region.empty())
															#line 2414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = b->doRequest("PUT", resource, headers, nullptr, 0, { 200, 409 });
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 3;
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				Standalone<StringRef> body( format("<CreateBucketConfiguration xmlns=\"http://s3.amazonaws.com/doc/2006-03-01/\">" "  <LocationConstraint>%s</LocationConstraint>" "</CreateBucketConfiguration>", region.c_str()));
															#line 605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketWriter pw(packets.getWriteBuffer(), nullptr, Unversioned());
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				pw.serializeBytes(body);
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = b->doRequest("PUT", resource, headers, &packets, body.size(), { 200, 409 });
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 4;
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(bool && exists,int loopDepth) 
	{
															#line 591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!exists)
															#line 2460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource = b->constructResourcePath(bucket, "");
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers = HTTP::Headers();
															#line 595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string region = b->getRegion();
															#line 596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (region.empty())
															#line 2470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = b->doRequest("PUT", resource, headers, nullptr, 0, { 200, 409 });
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 3;
															#line 597 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2481 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				Standalone<StringRef> body( format("<CreateBucketConfiguration xmlns=\"http://s3.amazonaws.com/doc/2006-03-01/\">" "  <LocationConstraint>%s</LocationConstraint>" "</CreateBucketConfiguration>", region.c_str()));
															#line 605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketWriter pw(packets.getWriteBuffer(), nullptr, Unversioned());
															#line 606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				pw.serializeBytes(body);
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = b->doRequest("PUT", resource, headers, &packets, body.size(), { 200, 409 });
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
				static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 4;
															#line 608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<CreateBucket_implActor*>(this)));
															#line 2501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(bool const& exists,int loopDepth) 
	{
		loopDepth = a_body1cont2(exists, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(bool && exists,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(exists), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 1, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 1, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 1, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 1, bool >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<CreateBucket_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CreateBucket_implActorState(); static_cast<CreateBucket_implActor*>(this)->destroy(); return 0; }
															#line 2591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<CreateBucket_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CreateBucket_implActorState();
		static_cast<CreateBucket_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont5(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont6(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CreateBucket_implActor*>(this)->actor_wait_state > 0) static_cast<CreateBucket_implActor*>(this)->actor_wait_state = 0;
		static_cast<CreateBucket_implActor*>(this)->ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue packets;
															#line 592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 2789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via createBucket_impl()
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class CreateBucket_implActor final : public Actor<Void>, public ActorCallback< CreateBucket_implActor, 0, Void >, public ActorCallback< CreateBucket_implActor, 1, bool >, public ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >, public ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >, public FastAllocated<CreateBucket_implActor>, public CreateBucket_implActorState<CreateBucket_implActor> {
															#line 2794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<CreateBucket_implActor>::operator new;
	using FastAllocated<CreateBucket_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4469729475057238784UL, 8177689174689850112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CreateBucket_implActor, 0, Void >;
friend struct ActorCallback< CreateBucket_implActor, 1, bool >;
friend struct ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >;
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	CreateBucket_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket) 
															#line 2814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   CreateBucket_implActorState<CreateBucket_implActor>(b, bucket),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("createBucket_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13190372849516335360UL, 14355671705917093888UL);
		ActorExecutionContextHelper __helper(static_cast<CreateBucket_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("createBucket_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("createBucket_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CreateBucket_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CreateBucket_implActor, 1, bool >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CreateBucket_implActor, 2, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CreateBucket_implActor, 3, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> createBucket_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket ) {
															#line 586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new CreateBucket_implActor(b, bucket));
															#line 2850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::createBucket(std::string const& bucket) {
	return createBucket_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket);
}

															#line 2859 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via objectSize_impl()
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ObjectSize_implActor>
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectSize_implActorState {
															#line 2866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectSize_implActorState(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 2877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("objectSize_impl", reinterpret_cast<unsigned long>(this));

	}
	~ObjectSize_implActorState() 
	{
		fdb_probe_actor_destroy("objectSize_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = b->requestRateRead->getAllowance(1);
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 1;
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ObjectSize_implActor, 0, Void >*>(static_cast<ObjectSize_implActor*>(this)));
															#line 2899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ObjectSize_implActorState();
		static_cast<ObjectSize_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 2;
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectSize_implActor*>(this)));
															#line 2933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = b->constructResourcePath(bucket, object);
															#line 623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = b->doRequest("HEAD", resource, headers, nullptr, 0, { 200, 404 });
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 2;
															#line 625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ObjectSize_implActor*>(this)));
															#line 2953 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectSize_implActor*>(this)->ActorCallback< ObjectSize_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ObjectSize_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 3037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 3041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectSize_implActor*>(this)->SAV<int64_t>::futures) { (void)(r->data.contentLen); this->~ObjectSize_implActorState(); static_cast<ObjectSize_implActor*>(this)->destroy(); return 0; }
															#line 3045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectSize_implActor*>(this)->SAV< int64_t >::value()) int64_t(r->data.contentLen);
		this->~ObjectSize_implActorState();
		static_cast<ObjectSize_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 3057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 3061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ObjectSize_implActor*>(this)->SAV<int64_t>::futures) { (void)(r->data.contentLen); this->~ObjectSize_implActorState(); static_cast<ObjectSize_implActor*>(this)->destroy(); return 0; }
															#line 3065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ObjectSize_implActor*>(this)->SAV< int64_t >::value()) int64_t(r->data.contentLen);
		this->~ObjectSize_implActorState();
		static_cast<ObjectSize_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ObjectSize_implActor*>(this)->actor_wait_state > 0) static_cast<ObjectSize_implActor*>(this)->actor_wait_state = 0;
		static_cast<ObjectSize_implActor*>(this)->ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 3158 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via objectSize_impl()
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ObjectSize_implActor final : public Actor<int64_t>, public ActorCallback< ObjectSize_implActor, 0, Void >, public ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ObjectSize_implActor>, public ObjectSize_implActorState<ObjectSize_implActor> {
															#line 3163 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ObjectSize_implActor>::operator new;
	using FastAllocated<ObjectSize_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3642515447168793600UL, 12702718812805295616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int64_t>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ObjectSize_implActor, 0, Void >;
friend struct ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ObjectSize_implActor(Reference<S3BlobStoreEndpoint> const& b,std::string const& bucket,std::string const& object) 
															#line 3181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<int64_t>(),
		   ObjectSize_implActorState<ObjectSize_implActor>(b, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("objectSize_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4058515382792528128UL, 9575206348893316608UL);
		ActorExecutionContextHelper __helper(static_cast<ObjectSize_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("objectSize_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("objectSize_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ObjectSize_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ObjectSize_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<int64_t> objectSize_impl( Reference<S3BlobStoreEndpoint> const& b, std::string const& bucket, std::string const& object ) {
															#line 619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<int64_t>(new ObjectSize_implActor(b, bucket, object));
															#line 3215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<int64_t> S3BlobStoreEndpoint::objectSize(std::string const& bucket, std::string const& object) {
	return objectSize_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

// Try to read a file, parse it as JSON, and return the resulting document.
// It will NOT throw if any errors are encountered, it will just return an empty
// JSON object and will log trace events for the errors encountered.
															#line 3227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via tryReadJSONFile()
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class TryReadJSONFileActor>
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class TryReadJSONFileActorState {
															#line 3234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	TryReadJSONFileActorState(std::string const& path) 
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : path(path),
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   content(),
															#line 642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   errorEventType("BlobCredentialFileError")
															#line 3245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("tryReadJSONFile", reinterpret_cast<unsigned long>(this));

	}
	~TryReadJSONFileActorState() 
	{
		fdb_probe_actor_destroy("tryReadJSONFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_NO_AIO | IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED, 0);
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 1;
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*>(static_cast<TryReadJSONFileActor*>(this)));
															#line 3268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TryReadJSONFileActorState();
		static_cast<TryReadJSONFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<TryReadJSONFileActor*>(this)->SAV<Optional<json_spirit::mObject>>::futures) { (void)(Optional<json_spirit::mObject>()); this->~TryReadJSONFileActorState(); static_cast<TryReadJSONFileActor*>(this)->destroy(); return 0; }
															#line 3297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<TryReadJSONFileActor*>(this)->SAV< Optional<json_spirit::mObject> >::value()) Optional<json_spirit::mObject>(Optional<json_spirit::mObject>());
		this->~TryReadJSONFileActorState();
		static_cast<TryReadJSONFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 3310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, errorEventType).errorUnsuppressed(e).suppressFor(60).detail("File", path);
															#line 3314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = f->size();
															#line 647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 2;
															#line 647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TryReadJSONFileActor, 1, int64_t >*>(static_cast<TryReadJSONFileActor*>(this)));
															#line 3337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __f,int loopDepth) 
	{
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		f = __f;
															#line 3346 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __f,int loopDepth) 
	{
		f = std::move(__f);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state > 0) static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 0;
		static_cast<TryReadJSONFileActor*>(this)->ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		buf = makeString(size);
															#line 649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<int> __when_expr_2 = f->read(mutateString(buf), size, 0);
															#line 649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 3;
															#line 649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TryReadJSONFileActor, 2, int >*>(static_cast<TryReadJSONFileActor*>(this)));
															#line 3434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(int64_t const& __size,int loopDepth) 
	{
															#line 647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		size = __size;
															#line 3443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(int64_t && __size,int loopDepth) 
	{
		size = std::move(__size);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state > 0) static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 0;
		static_cast<TryReadJSONFileActor*>(this)->ActorCallback< TryReadJSONFileActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TryReadJSONFileActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int const& r,int loopDepth) 
	{
															#line 650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		ASSERT(r == size);
															#line 651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		content = buf.toString();
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		errorEventType = "BlobCredentialFileParseFailed";
															#line 655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::mValue json;
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::read_string(content, json);
															#line 657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (json.type() == json_spirit::obj_type)
															#line 3532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<TryReadJSONFileActor*>(this)->SAV<Optional<json_spirit::mObject>>::futures) { (void)(json.get_obj()); this->~TryReadJSONFileActorState(); static_cast<TryReadJSONFileActor*>(this)->destroy(); return 0; }
															#line 3536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<TryReadJSONFileActor*>(this)->SAV< Optional<json_spirit::mObject> >::value()) Optional<json_spirit::mObject>(json.get_obj());
			this->~TryReadJSONFileActorState();
			static_cast<TryReadJSONFileActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "BlobCredentialFileNotJSONObject").suppressFor(60).detail("File", path);
															#line 3546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(int && r,int loopDepth) 
	{
															#line 650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		ASSERT(r == size);
															#line 651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		content = buf.toString();
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		errorEventType = "BlobCredentialFileParseFailed";
															#line 655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::mValue json;
															#line 656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		json_spirit::read_string(content, json);
															#line 657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (json.type() == json_spirit::obj_type)
															#line 3566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<TryReadJSONFileActor*>(this)->SAV<Optional<json_spirit::mObject>>::futures) { (void)(json.get_obj()); this->~TryReadJSONFileActorState(); static_cast<TryReadJSONFileActor*>(this)->destroy(); return 0; }
															#line 3570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<TryReadJSONFileActor*>(this)->SAV< Optional<json_spirit::mObject> >::value()) Optional<json_spirit::mObject>(json.get_obj());
			this->~TryReadJSONFileActorState();
			static_cast<TryReadJSONFileActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "BlobCredentialFileNotJSONObject").suppressFor(60).detail("File", path);
															#line 3580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(int const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(int && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TryReadJSONFileActor*>(this)->actor_wait_state > 0) static_cast<TryReadJSONFileActor*>(this)->actor_wait_state = 0;
		static_cast<TryReadJSONFileActor*>(this)->ActorCallback< TryReadJSONFileActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TryReadJSONFileActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TryReadJSONFileActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string path;
															#line 639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string content;
															#line 642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	const char* errorEventType;
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<IAsyncFile> f;
															#line 647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int64_t size;
															#line 648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Standalone<StringRef> buf;
															#line 3686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via tryReadJSONFile()
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class TryReadJSONFileActor final : public Actor<Optional<json_spirit::mObject>>, public ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< TryReadJSONFileActor, 1, int64_t >, public ActorCallback< TryReadJSONFileActor, 2, int >, public FastAllocated<TryReadJSONFileActor>, public TryReadJSONFileActorState<TryReadJSONFileActor> {
															#line 3691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<TryReadJSONFileActor>::operator new;
	using FastAllocated<TryReadJSONFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6125719904223121920UL, 3977329210881511680UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<json_spirit::mObject>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< TryReadJSONFileActor, 1, int64_t >;
friend struct ActorCallback< TryReadJSONFileActor, 2, int >;
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	TryReadJSONFileActor(std::string const& path) 
															#line 3710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Optional<json_spirit::mObject>>(),
		   TryReadJSONFileActorState<TryReadJSONFileActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tryReadJSONFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16806631701943569920UL, 8005239482977197312UL);
		ActorExecutionContextHelper __helper(static_cast<TryReadJSONFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tryReadJSONFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tryReadJSONFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TryReadJSONFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TryReadJSONFileActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TryReadJSONFileActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Optional<json_spirit::mObject>> tryReadJSONFile( std::string const& path ) {
															#line 638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Optional<json_spirit::mObject>>(new TryReadJSONFileActor(path));
															#line 3745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

// If the credentials expire, the connection will eventually fail and be discarded from the pool, and then a new
// connection will be constructed, which will call this again to get updated credentials
static S3BlobStoreEndpoint::Credentials getSecretSdk() {
#ifdef WITH_AWS_BACKUP
	double elapsed = -timer_monotonic();
	Aws::Auth::AWSCredentials awsCreds = FDBAWSCredentialsProvider::getAwsCredentials();
	elapsed += timer_monotonic();

	if (awsCreds.IsEmpty()) {
		TraceEvent(SevWarn, "S3BlobStoreAWSCredsEmpty");
		throw backup_auth_missing();
	}

	S3BlobStoreEndpoint::Credentials fdbCreds;
	fdbCreds.key = awsCreds.GetAWSAccessKeyId();
	fdbCreds.secret = awsCreds.GetAWSSecretKey();
	fdbCreds.securityToken = awsCreds.GetSessionToken();

	TraceEvent("S3BlobStoreGotSdkCredentials").suppressFor(60).detail("Duration", elapsed);

	return fdbCreds;
#else
	TraceEvent(SevError, "S3BlobStoreNoSDK");
	throw backup_auth_missing();
#endif
}

															#line 3777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via updateSecret_impl()
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class UpdateSecret_implActor>
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UpdateSecret_implActorState {
															#line 3784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UpdateSecret_implActorState(Reference<S3BlobStoreEndpoint> const& b) 
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b)
															#line 3791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("updateSecret_impl", reinterpret_cast<unsigned long>(this));

	}
	~UpdateSecret_implActorState() 
	{
		fdb_probe_actor_destroy("updateSecret_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (b->knobs.sdk_auth)
															#line 3806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				b->credentials = getSecretSdk();
															#line 700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateSecret_implActorState();
				static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::vector<std::string>* pFiles = (std::vector<std::string>*)g_network->global(INetwork::enBlobCredentialFiles);
															#line 703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (pFiles == nullptr)
															#line 3822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateSecret_implActorState();
				static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!b->credentials.present())
															#line 3834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3838 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateSecret_implActorState();
				static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			reads = std::vector<Future<Optional<json_spirit::mObject>>>();
															#line 711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for( auto& f : *pFiles ) {
															#line 712 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				reads.push_back(tryReadJSONFile(f));
															#line 3850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(reads);
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<UpdateSecret_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateSecret_implActor*>(this)->actor_wait_state = 1;
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateSecret_implActor, 0, Void >*>(static_cast<UpdateSecret_implActor*>(this)));
															#line 3861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateSecret_implActorState();
		static_cast<UpdateSecret_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string accessKey = b->lookupKey ? "" : b->credentials.get().key;
															#line 717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string credentialsFileKey = accessKey + "@" + b->host;
															#line 719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		int invalid = 0;
															#line 721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& f : reads ) {
															#line 723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!f.get().present())
															#line 3892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreAuthMissingNotPresent") .detail("Endpoint", b->host) .detail("Service", b->service);
															#line 727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				++invalid;
															#line 3898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				continue;
			}
															#line 731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			JSONDoc doc(f.get().get());
															#line 732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (doc.has("accounts") && doc.last().type() == json_spirit::obj_type)
															#line 3905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				JSONDoc accounts(doc.last().get_obj());
															#line 734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (accounts.has(credentialsFileKey, false) && accounts.last().type() == json_spirit::obj_type)
															#line 3911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					JSONDoc account(accounts.last());
															#line 736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					S3BlobStoreEndpoint::Credentials creds = b->credentials.get();
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupKey)
															#line 3919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string apiKey;
															#line 739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("api_key", apiKey))
															#line 3925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.key = apiKey;
															#line 3929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupSecret)
															#line 3938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string secret;
															#line 746 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("secret", secret))
															#line 3944 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.secret = secret;
															#line 3948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					std::string token;
															#line 752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (account.tryGet("token", token))
															#line 3959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						creds.securityToken = token;
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					b->credentials = creds;
															#line 755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 3969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~UpdateSecret_implActorState();
					static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
				else
				{
															#line 757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreAuthFoundAccountFAILED") .detail("CredentialFileKey", credentialsFileKey);
															#line 3979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
		}
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (invalid > 0)
															#line 3985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 765 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreInvalidCredentials") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Invalid", invalid);
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(backup_auth_unreadable(), loopDepth);
															#line 3991 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent(SevWarn, "S3BlobStoreAuthMissing") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Reason", "No valid credentials found");
															#line 777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		return a_body1Catch1(backup_auth_missing(), loopDepth);
															#line 3997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string accessKey = b->lookupKey ? "" : b->credentials.get().key;
															#line 717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string credentialsFileKey = accessKey + "@" + b->host;
															#line 719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		int invalid = 0;
															#line 721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& f : reads ) {
															#line 723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!f.get().present())
															#line 4013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreAuthMissingNotPresent") .detail("Endpoint", b->host) .detail("Service", b->service);
															#line 727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				++invalid;
															#line 4019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				continue;
			}
															#line 731 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			JSONDoc doc(f.get().get());
															#line 732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (doc.has("accounts") && doc.last().type() == json_spirit::obj_type)
															#line 4026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				JSONDoc accounts(doc.last().get_obj());
															#line 734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (accounts.has(credentialsFileKey, false) && accounts.last().type() == json_spirit::obj_type)
															#line 4032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					JSONDoc account(accounts.last());
															#line 736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					S3BlobStoreEndpoint::Credentials creds = b->credentials.get();
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupKey)
															#line 4040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string apiKey;
															#line 739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("api_key", apiKey))
															#line 4046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.key = apiKey;
															#line 4050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (b->lookupSecret)
															#line 4059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 745 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string secret;
															#line 746 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (account.tryGet("secret", secret))
															#line 4065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							creds.secret = secret;
															#line 4069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
							continue;
						}
					}
															#line 751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					std::string token;
															#line 752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (account.tryGet("token", token))
															#line 4080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						creds.securityToken = token;
															#line 4084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					b->credentials = creds;
															#line 755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<UpdateSecret_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateSecret_implActorState(); static_cast<UpdateSecret_implActor*>(this)->destroy(); return 0; }
															#line 4090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<UpdateSecret_implActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~UpdateSecret_implActorState();
					static_cast<UpdateSecret_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
				else
				{
															#line 757 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreAuthFoundAccountFAILED") .detail("CredentialFileKey", credentialsFileKey);
															#line 4100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
		}
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (invalid > 0)
															#line 4106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 765 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreInvalidCredentials") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Invalid", invalid);
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(backup_auth_unreadable(), loopDepth);
															#line 4112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent(SevWarn, "S3BlobStoreAuthMissing") .detail("Endpoint", b->host) .detail("Service", b->service) .detail("Reason", "No valid credentials found");
															#line 777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		return a_body1Catch1(backup_auth_missing(), loopDepth);
															#line 4118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateSecret_implActor*>(this)->actor_wait_state > 0) static_cast<UpdateSecret_implActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateSecret_implActor*>(this)->ActorCallback< UpdateSecret_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateSecret_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateSecret_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateSecret_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::vector<Future<Optional<json_spirit::mObject>>> reads;
															#line 4201 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via updateSecret_impl()
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UpdateSecret_implActor final : public Actor<Void>, public ActorCallback< UpdateSecret_implActor, 0, Void >, public FastAllocated<UpdateSecret_implActor>, public UpdateSecret_implActorState<UpdateSecret_implActor> {
															#line 4206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<UpdateSecret_implActor>::operator new;
	using FastAllocated<UpdateSecret_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12061555381630788352UL, 16015368066786078976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateSecret_implActor, 0, Void >;
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UpdateSecret_implActor(Reference<S3BlobStoreEndpoint> const& b) 
															#line 4223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateSecret_implActorState<UpdateSecret_implActor>(b),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateSecret_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7400895728881971712UL, 1782739488599957760UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateSecret_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateSecret_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateSecret_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateSecret_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> updateSecret_impl( Reference<S3BlobStoreEndpoint> const& b ) {
															#line 697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new UpdateSecret_implActor(b));
															#line 4256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::updateSecret() {
	return updateSecret_impl(Reference<S3BlobStoreEndpoint>::addRef(this));
}

															#line 4265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via connect_impl()
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class Connect_implActor>
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class Connect_implActorState {
															#line 4272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Connect_implActorState(Reference<S3BlobStoreEndpoint> const& b,bool* const& reusingConn) 
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : b(b),
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   reusingConn(reusingConn)
															#line 4281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("connect_impl", reinterpret_cast<unsigned long>(this));

	}
	~Connect_implActorState() 
	{
		fdb_probe_actor_destroy("connect_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			*reusingConn = false;
															#line 788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;!b->connectionPool->pool.empty();) {
															#line 789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint::ReusableConnection rconn = b->connectionPool->pool.front();
															#line 790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				b->connectionPool->pool.pop();
															#line 793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (rconn.expirationTime > now())
															#line 4304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					*reusingConn = true;
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					++b->blobStats->reusedConnections;
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent("S3BlobStoreEndpointReusingConnected") .suppressFor(60) .detail("RemoteEndpoint", rconn.conn->getPeerAddress()) .detail("ExpiresIn", rconn.expirationTime - now()) .detail("Proxy", b->proxyHost.orDefault(""));
															#line 801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<Connect_implActor*>(this)->SAV<S3BlobStoreEndpoint::ReusableConnection>::futures) { (void)(rconn); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 4314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<Connect_implActor*>(this)->SAV< S3BlobStoreEndpoint::ReusableConnection >::value()) S3BlobStoreEndpoint::ReusableConnection(rconn);
					this->~Connect_implActorState();
					static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				++b->blobStats->expiredConnections;
															#line 4322 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++b->blobStats->newConnections;
															#line 806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string host = b->host, service = b->service;
															#line 807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevDebug, "S3BlobStoreEndpointBuildingNewConnection") .detail("UseProxy", b->useProxy) .detail("TLS", b->knobs.secure_connection == 1) .detail("Host", host) .detail("Service", service) .log();
															#line 813 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (service.empty())
															#line 4332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (b->useProxy)
															#line 4336 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					fprintf(stderr, "ERROR: Port can't be empty when using HTTP proxy.\n");
															#line 816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1Catch1(connection_failed(), loopDepth);
															#line 4342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				service = b->knobs.secure_connection ? "https" : "http";
															#line 4346 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bool isTLS = b->knobs.isTLS();
															#line 821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			conn = Reference<IConnection>();
															#line 822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (b->useProxy)
															#line 4354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (isTLS)
															#line 4358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					StrictFuture<Reference<IConnection>> __when_expr_0 = HTTP::proxyConnect(host, service, b->proxyHost.get(), b->proxyPort.get());
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<Connect_implActor*>(this)->actor_wait_state = 1;
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 0, Reference<IConnection> >*>(static_cast<Connect_implActor*>(this)));
															#line 4369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 828 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					host = b->proxyHost.get();
															#line 829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					service = b->proxyPort.get();
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					StrictFuture<Reference<IConnection>> __when_expr_1 = INetworkConnections::net()->connect(host, service, false);
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
					static_cast<Connect_implActor*>(this)->actor_wait_state = 2;
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 1, Reference<IConnection> >*>(static_cast<Connect_implActor*>(this)));
															#line 4387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					loopDepth = 0;
				}
			}
			else
			{
															#line 834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_2 = store(conn, INetworkConnections::net()->connect(host, service, isTLS));
															#line 834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
				static_cast<Connect_implActor*>(this)->actor_wait_state = 3;
															#line 834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 2, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = conn->connectHandshake();
															#line 836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<Connect_implActor*>(this)->actor_wait_state = 4;
															#line 836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 3, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Reference<IConnection> const& _conn,int loopDepth) 
	{
															#line 826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4448 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Reference<IConnection> && _conn,int loopDepth) 
	{
															#line 826 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IConnection> const& _conn,int loopDepth) 
	{
		loopDepth = a_body1cont7(_conn, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IConnection> && _conn,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_conn), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 0, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont9(Reference<IConnection> const& _conn,int loopDepth) 
	{
															#line 831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4541 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Reference<IConnection> && _conn,int loopDepth) 
	{
															#line 831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		conn = _conn;
															#line 4550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Reference<IConnection> const& _conn,int loopDepth) 
	{
		loopDepth = a_body1cont9(_conn, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Reference<IConnection> && _conn,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_conn), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 1, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 1, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 1, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 1, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont12(Void const& _,int loopDepth) 
	{
															#line 838 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent("S3BlobStoreEndpointNewConnectionSuccess") .suppressFor(60) .detail("RemoteEndpoint", conn->getPeerAddress()) .detail("ExpiresIn", b->knobs.max_connection_life) .detail("Proxy", b->proxyHost.orDefault(""));
															#line 844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (b->lookupKey || b->lookupSecret || b->knobs.sdk_auth)
															#line 4723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = b->updateSecret();
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont12when1(__when_expr_4.get(), loopDepth); };
			static_cast<Connect_implActor*>(this)->actor_wait_state = 5;
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 4, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont12(Void && _,int loopDepth) 
	{
															#line 838 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent("S3BlobStoreEndpointNewConnectionSuccess") .suppressFor(60) .detail("RemoteEndpoint", conn->getPeerAddress()) .detail("ExpiresIn", b->knobs.max_connection_life) .detail("Proxy", b->proxyHost.orDefault(""));
															#line 844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (b->lookupKey || b->lookupSecret || b->knobs.sdk_auth)
															#line 4750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = b->updateSecret();
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont12when1(__when_expr_4.get(), loopDepth); };
			static_cast<Connect_implActor*>(this)->actor_wait_state = 5;
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 4, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 4761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12cont1(int loopDepth) 
	{
															#line 847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<Connect_implActor*>(this)->SAV<S3BlobStoreEndpoint::ReusableConnection>::futures) { (void)(S3BlobStoreEndpoint::ReusableConnection({ conn, now() + b->knobs.max_connection_life })); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 4850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<Connect_implActor*>(this)->SAV< S3BlobStoreEndpoint::ReusableConnection >::value()) S3BlobStoreEndpoint::ReusableConnection(S3BlobStoreEndpoint::ReusableConnection({ conn, now() + b->knobs.max_connection_life }));
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont12cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont12when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont12when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> b;
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool* reusingConn;
															#line 821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<IConnection> conn;
															#line 4951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via connect_impl()
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class Connect_implActor final : public Actor<S3BlobStoreEndpoint::ReusableConnection>, public ActorCallback< Connect_implActor, 0, Reference<IConnection> >, public ActorCallback< Connect_implActor, 1, Reference<IConnection> >, public ActorCallback< Connect_implActor, 2, Void >, public ActorCallback< Connect_implActor, 3, Void >, public ActorCallback< Connect_implActor, 4, Void >, public FastAllocated<Connect_implActor>, public Connect_implActorState<Connect_implActor> {
															#line 4956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<Connect_implActor>::operator new;
	using FastAllocated<Connect_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2212010700504369408UL, 16889135869316902400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<S3BlobStoreEndpoint::ReusableConnection>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Connect_implActor, 0, Reference<IConnection> >;
friend struct ActorCallback< Connect_implActor, 1, Reference<IConnection> >;
friend struct ActorCallback< Connect_implActor, 2, Void >;
friend struct ActorCallback< Connect_implActor, 3, Void >;
friend struct ActorCallback< Connect_implActor, 4, Void >;
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Connect_implActor(Reference<S3BlobStoreEndpoint> const& b,bool* const& reusingConn) 
															#line 4977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<S3BlobStoreEndpoint::ReusableConnection>(),
		   Connect_implActorState<Connect_implActor>(b, reusingConn),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7092978256320716288UL, 13901349198287939840UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connect_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Connect_implActor, 0, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Connect_implActor, 1, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< Connect_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< Connect_implActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< Connect_implActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<S3BlobStoreEndpoint::ReusableConnection> connect_impl( Reference<S3BlobStoreEndpoint> const& b, bool* const& reusingConn ) {
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<S3BlobStoreEndpoint::ReusableConnection>(new Connect_implActor(b, reusingConn));
															#line 5014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<S3BlobStoreEndpoint::ReusableConnection> S3BlobStoreEndpoint::connect(bool* reusing) {
	return connect_impl(Reference<S3BlobStoreEndpoint>::addRef(this), reusing);
}

void S3BlobStoreEndpoint::returnConnection(ReusableConnection& rconn) {
	// If it expires in the future then add it to the pool in the front
	if (rconn.expirationTime > now()) {
		connectionPool->pool.push(rconn);
	} else {
		++blobStats->expiredConnections;
	}
	rconn.conn = Reference<IConnection>();
}

std::string awsCanonicalURI(const std::string& resource, std::vector<std::string>& queryParameters, bool isV4) {
	StringRef resourceRef(resource);
	resourceRef.eat("/");
	std::string canonicalURI("/" + resourceRef.toString());
	size_t q = canonicalURI.find_last_of('?');
	if (q != canonicalURI.npos)
		canonicalURI.resize(q);
	if (isV4) {
		canonicalURI = HTTP::awsV4URIEncode(canonicalURI, false);
	} else {
		canonicalURI = HTTP::urlEncode(canonicalURI);
	}

	// Create the canonical query string
	std::string queryString;
	q = resource.find_last_of('?');
	if (q != queryString.npos)
		queryString = resource.substr(q + 1);

	StringRef qStr(queryString);
	StringRef queryParameter;
	while ((queryParameter = qStr.eat("&")) != StringRef()) {
		StringRef param = queryParameter.eat("=");
		StringRef value = queryParameter.eat();

		if (isV4) {
			queryParameters.push_back(HTTP::awsV4URIEncode(param.toString(), true) + "=" +
			                          HTTP::awsV4URIEncode(value.toString(), true));
		} else {
			queryParameters.push_back(HTTP::urlEncode(param.toString()) + "=" + HTTP::urlEncode(value.toString()));
		}
	}

	return canonicalURI;
}

// ref: https://docs.aws.amazon.com/AmazonS3/latest/API/ErrorResponses.html
std::string parseErrorCodeFromS3(std::string xmlResponse) {
	// Copy XML string to a modifiable buffer
	try {
		std::vector<char> xmlBuffer(xmlResponse.begin(), xmlResponse.end());
		xmlBuffer.push_back('\0'); // Ensure null-terminated string
		// Parse the XML
		xml_document<> doc;
		doc.parse<0>(&xmlBuffer[0]);
		// Find the root node
		xml_node<>* root = doc.first_node("Error");
		if (!root) {
			TraceEvent(SevWarn, "ParseS3XMLResponseNoError").detail("Response", xmlResponse).log();
			return "";
		}
		// Find the <Code> node
		xml_node<>* codeNode = root->first_node("Code");
		if (!codeNode) {
			TraceEvent(SevWarn, "ParseS3XMLResponseNoErrorCode").detail("Response", xmlResponse).log();
			return "";
		}
		return std::string(codeNode->value());
	} catch (Error e) {
		TraceEvent("BackupParseS3ErrorCodeFailure").errorUnsuppressed(e);
		throw backup_parse_s3_response_failure();
	} catch (...) {
		throw backup_parse_s3_response_failure();
	}
}

bool isS3TokenError(const std::string& s3Error) {
	return s3Error == "InvalidToken" || s3Error == "ExpiredToken";
}

void setHeaders(Reference<S3BlobStoreEndpoint> bstore, Reference<HTTP::OutgoingRequest> req) {
	// Finish/update the request headers (which includes Date header)
	// This must be done AFTER the connection is ready because if credentials are coming from disk they are
	// refreshed when a new connection is established and setAuthHeaders() would need the updated secret.
	if (bstore->credentials.present() && !bstore->credentials.get().securityToken.empty())
		req->data.headers["x-amz-security-token"] = bstore->credentials.get().securityToken;
	if (CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER) {
		bstore->setV4AuthHeaders(req->verb, req->resource, req->data.headers);
	} else {
		bstore->setAuthHeaders(req->verb, req->resource, req->data.headers);
	}
}

std::string getCanonicalURI(Reference<S3BlobStoreEndpoint> bstore, Reference<HTTP::OutgoingRequest> req) {
	std::vector<std::string> queryParameters;
	std::string canonicalURI =
	    awsCanonicalURI(req->resource, queryParameters, CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER);
	if (!queryParameters.empty()) {
		canonicalURI += "?";
		// Use raw ampersands when joining query parameters
		for (size_t i = 0; i < queryParameters.size(); ++i) {
			if (i > 0) {
				canonicalURI += "&";
			}
			canonicalURI += queryParameters[i];
		}
	}

	if (bstore->useProxy && bstore->knobs.secure_connection == 0) {
		// Has to be in absolute-form.
		canonicalURI = "http://" + bstore->host + ":" + bstore->service + canonicalURI;
	}
	return canonicalURI;
}

void populateDryrunRequest(Reference<HTTP::OutgoingRequest> dryrunRequest,
                           Reference<S3BlobStoreEndpoint> bstore,
                           std::string bucket) {
	// dryrun with a check bucket exist request, to avoid sending duplicate data
	HTTP::Headers headers;
	dryrunRequest->verb = "GET";
	dryrunRequest->data.contentLen = 0;
	dryrunRequest->data.headers = headers;
	dryrunRequest->data.headers["Host"] = bstore->host;
	dryrunRequest->data.headers["Accept"] = "application/xml";

	dryrunRequest->resource = bstore->constructResourcePath(bucket, "");
}

bool isWriteRequest(std::string verb) {
	return verb == "POST" || verb == "PUT";
}

std::string parseBucketFromURI(std::string uri) {
	if (uri.size() <= 1 || uri[0] != '/') {
		// there is no bucket in the uri
		return "";
	}
	uri = uri.substr(1);
	size_t secondSlash = uri.find('/');
	if (secondSlash == std::string::npos) {
		return uri;
	}
	return uri.substr(0, secondSlash);
}

// Do a request, get a Response.
// Request content is provided as UnsentPacketQueue *pContent which will be depleted as bytes are sent but the queue
// itself must live for the life of this actor and be destroyed by the caller
															#line 5172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via doRequest_impl()
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class DoRequest_implActor>
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DoRequest_implActorState {
															#line 5179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DoRequest_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& verb,std::string const& resource,HTTP::Headers const& headers,UnsentPacketQueue* const& pContent,int const& contentLen,std::set<unsigned int> const& successCodes) 
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   verb(verb),
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resource(resource),
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   headers(headers),
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pContent(pContent),
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentLen(contentLen),
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   successCodes(successCodes),
															#line 1010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentCopy(),
															#line 1011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   dryrunContentCopy(),
															#line 1012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   req(makeReference<HTTP::OutgoingRequest>()),
															#line 1013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   dryrunRequest(makeReference<HTTP::OutgoingRequest>())
															#line 5206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("doRequest_impl", reinterpret_cast<unsigned long>(this));

	}
	~DoRequest_implActorState() 
	{
		fdb_probe_actor_destroy("doRequest_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->verb = verb;
															#line 1015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.content = &contentCopy;
															#line 1016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.contentLen = contentLen;
															#line 1018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.headers = headers;
															#line 1019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.headers["Host"] = bstore->host;
															#line 1020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.headers["Accept"] = "application/xml";
															#line 1023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (resource.empty())
															#line 5233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource = "/";
															#line 5237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->resource = resource;
															#line 1030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for( const auto& [k, v] : bstore->extraHeaders ) {
															#line 1031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string& fieldValue = req->data.headers[k];
															#line 1032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!fieldValue.empty())
															#line 5247 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					fieldValue.append(",");
															#line 5251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				fieldValue.append(v);
															#line 5255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			int bandwidthThisRequest = 1 + bstore->knobs.max_send_bytes_per_second / bstore->knobs.concurrent_uploads;
															#line 1041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			int contentUploadSeconds = contentLen / bandwidthThisRequest;
															#line 1042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			requestTimeout = std::max(bstore->knobs.request_timeout_min, 3 * contentUploadSeconds);
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->concurrentRequests.take();
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 1;
															#line 1044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 0, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoRequest_implActorState();
		static_cast<DoRequest_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		globalReleaser = FlowLock::Releaser(bstore->concurrentRequests, 1);
															#line 1047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxTries = std::min(bstore->knobs.request_tries, bstore->knobs.connect_tries);
															#line 1048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		thisTry = 1;
															#line 1049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		badRequestCode = 400;
															#line 1050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		s3TokenError = false;
															#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		nextRetryDelay = 2.0;
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 5307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		globalReleaser = FlowLock::Releaser(bstore->concurrentRequests, 1);
															#line 1047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxTries = std::min(bstore->knobs.request_tries, bstore->knobs.connect_tries);
															#line 1048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		thisTry = 1;
															#line 1049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		badRequestCode = 400;
															#line 1050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		s3TokenError = false;
															#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		nextRetryDelay = 2.0;
															#line 1053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 5328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		err = Optional<Error>();
															#line 1055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		remoteAddress = Optional<NetworkAddress>();
															#line 1056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connectionEstablished = false;
															#line 1057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 1058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		dryrunR = Reference<HTTP::IncomingResponse>();
															#line 1059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		canonicalURI = resource;
															#line 1060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connID = UID();
															#line 1061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reqStartTimer = double();
															#line 1062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connectStartTimer = g_network->timer();
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reusingConn = false;
															#line 1064 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fastRetry = false;
															#line 1065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		simulateS3TokenError = false;
															#line 5441 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			Future<S3BlobStoreEndpoint::ReusableConnection> frconn = bstore->connect(&reusingConn);
															#line 1073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			req->data.content->discardAll();
															#line 1074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (pContent != nullptr)
															#line 5449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketBuffer* pFirst = pContent->getUnsent();
															#line 1076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				PacketBuffer* pLast = nullptr;
															#line 1077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(PacketBuffer* p = pFirst;p != nullptr;p = p->nextPacketBuffer()) {
															#line 1078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					p->addref();
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					p->bytes_sent = 0;
															#line 1081 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					pLast = p;
															#line 5463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				req->data.content->prependWriteBuffer(pFirst, pLast);
															#line 5467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<S3BlobStoreEndpoint::ReusableConnection> __when_expr_1 = timeoutError(frconn, bstore->knobs.connect_timeout);
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 2;
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5478 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 1203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		double end = g_network->timer();
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		double connectDuration = reqStartTimer - connectStartTimer;
															#line 1205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		double reqDuration = end - reqStartTimer;
															#line 1206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bstore->blobStats->requestLatency.addMeasurement(reqDuration);
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!err.present() && successCodes.count(r->code) != 0)
															#line 5501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevDebug, "S3BlobStoreDoRequestSuccessful") .detail("Verb", verb) .detail("Error", err.present()) .detail("ErrorString", err.present() ? err.get().name() : "") .detail("Resource", resource) .detail("ResponseCode", r->code) .detail("ResponseContentSize", r->data.content.size()) .log();
															#line 1219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bstore->s_stats.requests_successful++;
															#line 1220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->requestsSuccessful;
															#line 1221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<DoRequest_implActor*>(this)->SAV<Reference<HTTP::IncomingResponse>>::futures) { (void)(r); this->~DoRequest_implActorState(); static_cast<DoRequest_implActor*>(this)->destroy(); return 0; }
															#line 5511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<DoRequest_implActor*>(this)->SAV< Reference<HTTP::IncomingResponse> >::value()) Reference<HTTP::IncomingResponse>(std::move(r)); // state_var_RVO
			this->~DoRequest_implActorState();
			static_cast<DoRequest_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bstore->s_stats.requests_failed++;
															#line 1226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		++bstore->blobStats->requestsFailed;
															#line 1229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bool retryable = err.present() || r->code == 500 || r->code == 502 || r->code == 503 || r->code == 429;
															#line 1231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		retryable = retryable && (thisTry < maxTries);
															#line 1233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!retryable || !err.present())
															#line 5527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fastRetry = false;
															#line 5531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent event(SevWarn, retryable ? (fastRetry ? "S3BlobStoreEndpointRequestFailedFastRetryable" : "S3BlobStoreEndpointRequestFailedRetryable") : "S3BlobStoreEndpointRequestFailed");
															#line 1242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		bool connectionFailed = false;
															#line 1244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (err.present())
															#line 5539 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.errorUnsuppressed(err.get());
															#line 1246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (err.get().code() == error_code_connection_failed)
															#line 5545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1247 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				connectionFailed = true;
															#line 5549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.suppressFor(60);
															#line 1252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!err.present())
															#line 5556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ResponseCode", r->code);
															#line 1254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string s3Error = parseErrorCodeFromS3(r->data.content);
															#line 1255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("S3ErrorCode", s3Error);
															#line 1256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (r->code == badRequestCode)
															#line 5566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (isS3TokenError(s3Error) || simulateS3TokenError)
															#line 5570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					s3TokenError = true;
															#line 5574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarnAlways, "S3BlobStoreBadRequest") .detail("HttpCode", r->code) .detail("HttpResponseContent", r->data.content) .detail("S3Error", s3Error) .log();
															#line 5578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.detail("ConnectionEstablished", connectionEstablished);
															#line 1269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.detail("ReusingConn", reusingConn);
															#line 1270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (connectionEstablished)
															#line 5587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ConnID", connID);
															#line 1272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ConnectDuration", connectDuration);
															#line 1273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("ReqDuration", reqDuration);
															#line 5595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (remoteAddress.present())
															#line 5599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("RemoteEndpoint", remoteAddress.get());
															#line 5603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			event.detail("RemoteHost", bstore->host);
															#line 5609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		event.detail("Verb", verb) .detail("Resource", resource) .detail("ThisTry", thisTry) .detail("URI", canonicalURI) .detail("Proxy", bstore->proxyHost.orDefault(""));
															#line 1289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!fastRetry && (!r || r->code != 429))
															#line 5615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++thisTry;
															#line 5619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (fastRetry)
															#line 5623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->fastRetries;
															#line 1294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_7 = delay(0);
															#line 1294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 8;
															#line 1294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 7, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (retryable || s3TokenError)
															#line 5643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				double delay = nextRetryDelay;
															#line 1300 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				double limit = connectionFailed ? bstore->knobs.max_delay_connection_failed : bstore->knobs.max_delay_retryable_error;
															#line 1303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				nextRetryDelay = std::min(nextRetryDelay * 2, limit);
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (r)
															#line 5653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					auto iRetryAfter = r->data.headers.find("Retry-After");
															#line 1307 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (iRetryAfter != r->data.headers.end())
															#line 5659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						event.detail("RetryAfterHeader", iRetryAfter->second);
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						char* pEnd;
															#line 1310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						double retryAfter = strtod(iRetryAfter->second.c_str(), &pEnd);
															#line 1311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (*pEnd)
															#line 5669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1313 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							retryAfter = 300;
															#line 5673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
															#line 1315 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						delay = std::max(delay, retryAfter);
															#line 5677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
				}
															#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				event.detail("RetryDelay", delay);
															#line 1321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_8 = ::delay(delay);
															#line 1321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when2(__when_expr_8.get(), loopDepth); };
				static_cast<DoRequest_implActor*>(this)->actor_wait_state = 9;
															#line 1321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 8, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (r && r->code == 406)
															#line 5698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1Catch1(http_not_accepted(), std::max(0, loopDepth - 1));
															#line 5702 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (r && r->code == 401)
															#line 5706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1Catch1(http_auth_failed(), std::max(0, loopDepth - 1));
															#line 5710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (err.present())
															#line 5714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					int code = err.get().code();
															#line 1341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (code == error_code_timed_out && !connectionEstablished)
															#line 5720 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						TraceEvent(SevWarn, "S3BlobStoreEndpointConnectTimeout") .suppressFor(60) .detail("Timeout", requestTimeout);
															#line 1345 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1Catch1(connection_failed(), std::max(0, loopDepth - 1));
															#line 5726 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 1348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (code == error_code_timed_out || code == error_code_connection_failed || code == error_code_lookup_failed)
															#line 5730 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1Catch1(err.get(), std::max(0, loopDepth - 1));
															#line 5734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
				}
															#line 1353 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(http_request_failed(), std::max(0, loopDepth - 1));
															#line 5739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreDoRequestError") .errorUnsuppressed(e) .detail("Verb", verb) .detail("Resource", resource);
															#line 1197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 5752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			err = e;
															#line 5760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
															#line 1089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connectionEstablished = true;
															#line 1090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		connID = rconn.conn->getDebugID();
															#line 1091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		reqStartTimer = g_network->timer();
															#line 1092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		TraceEvent(SevDebug, "S3BlobStoreEndpointConnected") .suppressFor(60) .detail("RemoteEndpoint", rconn.conn->getPeerAddress()) .detail("Reusing", reusingConn) .detail("ConnID", connID) .detail("Verb", req->verb) .detail("Resource", resource) .detail("Proxy", bstore->proxyHost.orDefault(""));
															#line 5781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (s3TokenError && isWriteRequest(req->verb) && CLIENT_KNOBS->BACKUP_ALLOW_DRYRUN)
															#line 5785 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				bucket = parseBucketFromURI(resource);
															#line 1107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (bucket.empty())
															#line 5791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1108 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevError, "EmptyBucketRequest") .detail("S3TokenError", s3TokenError) .detail("Verb", req->verb) .detail("Resource", resource) .log();
															#line 1113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1cont1loopBody1cont2Catch1(bucket_not_in_url(), loopDepth);
															#line 5797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1115 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				dryrunRequest->data.content = &dryrunContentCopy;
															#line 1116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				dryrunRequest->data.content->discardAll();
															#line 1117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				populateDryrunRequest(dryrunRequest, bstore, bucket);
															#line 1118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				setHeaders(bstore, dryrunRequest);
															#line 1119 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				dryrunRequest->resource = getCanonicalURI(bstore, dryrunRequest);
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent("RetryS3RequestDueToTokenIssue") .detail("S3TokenError", s3TokenError) .detail("OriginalResource", resource) .detail("DryrunResource", dryrunRequest->resource) .detail("Bucket", bucket) .detail("V4", CLIENT_KNOBS->HTTP_REQUEST_AWS_V4_HEADER) .log();
															#line 1127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_2 = bstore->requestRate->getAllowance(1);
															#line 1127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<DoRequest_implActor*>(this)->actor_wait_state = 3;
															#line 1127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 2, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1loopBody1cont6(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(S3BlobStoreEndpoint::ReusableConnection const& __rconn,int loopDepth) 
	{
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rconn = __rconn;
															#line 5840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(S3BlobStoreEndpoint::ReusableConnection && __rconn,int loopDepth) 
	{
		rconn = std::move(__rconn);
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*,S3BlobStoreEndpoint::ReusableConnection const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*,S3BlobStoreEndpoint::ReusableConnection && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
															#line 1163 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		setHeaders(bstore, req);
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		req->resource = getCanonicalURI(bstore, req);
															#line 1166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		remoteAddress = rconn.conn->getPeerAddress();
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_5 = bstore->requestRate->getAllowance(1);
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 6;
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 5, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevError, "ErrorDuringRetryS3TokenIssue").errorUnsuppressed(e);
															#line 5942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 1128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		Future<Reference<HTTP::IncomingResponse>> dryrunResponse = HTTP::doRequest( rconn.conn, dryrunRequest, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = timeoutError(dryrunResponse, requestTimeout);
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 4;
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 1128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		Future<Reference<HTTP::IncomingResponse>> dryrunResponse = HTTP::doRequest( rconn.conn, dryrunRequest, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_3 = timeoutError(dryrunResponse, requestTimeout);
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont7when1(__when_expr_3.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 4;
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 5990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont9(Reference<HTTP::IncomingResponse> const& _dryrunR,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		dryrunR = _dryrunR;
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string s3Error = parseErrorCodeFromS3(dryrunR->data.content);
															#line 1133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (dryrunR->code == badRequestCode && isS3TokenError(s3Error))
															#line 6078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(bstore->knobs.max_delay_retryable_error);
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_4.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 5;
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 4, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (dryrunR->code == 200 || dryrunR->code == 404)
															#line 6096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1141 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent("S3TokenIssueResolved") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1147 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3TokenError = false;
															#line 6102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			else
			{
															#line 1149 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3UnexpectedError") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1cont1loopBody1cont2Catch1(http_bad_response(), loopDepth);
															#line 6110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Reference<HTTP::IncomingResponse> && _dryrunR,int loopDepth) 
	{
															#line 1131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		dryrunR = _dryrunR;
															#line 1132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string s3Error = parseErrorCodeFromS3(dryrunR->data.content);
															#line 1133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (dryrunR->code == badRequestCode && isS3TokenError(s3Error))
															#line 6125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(bstore->knobs.max_delay_retryable_error);
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_4.get(), loopDepth); };
			static_cast<DoRequest_implActor*>(this)->actor_wait_state = 5;
															#line 1137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 4, Void >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6136 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (dryrunR->code == 200 || dryrunR->code == 404)
															#line 6143 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1141 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent("S3TokenIssueResolved") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1147 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3TokenError = false;
															#line 6149 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			else
			{
															#line 1149 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3UnexpectedError") .detail("HttpCode", dryrunR->code) .detail("HttpResponseContent", dryrunR->data.content) .detail("S3Error", s3Error) .detail("URI", dryrunRequest->resource) .log();
															#line 1155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1cont1loopBody1cont2Catch1(http_bad_response(), loopDepth);
															#line 6157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7when1(Reference<HTTP::IncomingResponse> const& _dryrunR,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(_dryrunR, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7when1(Reference<HTTP::IncomingResponse> && _dryrunR,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(std::move(_dryrunR), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont10(int loopDepth) 
	{
		return a_body1cont1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16(Void const& _,int loopDepth) 
	{
															#line 1169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		Future<Reference<HTTP::IncomingResponse>> reqF = HTTP::doRequest(rconn.conn, req, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (reqF.isReady() && reusingConn)
															#line 6351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fastRetry = true;
															#line 6355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_6 = timeoutError(reqF, requestTimeout);
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont16when1(__when_expr_6.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 7;
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16(Void && _,int loopDepth) 
	{
															#line 1169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		Future<Reference<HTTP::IncomingResponse>> reqF = HTTP::doRequest(rconn.conn, req, bstore->sendRate, &bstore->s_stats.bytes_sent, bstore->recvRate);
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (reqF.isReady() && reusingConn)
															#line 6377 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			fastRetry = true;
															#line 6381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_6 = timeoutError(reqF, requestTimeout);
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont16when1(__when_expr_6.get(), loopDepth); };
		static_cast<DoRequest_implActor*>(this)->actor_wait_state = 7;
															#line 1177 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*>(static_cast<DoRequest_implActor*>(this)));
															#line 6392 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont16cont1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
															#line 1178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (g_network->isSimulated() && deterministicRandom()->random01() < 0.1)
															#line 6476 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			_r->code = badRequestCode;
															#line 1181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			simulateS3TokenError = true;
															#line 6482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.headers["Connection"] != "close")
															#line 6488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bstore->returnConnection(rconn);
															#line 6492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->expiredConnections;
															#line 6498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rconn.conn.clear();
															#line 6502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont16cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16cont1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
															#line 1178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (g_network->isSimulated() && deterministicRandom()->random01() < 0.1)
															#line 6511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			_r->code = badRequestCode;
															#line 1181 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			simulateS3TokenError = true;
															#line 6517 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.headers["Connection"] != "close")
															#line 6523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			bstore->returnConnection(rconn);
															#line 6527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			++bstore->blobStats->expiredConnections;
															#line 6533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rconn.conn.clear();
															#line 6537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont16cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16when1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16cont1(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont16when1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont16cont1(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont16when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont16when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont16cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont17(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1loopBody1cont30(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont31(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont31(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<DoRequest_implActor*>(this)->actor_wait_state > 0) static_cast<DoRequest_implActor*>(this)->actor_wait_state = 0;
		static_cast<DoRequest_implActor*>(this)->ActorCallback< DoRequest_implActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< DoRequest_implActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< DoRequest_implActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string verb;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue* pContent;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int contentLen;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::set<unsigned int> successCodes;
															#line 1010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue contentCopy;
															#line 1011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue dryrunContentCopy;
															#line 1012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::OutgoingRequest> req;
															#line 1013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::OutgoingRequest> dryrunRequest;
															#line 1042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int requestTimeout;
															#line 1045 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser globalReleaser;
															#line 1047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxTries;
															#line 1048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int thisTry;
															#line 1049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int badRequestCode;
															#line 1050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool s3TokenError;
															#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double nextRetryDelay;
															#line 1054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<Error> err;
															#line 1055 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<NetworkAddress> remoteAddress;
															#line 1056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool connectionEstablished;
															#line 1057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 1058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> dryrunR;
															#line 1059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string canonicalURI;
															#line 1060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UID connID;
															#line 1061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double reqStartTimer;
															#line 1062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double connectStartTimer;
															#line 1063 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool reusingConn;
															#line 1064 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool fastRetry;
															#line 1065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool simulateS3TokenError;
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	S3BlobStoreEndpoint::ReusableConnection rconn;
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 6880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via doRequest_impl()
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class DoRequest_implActor final : public Actor<Reference<HTTP::IncomingResponse>>, public ActorCallback< DoRequest_implActor, 0, Void >, public ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >, public ActorCallback< DoRequest_implActor, 2, Void >, public ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >, public ActorCallback< DoRequest_implActor, 4, Void >, public ActorCallback< DoRequest_implActor, 5, Void >, public ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >, public ActorCallback< DoRequest_implActor, 7, Void >, public ActorCallback< DoRequest_implActor, 8, Void >, public FastAllocated<DoRequest_implActor>, public DoRequest_implActorState<DoRequest_implActor> {
															#line 6885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<DoRequest_implActor>::operator new;
	using FastAllocated<DoRequest_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8540340532207610880UL, 598757887474979328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<HTTP::IncomingResponse>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoRequest_implActor, 0, Void >;
friend struct ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >;
friend struct ActorCallback< DoRequest_implActor, 2, Void >;
friend struct ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< DoRequest_implActor, 4, Void >;
friend struct ActorCallback< DoRequest_implActor, 5, Void >;
friend struct ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< DoRequest_implActor, 7, Void >;
friend struct ActorCallback< DoRequest_implActor, 8, Void >;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	DoRequest_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& verb,std::string const& resource,HTTP::Headers const& headers,UnsentPacketQueue* const& pContent,int const& contentLen,std::set<unsigned int> const& successCodes) 
															#line 6910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Reference<HTTP::IncomingResponse>>(),
		   DoRequest_implActorState<DoRequest_implActor>(bstore, verb, resource, headers, pContent, contentLen, successCodes),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doRequest_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8762720438240590848UL, 16675552070853705472UL);
		ActorExecutionContextHelper __helper(static_cast<DoRequest_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doRequest_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doRequest_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoRequest_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoRequest_implActor, 1, S3BlobStoreEndpoint::ReusableConnection >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoRequest_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DoRequest_implActor, 3, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DoRequest_implActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DoRequest_implActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< DoRequest_implActor, 6, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< DoRequest_implActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< DoRequest_implActor, 8, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Reference<HTTP::IncomingResponse>> doRequest_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& verb, std::string const& resource, HTTP::Headers const& headers, UnsentPacketQueue* const& pContent, int const& contentLen, std::set<unsigned int> const& successCodes ) {
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Reference<HTTP::IncomingResponse>>(new DoRequest_implActor(bstore, verb, resource, headers, pContent, contentLen, successCodes));
															#line 6951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Reference<HTTP::IncomingResponse>> S3BlobStoreEndpoint::doRequest(std::string const& verb,
                                                                         std::string const& resource,
                                                                         const HTTP::Headers& headers,
                                                                         UnsentPacketQueue* pContent,
                                                                         int contentLen,
                                                                         std::set<unsigned int> successCodes) {
	return doRequest_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), verb, resource, headers, pContent, contentLen, successCodes);
}

															#line 6966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via listObjectsStream_impl()
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ListObjectsStream_implActor>
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjectsStream_implActorState {
															#line 6973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjectsStream_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,PromiseStream<S3BlobStoreEndpoint::ListResult> const& results,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   results(results),
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   prefix(prefix),
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   delimiter(delimiter),
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   maxDepth(maxDepth),
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   recurseFilter(recurseFilter),
															#line 1376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resource(bstore->constructResourcePath(bucket, ""))
															#line 6994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("listObjectsStream_impl", reinterpret_cast<unsigned long>(this));

	}
	~ListObjectsStream_implActorState() 
	{
		fdb_probe_actor_destroy("listObjectsStream_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource.append("/?max-keys=1000");
															#line 1379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (prefix.present())
															#line 7011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1380 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource.append("&prefix=").append(prefix.get());
															#line 7015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (delimiter.present())
															#line 7019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				resource.append("&delimiter=").append(std::string(1, delimiter.get()));
															#line 7023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			resource.append("&marker=");
															#line 1384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			lastFile = std::string();
															#line 1385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = true;
															#line 1387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			subLists = std::vector<Future<Void>>();
															#line 1389 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			;
															#line 7035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListObjectsStream_implActorState();
		static_cast<ListObjectsStream_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_2 = waitForAll(subLists);
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 3;
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 2, Void >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1389 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(more))
															#line 7081 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bstore->concurrentLists.take();
															#line 1390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 1;
															#line 1390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 0, Void >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource + lastFile;
															#line 1395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		lastFile.clear();
															#line 1396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", fullResource, headers, nullptr, 0, { 200, 404 });
															#line 1396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7126 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 2;
															#line 1396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource + lastFile;
															#line 1395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		lastFile.clear();
															#line 1396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", fullResource, headers, nullptr, 0, { 200, 404 });
															#line 1396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 2;
															#line 1396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListObjectsStream_implActor*>(this)));
															#line 7155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjectsStream_implActor*>(this)->ActorCallback< ListObjectsStream_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListObjectsStream_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 7239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ListResult listResult;
															#line 1404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (r->code == 404)
															#line 7245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1405 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3BlobStoreResourceNotFound") .detail("Bucket", bucket) .detail("Prefix", prefix.present() ? prefix.get() : "") .detail("Resource", fullResource);
															#line 1409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(resource_not_found(), loopDepth);
															#line 7251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListBucketResult") != 0)
															#line 7263 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* n = result->first_node();
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;n != nullptr;) {
															#line 1426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				const char* name = n->name();
															#line 1427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (strcmp(name, "IsTruncated") == 0)
															#line 7277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* val = n->value();
															#line 1429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(val, "true") == 0)
															#line 7283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						more = true;
															#line 7287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
					else
					{
															#line 1431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(val, "false") == 0)
															#line 7293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							more = false;
															#line 7297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
															#line 1434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
				else
				{
															#line 1436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(name, "Contents") == 0)
															#line 7311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1437 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						S3BlobStoreEndpoint::ObjectInfo object;
															#line 1439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						xml_node<>* key = n->first_node("Key");
															#line 1440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (key == nullptr)
															#line 7319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1441 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
															#line 1443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						object.name = key->value();
															#line 1445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						xml_node<>* size = n->first_node("Size");
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (size == nullptr)
															#line 7331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
															#line 1449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						object.size = strtoull(size->value(), nullptr, 10);
															#line 1451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						listResult.objects.push_back(object);
															#line 7341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
					else
					{
															#line 1452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(name, "CommonPrefixes") == 0)
															#line 7347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1453 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							xml_node<>* prefixNode = n->first_node("Prefix");
															#line 1454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							for(;prefixNode != nullptr;) {
															#line 1455 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								const char* prefix = prefixNode->value();
															#line 1457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								if (maxDepth > 0)
															#line 7357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								{
															#line 1459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									if (!recurseFilter || recurseFilter(prefix))
															#line 7361 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
									{
															#line 1460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
										subLists.push_back(bstore->listObjectsStream( bucket, results, prefix, delimiter, maxDepth - 1, recurseFilter));
															#line 7365 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
									}
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									lastFile = prefix;
															#line 7369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								}
								else
								{
															#line 1467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									listResult.commonPrefixes.push_back(prefix);
															#line 7375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								}
															#line 1470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								prefixNode = prefixNode->next_sibling("Prefix");
															#line 7379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
							}
						}
					}
				}
															#line 1474 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				n = n->next_sibling();
															#line 7386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			results.send(listResult);
															#line 1479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (more)
															#line 7392 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!listResult.objects.empty() && lastFile < listResult.objects.back().name)
															#line 7396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					lastFile = listResult.objects.back().name;
															#line 7400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!listResult.commonPrefixes.empty() && lastFile < listResult.commonPrefixes.back())
															#line 7404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					lastFile = listResult.commonPrefixes.back();
															#line 7408 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (lastFile.empty())
															#line 7412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreEndpointListNoNextMarker") .suppressFor(60) .detail("Resource", fullResource);
															#line 1496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
			loopDepth = a_body1loopBody1cont28(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 7435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1401 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ListResult listResult;
															#line 1404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (r->code == 404)
															#line 7441 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1405 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevError, "S3BlobStoreResourceNotFound") .detail("Bucket", bucket) .detail("Prefix", prefix.present() ? prefix.get() : "") .detail("Resource", fullResource);
															#line 1409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(resource_not_found(), loopDepth);
															#line 7447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListBucketResult") != 0)
															#line 7459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7463 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1424 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* n = result->first_node();
															#line 1425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;n != nullptr;) {
															#line 1426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				const char* name = n->name();
															#line 1427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (strcmp(name, "IsTruncated") == 0)
															#line 7473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1428 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* val = n->value();
															#line 1429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(val, "true") == 0)
															#line 7479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						more = true;
															#line 7483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
					else
					{
															#line 1431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(val, "false") == 0)
															#line 7489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							more = false;
															#line 7493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
						else
						{
															#line 1434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7499 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
				else
				{
															#line 1436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(name, "Contents") == 0)
															#line 7507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1437 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						S3BlobStoreEndpoint::ObjectInfo object;
															#line 1439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						xml_node<>* key = n->first_node("Key");
															#line 1440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (key == nullptr)
															#line 7515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1441 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
															#line 1443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						object.name = key->value();
															#line 1445 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						xml_node<>* size = n->first_node("Size");
															#line 1446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (size == nullptr)
															#line 7527 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
															#line 1449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						object.size = strtoull(size->value(), nullptr, 10);
															#line 1451 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						listResult.objects.push_back(object);
															#line 7537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
					else
					{
															#line 1452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (strcmp(name, "CommonPrefixes") == 0)
															#line 7543 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 1453 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							xml_node<>* prefixNode = n->first_node("Prefix");
															#line 1454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							for(;prefixNode != nullptr;) {
															#line 1455 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								const char* prefix = prefixNode->value();
															#line 1457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								if (maxDepth > 0)
															#line 7553 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								{
															#line 1459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									if (!recurseFilter || recurseFilter(prefix))
															#line 7557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
									{
															#line 1460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
										subLists.push_back(bstore->listObjectsStream( bucket, results, prefix, delimiter, maxDepth - 1, recurseFilter));
															#line 7561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
									}
															#line 1465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									lastFile = prefix;
															#line 7565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								}
								else
								{
															#line 1467 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									listResult.commonPrefixes.push_back(prefix);
															#line 7571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								}
															#line 1470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								prefixNode = prefixNode->next_sibling("Prefix");
															#line 7575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
							}
						}
					}
				}
															#line 1474 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				n = n->next_sibling();
															#line 7582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1477 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			results.send(listResult);
															#line 1479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (more)
															#line 7588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!listResult.objects.empty() && lastFile < listResult.objects.back().name)
															#line 7592 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					lastFile = listResult.objects.back().name;
															#line 7596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (!listResult.commonPrefixes.empty() && lastFile < listResult.commonPrefixes.back())
															#line 7600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					lastFile = listResult.commonPrefixes.back();
															#line 7604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (lastFile.empty())
															#line 7608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 1493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreEndpointListNoNextMarker") .suppressFor(60) .detail("Resource", fullResource);
															#line 1496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 7614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
			loopDepth = a_body1loopBody1cont28(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(r, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjectsStream_implActor*>(this)->ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 7713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreEndpointListResultParseError") .errorUnsuppressed(e) .suppressFor(60) .detail("Resource", fullResource);
															#line 7717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 7721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont28(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListObjectsStream_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ListObjectsStream_implActorState(); static_cast<ListObjectsStream_implActor*>(this)->destroy(); return 0; }
															#line 7748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListObjectsStream_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ListObjectsStream_implActorState();
		static_cast<ListObjectsStream_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListObjectsStream_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ListObjectsStream_implActorState(); static_cast<ListObjectsStream_implActor*>(this)->destroy(); return 0; }
															#line 7760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListObjectsStream_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ListObjectsStream_implActorState();
		static_cast<ListObjectsStream_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjectsStream_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjectsStream_implActor*>(this)->ActorCallback< ListObjectsStream_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ListObjectsStream_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ListObjectsStream_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PromiseStream<S3BlobStoreEndpoint::ListResult> results;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<std::string> prefix;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<char> delimiter;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxDepth;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::function<bool(std::string const&)> recurseFilter;
															#line 1376 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string lastFile;
															#line 1385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool more;
															#line 1387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::vector<Future<Void>> subLists;
															#line 1391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser listReleaser;
															#line 1393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string fullResource;
															#line 7871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via listObjectsStream_impl()
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjectsStream_implActor final : public Actor<Void>, public ActorCallback< ListObjectsStream_implActor, 0, Void >, public ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >, public ActorCallback< ListObjectsStream_implActor, 2, Void >, public FastAllocated<ListObjectsStream_implActor>, public ListObjectsStream_implActorState<ListObjectsStream_implActor> {
															#line 7876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ListObjectsStream_implActor>::operator new;
	using FastAllocated<ListObjectsStream_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6491306212151629056UL, 7742495372137777920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListObjectsStream_implActor, 0, Void >;
friend struct ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< ListObjectsStream_implActor, 2, Void >;
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjectsStream_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,PromiseStream<S3BlobStoreEndpoint::ListResult> const& results,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 7895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   ListObjectsStream_implActorState<ListObjectsStream_implActor>(bstore, bucket, results, prefix, delimiter, maxDepth, recurseFilter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2139282905966115328UL, 13274857023515672576UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjectsStream_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listObjectsStream_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listObjectsStream_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListObjectsStream_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ListObjectsStream_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ListObjectsStream_implActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> listObjectsStream_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, PromiseStream<S3BlobStoreEndpoint::ListResult> const& results, Optional<std::string> const& prefix, Optional<char> const& delimiter, int const& maxDepth, std::function<bool(std::string const&)> const& recurseFilter ) {
															#line 1368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new ListObjectsStream_implActor(bstore, bucket, results, prefix, delimiter, maxDepth, recurseFilter));
															#line 7930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::listObjectsStream(std::string const& bucket,
                                                    PromiseStream<ListResult> results,
                                                    Optional<std::string> prefix,
                                                    Optional<char> delimiter,
                                                    int maxDepth,
                                                    std::function<bool(std::string const&)> recurseFilter) {
	return listObjectsStream_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, results, prefix, delimiter, maxDepth, recurseFilter);
}

															#line 7945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via listObjects_impl()
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ListObjects_implActor>
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjects_implActorState {
															#line 7952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjects_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   prefix(prefix),
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   delimiter(delimiter),
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   maxDepth(maxDepth),
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   recurseFilter(recurseFilter),
															#line 1530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   results(),
															#line 1531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resultStream(),
															#line 1532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   done(bstore->listObjectsStream(bucket, resultStream, prefix, delimiter, maxDepth, recurseFilter))
															#line 7975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("listObjects_impl", reinterpret_cast<unsigned long>(this));

	}
	~ListObjects_implActorState() 
	{
		fdb_probe_actor_destroy("listObjects_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			done = map(done, [=](Void) mutable { resultStream.sendError(end_of_stream()); return Void(); });
															#line 7990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			try {
															#line 1542 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				;
															#line 7994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListObjects_implActorState();
		static_cast<ListObjects_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListObjects_implActor*>(this)->SAV<S3BlobStoreEndpoint::ListResult>::futures) { (void)(results); this->~ListObjects_implActorState(); static_cast<ListObjects_implActor*>(this)->destroy(); return 0; }
															#line 8023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListObjects_implActor*>(this)->SAV< S3BlobStoreEndpoint::ListResult >::value()) S3BlobStoreEndpoint::ListResult(std::move(results)); // state_var_RVO
		this->~ListObjects_implActorState();
		static_cast<ListObjects_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_end_of_stream)
															#line 8036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 8040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = done;
															#line 1543 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListObjects_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 1549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		auto __when_expr_1 = resultStream.getFuture();
															#line 8069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<S3BlobStoreEndpoint::ListResult>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<S3BlobStoreEndpoint::ListResult>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
		static_cast<ListObjects_implActor*>(this)->actor_wait_state = 1;
															#line 1545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListObjects_implActor, 0, Void >*>(static_cast<ListObjects_implActor*>(this)));
															#line 1549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*>(static_cast<ListObjects_implActor*>(this)));
															#line 8077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 1546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 8092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 1546 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		done = Never();
															#line 8101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult const& info,int loopDepth) 
	{
															#line 1550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.commonPrefixes.insert( results.commonPrefixes.end(), info.commonPrefixes.begin(), info.commonPrefixes.end());
															#line 1552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.objects.insert(results.objects.end(), info.objects.begin(), info.objects.end());
															#line 8112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(S3BlobStoreEndpoint::ListResult && info,int loopDepth) 
	{
															#line 1550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.commonPrefixes.insert( results.commonPrefixes.end(), info.commonPrefixes.begin(), info.commonPrefixes.end());
															#line 1552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		results.objects.insert(results.objects.end(), info.objects.begin(), info.objects.end());
															#line 8123 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListObjects_implActor*>(this)->actor_wait_state > 0) static_cast<ListObjects_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListObjects_implActor*>(this)->ActorCallback< ListObjects_implActor, 0, Void >::remove();
		static_cast<ListObjects_implActor*>(this)->ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >::remove();

	}
	void a_callback_fire(ActorCallback< ListObjects_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListObjects_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListObjects_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult const& value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*,S3BlobStoreEndpoint::ListResult && value) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >*,Error err) 
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<std::string> prefix;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Optional<char> delimiter;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxDepth;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::function<bool(std::string const&)> recurseFilter;
															#line 1530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	S3BlobStoreEndpoint::ListResult results;
															#line 1531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PromiseStream<S3BlobStoreEndpoint::ListResult> resultStream;
															#line 1532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Future<Void> done;
															#line 8267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via listObjects_impl()
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListObjects_implActor final : public Actor<S3BlobStoreEndpoint::ListResult>, public ActorCallback< ListObjects_implActor, 0, Void >, public ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >, public FastAllocated<ListObjects_implActor>, public ListObjects_implActorState<ListObjects_implActor> {
															#line 8272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ListObjects_implActor>::operator new;
	using FastAllocated<ListObjects_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14993851294402418432UL, 12170159910383684352UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<S3BlobStoreEndpoint::ListResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListObjects_implActor, 0, Void >;
friend struct ActorSingleCallback< ListObjects_implActor, 1, S3BlobStoreEndpoint::ListResult >;
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListObjects_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,Optional<std::string> const& prefix,Optional<char> const& delimiter,int const& maxDepth,std::function<bool(std::string const&)> const& recurseFilter) 
															#line 8290 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<S3BlobStoreEndpoint::ListResult>(),
		   ListObjects_implActorState<ListObjects_implActor>(bstore, bucket, prefix, delimiter, maxDepth, recurseFilter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listObjects_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1153006404795665152UL, 10728906690639680256UL);
		ActorExecutionContextHelper __helper(static_cast<ListObjects_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listObjects_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listObjects_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListObjects_implActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<S3BlobStoreEndpoint::ListResult> listObjects_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, Optional<std::string> const& prefix, Optional<char> const& delimiter, int const& maxDepth, std::function<bool(std::string const&)> const& recurseFilter ) {
															#line 1524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<S3BlobStoreEndpoint::ListResult>(new ListObjects_implActor(bstore, bucket, prefix, delimiter, maxDepth, recurseFilter));
															#line 8323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1563 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<S3BlobStoreEndpoint::ListResult> S3BlobStoreEndpoint::listObjects(
    std::string const& bucket,
    Optional<std::string> prefix,
    Optional<char> delimiter,
    int maxDepth,
    std::function<bool(std::string const&)> recurseFilter) {
	return listObjects_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, prefix, delimiter, maxDepth, recurseFilter);
}

															#line 8338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via listBuckets_impl()
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ListBuckets_implActor>
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListBuckets_implActorState {
															#line 8345 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListBuckets_implActorState(Reference<S3BlobStoreEndpoint> const& bstore) 
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   resource("/?marker="),
															#line 1576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   lastName(),
															#line 1577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   more(true),
															#line 1578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   buckets()
															#line 8360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("listBuckets_impl", reinterpret_cast<unsigned long>(this));

	}
	~ListBuckets_implActorState() 
	{
		fdb_probe_actor_destroy("listBuckets_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			;
															#line 8375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListBuckets_implActorState();
		static_cast<ListBuckets_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1638 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ListBuckets_implActor*>(this)->SAV<std::vector<std::string>>::futures) { (void)(buckets); this->~ListBuckets_implActorState(); static_cast<ListBuckets_implActor*>(this)->destroy(); return 0; }
															#line 8398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ListBuckets_implActor*>(this)->SAV< std::vector<std::string> >::value()) std::vector<std::string>(std::move(buckets)); // state_var_RVO
		this->~ListBuckets_implActorState();
		static_cast<ListBuckets_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1580 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!(more))
															#line 8417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bstore->concurrentLists.take();
															#line 1581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 1;
															#line 1581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListBuckets_implActor, 0, Void >*>(static_cast<ListBuckets_implActor*>(this)));
															#line 8430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource + lastName;
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", fullResource, headers, nullptr, 0, { 200 });
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 2;
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListBuckets_implActor*>(this)));
															#line 8465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser = FlowLock::Releaser(bstore->concurrentLists, 1);
															#line 1584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		fullResource = resource + lastName;
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", fullResource, headers, nullptr, 0, { 200 });
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8482 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 2;
															#line 1586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ListBuckets_implActor*>(this)));
															#line 8487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state > 0) static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListBuckets_implActor*>(this)->ActorCallback< ListBuckets_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListBuckets_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 8571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListAllMyBucketsResult") != 0)
															#line 8583 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = false;
															#line 1604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* truncated = result->first_node("IsTruncated");
															#line 1605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (truncated != nullptr && strcmp(truncated->value(), "true") == 0)
															#line 8595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				more = true;
															#line 8599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* bucketsNode = result->first_node("Buckets");
															#line 1610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (bucketsNode != nullptr)
															#line 8605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* bucketNode = bucketsNode->first_node("Bucket");
															#line 1612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(;bucketNode != nullptr;) {
															#line 1613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					xml_node<>* nameNode = bucketNode->first_node("Name");
															#line 1614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (nameNode == nullptr)
															#line 8615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 1617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* name = nameNode->value();
															#line 1618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					buckets.push_back(name);
															#line 1620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					bucketNode = bucketNode->next_sibling("Bucket");
															#line 8627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
															#line 1624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (more)
															#line 8632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				lastName = buckets.back();
															#line 8636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		listReleaser.release();
															#line 8652 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		try {
															#line 1591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 1594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 1595 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 1598 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 1599 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result == nullptr || strcmp(result->name(), "ListAllMyBucketsResult") != 0)
															#line 8664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			more = false;
															#line 1604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* truncated = result->first_node("IsTruncated");
															#line 1605 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (truncated != nullptr && strcmp(truncated->value(), "true") == 0)
															#line 8676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				more = true;
															#line 8680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* bucketsNode = result->first_node("Buckets");
															#line 1610 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (bucketsNode != nullptr)
															#line 8686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* bucketNode = bucketsNode->first_node("Bucket");
															#line 1612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(;bucketNode != nullptr;) {
															#line 1613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					xml_node<>* nameNode = bucketNode->first_node("Name");
															#line 1614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (nameNode == nullptr)
															#line 8696 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 1615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						return a_body1loopBody1cont3Catch1(http_bad_response(), loopDepth);
															#line 8700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					}
															#line 1617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					const char* name = nameNode->value();
															#line 1618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					buckets.push_back(name);
															#line 1620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					bucketNode = bucketNode->next_sibling("Bucket");
															#line 8708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				}
			}
															#line 1624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (more)
															#line 8713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1625 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				lastName = buckets.back();
															#line 8717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(r, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ListBuckets_implActor*>(this)->actor_wait_state > 0) static_cast<ListBuckets_implActor*>(this)->actor_wait_state = 0;
		static_cast<ListBuckets_implActor*>(this)->ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1629 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 8815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				TraceEvent(SevWarn, "S3BlobStoreEndpointListBucketResultParseError") .errorUnsuppressed(e) .suppressFor(60) .detail("Resource", fullResource);
															#line 8819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1634 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(http_bad_response(), std::max(0, loopDepth - 1));
															#line 8823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1575 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string lastName;
															#line 1577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	bool more;
															#line 1578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::vector<std::string> buckets;
															#line 1582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser listReleaser;
															#line 1584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 1585 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string fullResource;
															#line 8862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via listBuckets_impl()
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ListBuckets_implActor final : public Actor<std::vector<std::string>>, public ActorCallback< ListBuckets_implActor, 0, Void >, public ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ListBuckets_implActor>, public ListBuckets_implActorState<ListBuckets_implActor> {
															#line 8867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ListBuckets_implActor>::operator new;
	using FastAllocated<ListBuckets_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9805581433743692800UL, 7869983643919847168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListBuckets_implActor, 0, Void >;
friend struct ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ListBuckets_implActor(Reference<S3BlobStoreEndpoint> const& bstore) 
															#line 8885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::vector<std::string>>(),
		   ListBuckets_implActorState<ListBuckets_implActor>(bstore),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listBuckets_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5964580658728215808UL, 13914731692276139264UL);
		ActorExecutionContextHelper __helper(static_cast<ListBuckets_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listBuckets_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listBuckets_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListBuckets_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ListBuckets_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::vector<std::string>> listBuckets_impl( Reference<S3BlobStoreEndpoint> const& bstore ) {
															#line 1574 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::vector<std::string>>(new ListBuckets_implActor(bstore));
															#line 8919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::vector<std::string>> S3BlobStoreEndpoint::listBuckets() {
	return listBuckets_impl(Reference<S3BlobStoreEndpoint>::addRef(this));
}

std::string S3BlobStoreEndpoint::hmac_sha1(Credentials const& creds, std::string const& msg) {
	std::string key = creds.secret;

	// Hash key to shorten it if it is longer than SHA1 block size
	if (key.size() > 64) {
		key = SHA1::from_string(key);
	}

	// Pad key up to SHA1 block size if needed
	key.append(64 - key.size(), '\0');

	std::string kipad = key;
	for (int i = 0; i < 64; ++i)
		kipad[i] ^= '\x36';

	std::string kopad = key;
	for (int i = 0; i < 64; ++i)
		kopad[i] ^= '\x5c';

	kipad.append(msg);
	std::string hkipad = SHA1::from_string(kipad);
	kopad.append(hkipad);
	return SHA1::from_string(kopad);
}

static void sha256(const unsigned char* data, const size_t len, unsigned char* hash) {
	SHA256_CTX sha256;
	SHA256_Init(&sha256);
	SHA256_Update(&sha256, data, len);
	SHA256_Final(hash, &sha256);
}

std::string sha256_hex(std::string str) {
	unsigned char hash[SHA256_DIGEST_LENGTH];
	sha256((const unsigned char*)str.c_str(), str.size(), hash);
	std::stringstream ss;
	for (int i = 0; i < SHA256_DIGEST_LENGTH; i++) {
		ss << std::hex << std::setw(2) << std::setfill('0') << (int)hash[i];
	}
	return ss.str();
}

// Return base64'd SHA256 hash of input string.
std::string sha256_base64(std::string str) {
	unsigned char hash[SHA256_DIGEST_LENGTH];
	sha256((const unsigned char*)str.c_str(), str.size(), hash);
	std::string hashAsStr = std::string((char*)hash, SHA256_DIGEST_LENGTH);
	std::string sig = base64::encoder::from_string(hashAsStr);
	// base64 encoded blocks end in \n so remove last character.
	sig.resize(sig.size() - 1);
	return sig;
}

std::string hmac_sha256_hex(std::string key, std::string msg) {
	unsigned char hash[32];

	HMAC_CTX* hmac = HMAC_CTX_new();
	HMAC_Init_ex(hmac, &key[0], key.length(), EVP_sha256(), NULL);
	HMAC_Update(hmac, (unsigned char*)&msg[0], msg.length());
	unsigned int len = 32;
	HMAC_Final(hmac, hash, &len);
	HMAC_CTX_free(hmac);

	std::stringstream ss;
	ss << std::hex << std::setfill('0');
	for (int i = 0; i < len; i++) {
		ss << std::hex << std::setw(2) << (unsigned int)hash[i];
	}
	return (ss.str());
}

std::string hmac_sha256(std::string key, std::string msg) {
	unsigned char hash[32];

	HMAC_CTX* hmac = HMAC_CTX_new();
	HMAC_Init_ex(hmac, &key[0], key.length(), EVP_sha256(), NULL);
	HMAC_Update(hmac, (unsigned char*)&msg[0], msg.length());
	unsigned int len = 32;
	HMAC_Final(hmac, hash, &len);
	HMAC_CTX_free(hmac);

	std::stringstream ss;
	ss << std::setfill('0');
	for (int i = 0; i < len; i++) {
		ss << hash[i];
	}
	return (ss.str());
}

// Date and Time parameters are used for unit testing
void S3BlobStoreEndpoint::setV4AuthHeaders(std::string const& verb,
                                           std::string const& resource,
                                           HTTP::Headers& headers,
                                           std::string date,
                                           std::string datestamp) {
	if (!credentials.present()) {
		return;
	}
	Credentials creds = credentials.get();
	// std::cout << "========== Starting===========" << std::endl;
	std::string accessKey = creds.key;
	std::string secretKey = creds.secret;
	// Create a date for headers and the credential string
	std::string amzDate;
	std::string dateStamp;
	if (date.empty() || datestamp.empty()) {
		time_t ts;
		time(&ts);
		char dateBuf[20];
		// ISO 8601 format YYYYMMDD'T'HHMMSS'Z'
		strftime(dateBuf, 20, "%Y%m%dT%H%M%SZ", gmtime(&ts));
		amzDate = dateBuf;
		strftime(dateBuf, 20, "%Y%m%d", gmtime(&ts));
		dateStamp = dateBuf;
	} else {
		amzDate = date;
		dateStamp = datestamp;
	}

	// ************* TASK 1: CREATE A CANONICAL REQUEST *************
	// Create Create canonical URI--the part of the URI from domain to query string (use '/' if no path)
	std::vector<std::string> queryParameters;
	std::string canonicalURI = awsCanonicalURI(resource, queryParameters, true);

	std::string canonicalQueryString;
	if (!queryParameters.empty()) {
		std::sort(queryParameters.begin(), queryParameters.end());
		canonicalQueryString = boost::algorithm::join(queryParameters, "&");
	}

	using namespace boost::algorithm;
	// Create the canonical headers and signed headers
	ASSERT(!headers["Host"].empty());
	// Be careful. There is x-amz-content-sha256 for auth and then
	// x-amz-checksum-sha256 for object integrity check.
	headers["x-amz-content-sha256"] = "UNSIGNED-PAYLOAD";
	headers["x-amz-date"] = amzDate;
	std::vector<std::pair<std::string, std::string>> headersList;
	headersList.push_back({ "host", trim_copy(headers["Host"]) + "\n" });
	if (headers.find("Content-Type") != headers.end())
		headersList.push_back({ "content-type", trim_copy(headers["Content-Type"]) + "\n" });
	if (headers.find("Content-MD5") != headers.end())
		headersList.push_back({ "content-md5", trim_copy(headers["Content-MD5"]) + "\n" });
	for (auto h : headers) {
		if (StringRef(h.first).startsWith("x-amz"_sr))
			headersList.push_back({ to_lower_copy(h.first), trim_copy(h.second) + "\n" });
	}
	std::sort(headersList.begin(), headersList.end());
	std::string canonicalHeaders;
	std::string signedHeaders;
	for (auto& i : headersList) {
		canonicalHeaders += i.first + ":" + i.second;
		signedHeaders += i.first + ";";
	}
	signedHeaders.pop_back();
	std::string canonicalRequest = verb + "\n" + canonicalURI + "\n" + canonicalQueryString + "\n" + canonicalHeaders +
	                               "\n" + signedHeaders + "\n" + headers["x-amz-content-sha256"];

	// ************* TASK 2: CREATE THE STRING TO SIGN*************
	std::string algorithm = "AWS4-HMAC-SHA256";
	std::string credentialScope = dateStamp + "/" + region + "/s3/" + "aws4_request";
	std::string stringToSign =
	    algorithm + "\n" + amzDate + "\n" + credentialScope + "\n" + sha256_hex(canonicalRequest);

	// ************* TASK 3: CALCULATE THE SIGNATURE *************
	// Create the signing key using the function defined above.
	std::string signingKey =
	    hmac_sha256(hmac_sha256(hmac_sha256(hmac_sha256("AWS4" + secretKey, dateStamp), region), "s3"), "aws4_request");
	// Sign the string_to_sign using the signing_key
	std::string signature = hmac_sha256_hex(signingKey, stringToSign);
	// ************* TASK 4: ADD SIGNING INFORMATION TO THE Header *************
	std::string authorizationHeader = algorithm + " " + "Credential=" + accessKey + "/" + credentialScope + ", " +
	                                  "SignedHeaders=" + signedHeaders + ", " + "Signature=" + signature;
	headers["Authorization"] = authorizationHeader;
}

void S3BlobStoreEndpoint::setAuthHeaders(std::string const& verb, std::string const& resource, HTTP::Headers& headers) {
	if (!credentials.present()) {
		return;
	}
	Credentials creds = credentials.get();

	std::string& date = headers["Date"];

	char dateBuf[64];
	time_t ts;
	time(&ts);
	strftime(dateBuf, 64, "%a, %d %b %Y %H:%M:%S GMT", gmtime(&ts));
	date = dateBuf;

	std::string msg;
	msg.append(verb);
	msg.append("\n");
	auto contentMD5 = headers.find("Content-MD5");
	if (contentMD5 != headers.end())
		msg.append(contentMD5->second);
	msg.append("\n");
	auto contentType = headers.find("Content-Type");
	if (contentType != headers.end())
		msg.append(contentType->second);
	msg.append("\n");
	msg.append(date);
	msg.append("\n");
	for (auto h : headers) {
		StringRef name = h.first;
		if (name.startsWith("x-amz"_sr) || name.startsWith("x-icloud"_sr)) {
			msg.append(h.first);
			msg.append(":");
			msg.append(h.second);
			msg.append("\n");
		}
	}

	msg.append(resource);
	if (verb == "GET") {
		size_t q = resource.find_last_of('?');
		if (q != resource.npos)
			msg.resize(msg.size() - (resource.size() - q));
	}

	std::string sig = base64::encoder::from_string(hmac_sha1(creds, msg));
	// base64 encoded blocks end in \n so remove it.
	sig.resize(sig.size() - 1);
	std::string auth = "AWS ";
	auth.append(creds.key);
	auth.append(":");
	auth.append(sig);
	headers["Authorization"] = auth;
}

															#line 9158 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via readEntireFile_impl()
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ReadEntireFile_implActor>
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadEntireFile_implActorState {
															#line 9165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadEntireFile_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 9176 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("readEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	~ReadEntireFile_implActorState() 
	{
		fdb_probe_actor_destroy("readEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateRead->getAllowance(1);
															#line 1878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 1;
															#line 1878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadEntireFile_implActor, 0, Void >*>(static_cast<ReadEntireFile_implActor*>(this)));
															#line 9198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadEntireFile_implActorState();
		static_cast<ReadEntireFile_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["x-amz-checksum-mode"] = "ENABLED";
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", resource, headers, nullptr, 0, { 200, 404 });
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadEntireFile_implActor*>(this)));
															#line 9234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["x-amz-checksum-mode"] = "ENABLED";
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", resource, headers, nullptr, 0, { 200, 404 });
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9251 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1885 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadEntireFile_implActor*>(this)));
															#line 9256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadEntireFile_implActor*>(this)->ActorCallback< ReadEntireFile_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadEntireFile_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 9340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 9344 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string checksumSHA256 = r->data.headers["x-amz-checksum-sha256"];
															#line 1894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256.empty())
															#line 9354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 9358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string contentSHA256 = sha256_base64(r->data.content);
															#line 1900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256 != contentSHA256)
															#line 9364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(checksum_failed(), loopDepth);
															#line 9368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ReadEntireFile_implActor*>(this)->SAV<std::string>::futures) { (void)(r->data.content); this->~ReadEntireFile_implActorState(); static_cast<ReadEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 9373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ReadEntireFile_implActor*>(this)->SAV< std::string >::value()) std::string(r->data.content);
		this->~ReadEntireFile_implActorState();
		static_cast<ReadEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1886 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 9385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(file_not_found(), loopDepth);
															#line 9389 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1893 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string checksumSHA256 = r->data.headers["x-amz-checksum-sha256"];
															#line 1894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256.empty())
															#line 9399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(http_bad_response(), loopDepth);
															#line 9403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string contentSHA256 = sha256_base64(r->data.content);
															#line 1900 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (checksumSHA256 != contentSHA256)
															#line 9409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(checksum_failed(), loopDepth);
															#line 9413 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
		}
															#line 1904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<ReadEntireFile_implActor*>(this)->SAV<std::string>::futures) { (void)(r->data.content); this->~ReadEntireFile_implActorState(); static_cast<ReadEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 9418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<ReadEntireFile_implActor*>(this)->SAV< std::string >::value()) std::string(r->data.content);
		this->~ReadEntireFile_implActorState();
		static_cast<ReadEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<ReadEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadEntireFile_implActor*>(this)->ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 1880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 9511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via readEntireFile_impl()
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadEntireFile_implActor final : public Actor<std::string>, public ActorCallback< ReadEntireFile_implActor, 0, Void >, public ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ReadEntireFile_implActor>, public ReadEntireFile_implActorState<ReadEntireFile_implActor> {
															#line 9516 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ReadEntireFile_implActor>::operator new;
	using FastAllocated<ReadEntireFile_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(464499385056073984UL, 11908728997948563200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadEntireFile_implActor, 0, Void >;
friend struct ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadEntireFile_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 9534 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::string>(),
		   ReadEntireFile_implActorState<ReadEntireFile_implActor>(bstore, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9869764261496093184UL, 2881618862979785984UL);
		ActorExecutionContextHelper __helper(static_cast<ReadEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readEntireFile_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadEntireFile_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadEntireFile_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::string> readEntireFile_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object ) {
															#line 1875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::string>(new ReadEntireFile_implActor(bstore, bucket, object));
															#line 9568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::string> S3BlobStoreEndpoint::readEntireFile(std::string const& bucket, std::string const& object) {
	return readEntireFile_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 9577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via writeEntireFileFromBuffer_impl()
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class WriteEntireFileFromBuffer_implActor>
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFileFromBuffer_implActorState {
															#line 9584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFileFromBuffer_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentHash) 
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pContent(pContent),
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentLen(contentLen),
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentHash(contentHash)
															#line 9601 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this));

	}
	~WriteEntireFileFromBuffer_implActorState() 
	{
		fdb_probe_actor_destroy("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (contentLen > bstore->knobs.multipart_max_part_size)
															#line 9616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(file_too_large(), loopDepth);
															#line 9620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 1920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 1;
															#line 1920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteEntireFileFromBuffer_implActorState();
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 1921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 2;
															#line 1921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 1921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 2;
															#line 1921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 1924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-sha256"] = contentHash;
															#line 1932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 9773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-MD5"] = contentHash;
															#line 9779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 9783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 9787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = bstore->doRequest("PUT", resource, headers, pContent, contentLen, { 200 });
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 3;
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 1924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 1925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 1930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (bstore->knobs.enable_object_integrity_check)
															#line 9813 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-sha256"] = contentHash;
															#line 1932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-checksum-algorithm"] = "SHA256";
															#line 9819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-MD5"] = contentHash;
															#line 9825 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 9829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 9833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = bstore->doRequest("PUT", resource, headers, pContent, contentLen, { 200 });
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 3;
															#line 1938 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<WriteEntireFileFromBuffer_implActor*>(this)));
															#line 9844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 1940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFileFromBuffer_implActorState(); static_cast<WriteEntireFileFromBuffer_implActor*>(this)->destroy(); return 0; }
															#line 9928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFileFromBuffer_implActorState();
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 1940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFileFromBuffer_implActorState(); static_cast<WriteEntireFileFromBuffer_implActor*>(this)->destroy(); return 0; }
															#line 9940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFileFromBuffer_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFileFromBuffer_implActorState();
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont4(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFileFromBuffer_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFileFromBuffer_implActor*>(this)->ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue* pContent;
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int contentLen;
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string contentHash;
															#line 1922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser uploadReleaser;
															#line 1924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 1925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 10041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via writeEntireFileFromBuffer_impl()
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFileFromBuffer_implActor final : public Actor<Void>, public ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >, public ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >, public ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >, public FastAllocated<WriteEntireFileFromBuffer_implActor>, public WriteEntireFileFromBuffer_implActorState<WriteEntireFileFromBuffer_implActor> {
															#line 10046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<WriteEntireFileFromBuffer_implActor>::operator new;
	using FastAllocated<WriteEntireFileFromBuffer_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11698858338014824960UL, 10426519363300760320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >;
friend struct ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >;
friend struct ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >;
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFileFromBuffer_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentHash) 
															#line 10065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   WriteEntireFileFromBuffer_implActorState<WriteEntireFileFromBuffer_implActor>(bstore, bucket, object, pContent, contentLen, contentHash),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15594163113904011264UL, 9204598572552334336UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFileFromBuffer_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeEntireFileFromBuffer_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeEntireFileFromBuffer_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteEntireFileFromBuffer_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteEntireFileFromBuffer_implActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteEntireFileFromBuffer_implActor, 2, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> writeEntireFileFromBuffer_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, UnsentPacketQueue* const& pContent, int const& contentLen, std::string const& contentHash ) {
															#line 1911 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new WriteEntireFileFromBuffer_implActor(bstore, bucket, object, pContent, contentLen, contentHash));
															#line 10100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 10105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via writeEntireFile_impl()
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class WriteEntireFile_implActor>
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFile_implActorState {
															#line 10112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFile_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& content) 
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   content(content),
															#line 1947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   packets()
															#line 10127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("writeEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	~WriteEntireFile_implActorState() 
	{
		fdb_probe_actor_destroy("writeEntireFile_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (content.size() > bstore->knobs.multipart_max_part_size)
															#line 10142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 1949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(file_too_large(), loopDepth);
															#line 10146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 1951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			PacketWriter pw(packets.getWriteBuffer(content.size()), nullptr, Unversioned());
															#line 1952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			pw.serializeBytes(content);
															#line 1958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = yield();
															#line 1958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 1;
															#line 1958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFile_implActor, 0, Void >*>(static_cast<WriteEntireFile_implActor*>(this)));
															#line 10161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteEntireFile_implActorState();
		static_cast<WriteEntireFile_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string contentHash;
															#line 1970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (CLIENT_KNOBS->BLOBSTORE_ENABLE_OBJECT_INTEGRITY_CHECK)
															#line 10186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = sha256_base64(content);
															#line 10190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			MD5_CTX sum;
															#line 1975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Init(&sum);
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Update(&sum, content.data(), content.size());
															#line 1977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string sumBytes;
															#line 1978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			sumBytes.resize(16);
															#line 1979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Final((unsigned char*)sumBytes.data(), &sum);
															#line 1980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = base64::encoder::from_string(sumBytes);
															#line 1981 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash.resize(contentHash.size() - 1);
															#line 10210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = writeEntireFileFromBuffer_impl(bstore, bucket, object, &packets, content.size(), contentHash);
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFile_implActor, 1, Void >*>(static_cast<WriteEntireFile_implActor*>(this)));
															#line 10221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string contentHash;
															#line 1970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (CLIENT_KNOBS->BLOBSTORE_ENABLE_OBJECT_INTEGRITY_CHECK)
															#line 10232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 1972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = sha256_base64(content);
															#line 10236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		else
		{
															#line 1974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			MD5_CTX sum;
															#line 1975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Init(&sum);
															#line 1976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Update(&sum, content.data(), content.size());
															#line 1977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string sumBytes;
															#line 1978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			sumBytes.resize(16);
															#line 1979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			::MD5_Final((unsigned char*)sumBytes.data(), &sum);
															#line 1980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash = base64::encoder::from_string(sumBytes);
															#line 1981 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			contentHash.resize(contentHash.size() - 1);
															#line 10256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = writeEntireFileFromBuffer_impl(bstore, bucket, object, &packets, content.size(), contentHash);
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 2;
															#line 1984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEntireFile_implActor, 1, Void >*>(static_cast<WriteEntireFile_implActor*>(this)));
															#line 10267 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFile_implActor*>(this)->ActorCallback< WriteEntireFile_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteEntireFile_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFile_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFile_implActorState(); static_cast<WriteEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 10351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFile_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFile_implActorState();
		static_cast<WriteEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<WriteEntireFile_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteEntireFile_implActorState(); static_cast<WriteEntireFile_implActor*>(this)->destroy(); return 0; }
															#line 10363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<WriteEntireFile_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteEntireFile_implActorState();
		static_cast<WriteEntireFile_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state > 0) static_cast<WriteEntireFile_implActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEntireFile_implActor*>(this)->ActorCallback< WriteEntireFile_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteEntireFile_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteEntireFile_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string content;
															#line 1947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue packets;
															#line 10456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via writeEntireFile_impl()
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class WriteEntireFile_implActor final : public Actor<Void>, public ActorCallback< WriteEntireFile_implActor, 0, Void >, public ActorCallback< WriteEntireFile_implActor, 1, Void >, public FastAllocated<WriteEntireFile_implActor>, public WriteEntireFile_implActorState<WriteEntireFile_implActor> {
															#line 10461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<WriteEntireFile_implActor>::operator new;
	using FastAllocated<WriteEntireFile_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15420081797857550848UL, 18403681503561470464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteEntireFile_implActor, 0, Void >;
friend struct ActorCallback< WriteEntireFile_implActor, 1, Void >;
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	WriteEntireFile_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& content) 
															#line 10479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   WriteEntireFile_implActorState<WriteEntireFile_implActor>(bstore, bucket, object, content),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18357809352626313472UL, 5788567832381847040UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEntireFile_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeEntireFile_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeEntireFile_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteEntireFile_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteEntireFile_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> writeEntireFile_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& content ) {
															#line 1943 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new WriteEntireFile_implActor(bstore, bucket, object, content));
															#line 10513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 1987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::writeEntireFile(std::string const& bucket,
                                                  std::string const& object,
                                                  std::string const& content) {
	return writeEntireFile_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, content);
}

Future<Void> S3BlobStoreEndpoint::writeEntireFileFromBuffer(std::string const& bucket,
                                                            std::string const& object,
                                                            UnsentPacketQueue* pContent,
                                                            int contentLen,
                                                            std::string const& contentHash) {
	return writeEntireFileFromBuffer_impl(
	    Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, pContent, contentLen, contentHash);
}

															#line 10533 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via readObject_impl()
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class ReadObject_implActor>
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadObject_implActorState {
															#line 10540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadObject_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,void* const& data,int const& length,int64_t const& offset) 
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   data(data),
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   length(length),
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   offset(offset)
															#line 10557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("readObject_impl", reinterpret_cast<unsigned long>(this));

	}
	~ReadObject_implActorState() 
	{
		fdb_probe_actor_destroy("readObject_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 2010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (length <= 0)
															#line 10573 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 2011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					TraceEvent(SevWarn, "S3BlobStoreReadObjectEmptyRead").detail("Length", length);
															#line 2012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<ReadObject_implActor*>(this)->SAV<int>::futures) { (void)(0); this->~ReadObject_implActorState(); static_cast<ReadObject_implActor*>(this)->destroy(); return 0; }
															#line 10579 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<ReadObject_implActor*>(this)->SAV< int >::value()) int(0);
					this->~ReadObject_implActorState();
					static_cast<ReadObject_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 2016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				StrictFuture<Void> __when_expr_0 = bstore->requestRateRead->getAllowance(1);
															#line 2016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (static_cast<ReadObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 10589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadObject_implActor*>(this)->actor_wait_state = 1;
															#line 2016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadObject_implActor, 0, Void >*>(static_cast<ReadObject_implActor*>(this)));
															#line 10594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadObject_implActorState();
		static_cast<ReadObject_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreEndpoint_ReadError") .error(e) .detail("Bucket", bucket) .detail("Object", object) .detail("Length", length) .detail("Offset", offset);
															#line 2056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 10626 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["Range"] = format("bytes=%lld-%lld", offset, offset + length - 1);
															#line 2023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", resource, headers, nullptr, 0, { 200, 206, 404 });
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 10650 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadObject_implActor*>(this)->actor_wait_state = 2;
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadObject_implActor*>(this)));
															#line 10655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["Range"] = format("bytes=%lld-%lld", offset, offset + length - 1);
															#line 2023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = Reference<HTTP::IncomingResponse>();
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", resource, headers, nullptr, 0, { 200, 206, 404 });
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 10674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadObject_implActor*>(this)->actor_wait_state = 2;
															#line 2024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<ReadObject_implActor*>(this)));
															#line 10679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state > 0) static_cast<ReadObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadObject_implActor*>(this)->ActorCallback< ReadObject_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadObject_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
															#line 2026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 2028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 10765 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(file_not_found(), loopDepth);
															#line 10769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.contentLen != r->data.content.size())
															#line 10773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreReadObjectContentLengthMismatch") .detail("Expected", r->data.contentLen) .detail("Actual", r->data.content.size());
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(io_error(), loopDepth);
															#line 10779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		try {
															#line 2043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			memcpy(data, r->data.content.data(), std::min<int64_t>(r->data.contentLen, length));
															#line 2044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<ReadObject_implActor*>(this)->SAV<int>::futures) { (void)(r->data.contentLen); this->~ReadObject_implActorState(); static_cast<ReadObject_implActor*>(this)->destroy(); return 0; }
															#line 10786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<ReadObject_implActor*>(this)->SAV< int >::value()) int(r->data.contentLen);
			this->~ReadObject_implActorState();
			static_cast<ReadObject_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
															#line 2026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = _r;
															#line 2028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->code == 404)
															#line 10806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(file_not_found(), loopDepth);
															#line 10810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (r->data.contentLen != r->data.content.size())
															#line 10814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreReadObjectContentLengthMismatch") .detail("Expected", r->data.contentLen) .detail("Actual", r->data.content.size());
															#line 2037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(io_error(), loopDepth);
															#line 10820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		try {
															#line 2043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			memcpy(data, r->data.content.data(), std::min<int64_t>(r->data.contentLen, length));
															#line 2044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<ReadObject_implActor*>(this)->SAV<int>::futures) { (void)(r->data.contentLen); this->~ReadObject_implActorState(); static_cast<ReadObject_implActor*>(this)->destroy(); return 0; }
															#line 10827 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<ReadObject_implActor*>(this)->SAV< int >::value()) int(r->data.contentLen);
			this->~ReadObject_implActorState();
			static_cast<ReadObject_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont4(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> && _r,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadObject_implActor*>(this)->actor_wait_state > 0) static_cast<ReadObject_implActor*>(this)->actor_wait_state = 0;
		static_cast<ReadObject_implActor*>(this)->ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStoreReadObjectMemcpyError").detail("Error", e.what());
															#line 2047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch2(io_error(), loopDepth);
															#line 10923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	void* data;
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int length;
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int64_t offset;
															#line 2018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 2023 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 10951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via readObject_impl()
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class ReadObject_implActor final : public Actor<int>, public ActorCallback< ReadObject_implActor, 0, Void >, public ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<ReadObject_implActor>, public ReadObject_implActorState<ReadObject_implActor> {
															#line 10956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<ReadObject_implActor>::operator new;
	using FastAllocated<ReadObject_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10594217179425133568UL, 4832746645433443584UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadObject_implActor, 0, Void >;
friend struct ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	ReadObject_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,void* const& data,int const& length,int64_t const& offset) 
															#line 10974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<int>(),
		   ReadObject_implActorState<ReadObject_implActor>(bstore, bucket, object, data, length, offset),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readObject_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7919612023543018496UL, 14030950089782221056UL);
		ActorExecutionContextHelper __helper(static_cast<ReadObject_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readObject_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readObject_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadObject_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadObject_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<int> readObject_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, void* const& data, int const& length, int64_t const& offset ) {
															#line 2003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<int>(new ReadObject_implActor(bstore, bucket, object, data, length, offset));
															#line 11008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2059 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<int> S3BlobStoreEndpoint::readObject(std::string const& bucket,
                                            std::string const& object,
                                            void* data,
                                            int length,
                                            int64_t offset) {
	return readObject_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, data, length, offset);
}

															#line 11021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via beginMultiPartUpload_impl()
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class BeginMultiPartUpload_implActor>
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BeginMultiPartUpload_implActorState {
															#line 11028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BeginMultiPartUpload_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 11039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	~BeginMultiPartUpload_implActorState() 
	{
		fdb_probe_actor_destroy("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 1;
															#line 2071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*>(static_cast<BeginMultiPartUpload_implActor*>(this)));
															#line 11061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BeginMultiPartUpload_implActorState();
		static_cast<BeginMultiPartUpload_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?uploads";
															#line 2075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 11090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 11094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("POST", resource, headers, nullptr, 0, { 200 });
															#line 2078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BeginMultiPartUpload_implActor*>(this)));
															#line 11105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?uploads";
															#line 2075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2076 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE.empty())
															#line 11120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["x-amz-server-side-encryption"] = CLIENT_KNOBS->BLOBSTORE_ENCRYPTION_TYPE;
															#line 11124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("POST", resource, headers, nullptr, 0, { 200 });
															#line 2078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<BeginMultiPartUpload_implActor*>(this)));
															#line 11135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<BeginMultiPartUpload_implActor*>(this)->ActorCallback< BeginMultiPartUpload_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		try {
															#line 2081 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 2083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 2085 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 2088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 2089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result != nullptr && strcmp(result->name(), "InitiateMultipartUploadResult") == 0)
															#line 11228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* id = result->first_node("UploadId");
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (id != nullptr)
															#line 11234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<BeginMultiPartUpload_implActor*>(this)->SAV<std::string>::futures) { (void)(id->value()); this->~BeginMultiPartUpload_implActorState(); static_cast<BeginMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 11238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<BeginMultiPartUpload_implActor*>(this)->SAV< std::string >::value()) std::string(id->value());
					this->~BeginMultiPartUpload_implActorState();
					static_cast<BeginMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
			loopDepth = a_body1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		try {
															#line 2081 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_document<> doc;
															#line 2083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string content = r->data.content;
															#line 2085 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			doc.parse<0>((char*)content.c_str());
															#line 2088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			xml_node<>* result = doc.first_node();
															#line 2089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (result != nullptr && strcmp(result->name(), "InitiateMultipartUploadResult") == 0)
															#line 11268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				xml_node<>* id = result->first_node("UploadId");
															#line 2091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				if (id != nullptr)
															#line 11274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				{
															#line 2092 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (!static_cast<BeginMultiPartUpload_implActor*>(this)->SAV<std::string>::futures) { (void)(id->value()); this->~BeginMultiPartUpload_implActorState(); static_cast<BeginMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 11278 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					new (&static_cast<BeginMultiPartUpload_implActor*>(this)->SAV< std::string >::value()) std::string(id->value());
					this->~BeginMultiPartUpload_implActorState();
					static_cast<BeginMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
			loopDepth = a_body1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<BeginMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<BeginMultiPartUpload_implActor*>(this)->ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		return a_body1Catch1(http_bad_response(), loopDepth);
															#line 11374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont2Catch1(const Error& __current_error,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 11414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via beginMultiPartUpload_impl()
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class BeginMultiPartUpload_implActor final : public Actor<std::string>, public ActorCallback< BeginMultiPartUpload_implActor, 0, Void >, public ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<BeginMultiPartUpload_implActor>, public BeginMultiPartUpload_implActorState<BeginMultiPartUpload_implActor> {
															#line 11419 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<BeginMultiPartUpload_implActor>::operator new;
	using FastAllocated<BeginMultiPartUpload_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13705612607315325952UL, 2493551820955568128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BeginMultiPartUpload_implActor, 0, Void >;
friend struct ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	BeginMultiPartUpload_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 11437 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::string>(),
		   BeginMultiPartUpload_implActorState<BeginMultiPartUpload_implActor>(bstore, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4949678732428246272UL, 2657853908492627456UL);
		ActorExecutionContextHelper __helper(static_cast<BeginMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("beginMultiPartUpload_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("beginMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BeginMultiPartUpload_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BeginMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] static Future<std::string> beginMultiPartUpload_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object ) {
															#line 2068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::string>(new BeginMultiPartUpload_implActor(bstore, bucket, object));
															#line 11471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2099 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::string> S3BlobStoreEndpoint::beginMultiPartUpload(std::string const& bucket, std::string const& object) {
	return beginMultiPartUpload_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 11480 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via uploadPart_impl()
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class UploadPart_implActor>
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UploadPart_implActorState {
															#line 11487 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UploadPart_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,unsigned int const& partNumber,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentMD5) 
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   uploadID(uploadID),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   partNumber(partNumber),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   pContent(pContent),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentLen(contentLen),
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   contentMD5(contentMD5)
															#line 11508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("uploadPart_impl", reinterpret_cast<unsigned long>(this));

	}
	~UploadPart_implActorState() 
	{
		fdb_probe_actor_destroy("uploadPart_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UploadPart_implActor*>(this)->actor_wait_state = 1;
															#line 2112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 0, Void >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UploadPart_implActorState();
		static_cast<UploadPart_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 2;
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 1, Void >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bstore->concurrentUploads.take();
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 2;
															#line 2113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 1, Void >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state > 0) static_cast<UploadPart_implActor*>(this)->actor_wait_state = 0;
		static_cast<UploadPart_implActor*>(this)->ActorCallback< UploadPart_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UploadPart_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 2116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?partNumber=%d&uploadId=%s", partNumber, uploadID.c_str());
															#line 2118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["Content-MD5"] = contentMD5;
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = bstore->doRequest("PUT", resource, headers, pContent, contentLen, { 200 });
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11672 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 3;
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		uploadReleaser = FlowLock::Releaser(bstore->concurrentUploads, 1);
															#line 2116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?partNumber=%d&uploadId=%s", partNumber, uploadID.c_str());
															#line 2118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers["Content-MD5"] = contentMD5;
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_2 = bstore->doRequest("PUT", resource, headers, pContent, contentLen, { 200 });
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11698 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<UploadPart_implActor*>(this)->actor_wait_state = 3;
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*>(static_cast<UploadPart_implActor*>(this)));
															#line 11703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state > 0) static_cast<UploadPart_implActor*>(this)->actor_wait_state = 0;
		static_cast<UploadPart_implActor*>(this)->ActorCallback< UploadPart_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< UploadPart_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!HTTP::verifyMD5(&r->data, false, contentMD5))
															#line 11787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(checksum_failed(), loopDepth);
															#line 11791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string etag = r->data.headers["ETag"];
															#line 2133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (etag.empty())
															#line 11797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2134 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1Catch1(http_bad_response(), loopDepth);
															#line 11801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2136 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<UploadPart_implActor*>(this)->SAV<std::string>::futures) { (void)(etag); this->~UploadPart_implActorState(); static_cast<UploadPart_implActor*>(this)->destroy(); return 0; }
															#line 11805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<UploadPart_implActor*>(this)->SAV< std::string >::value()) std::string(etag);
		this->~UploadPart_implActorState();
		static_cast<UploadPart_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> const& __r,int loopDepth) 
	{
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		r = __r;
															#line 11817 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Reference<HTTP::IncomingResponse> && __r,int loopDepth) 
	{
		r = std::move(__r);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<UploadPart_implActor*>(this)->actor_wait_state > 0) static_cast<UploadPart_implActor*>(this)->actor_wait_state = 0;
		static_cast<UploadPart_implActor*>(this)->ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string uploadID;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	unsigned int partNumber;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue* pContent;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int contentLen;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string contentMD5;
															#line 2114 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowLock::Releaser uploadReleaser;
															#line 2116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 2121 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<HTTP::IncomingResponse> r;
															#line 11916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via uploadPart_impl()
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class UploadPart_implActor final : public Actor<std::string>, public ActorCallback< UploadPart_implActor, 0, Void >, public ActorCallback< UploadPart_implActor, 1, Void >, public ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >, public FastAllocated<UploadPart_implActor>, public UploadPart_implActorState<UploadPart_implActor> {
															#line 11921 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<UploadPart_implActor>::operator new;
	using FastAllocated<UploadPart_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8828458547624225280UL, 8649983621687054080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UploadPart_implActor, 0, Void >;
friend struct ActorCallback< UploadPart_implActor, 1, Void >;
friend struct ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >;
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UploadPart_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,unsigned int const& partNumber,UnsentPacketQueue* const& pContent,int const& contentLen,std::string const& contentMD5) 
															#line 11940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::string>(),
		   UploadPart_implActorState<UploadPart_implActor>(bstore, bucket, object, uploadID, partNumber, pContent, contentLen, contentMD5),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("uploadPart_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1338121866253869824UL, 10943970907985309952UL);
		ActorExecutionContextHelper __helper(static_cast<UploadPart_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("uploadPart_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("uploadPart_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UploadPart_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< UploadPart_implActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< UploadPart_implActor, 2, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::string> uploadPart_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& uploadID, unsigned int const& partNumber, UnsentPacketQueue* const& pContent, int const& contentLen, std::string const& contentMD5 ) {
															#line 2104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::string>(new UploadPart_implActor(bstore, bucket, object, uploadID, partNumber, pContent, contentLen, contentMD5));
															#line 11975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::string> S3BlobStoreEndpoint::uploadPart(std::string const& bucket,
                                                    std::string const& object,
                                                    std::string const& uploadID,
                                                    unsigned int partNumber,
                                                    UnsentPacketQueue* pContent,
                                                    int contentLen,
                                                    std::string const& contentMD5) {
	return uploadPart_impl(Reference<S3BlobStoreEndpoint>::addRef(this),
	                       bucket,
	                       object,
	                       uploadID,
	                       partNumber,
	                       pContent,
	                       contentLen,
	                       contentMD5);
}

															#line 11997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via finishMultiPartUpload_impl()
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FinishMultiPartUpload_implActor>
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FinishMultiPartUpload_implActorState {
															#line 12004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FinishMultiPartUpload_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,S3BlobStoreEndpoint::MultiPartSetT const& parts) 
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   uploadID(uploadID),
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   parts(parts),
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   part_list()
															#line 12021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	~FinishMultiPartUpload_implActorState() 
	{
		fdb_probe_actor_destroy("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 1;
															#line 2162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*>(static_cast<FinishMultiPartUpload_implActor*>(this)));
															#line 12043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinishMultiPartUpload_implActorState();
		static_cast<FinishMultiPartUpload_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string manifest = "<CompleteMultipartUpload>";
															#line 2165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& p : parts ) {
															#line 2166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += format("<Part><PartNumber>%d</PartNumber><ETag>%s</ETag></Part>\n", p.first, p.second.c_str());
															#line 12070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		manifest += "</CompleteMultipartUpload>";
															#line 2169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2171 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		PacketWriter pw(part_list.getWriteBuffer(manifest.size()), nullptr, Unversioned());
															#line 2173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		pw.serializeBytes(manifest);
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("POST", resource, headers, &part_list, manifest.size(), { 200 });
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<FinishMultiPartUpload_implActor*>(this)));
															#line 12093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string manifest = "<CompleteMultipartUpload>";
															#line 2165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for( auto& p : parts ) {
															#line 2166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += format("<Part><PartNumber>%d</PartNumber><ETag>%s</ETag></Part>\n", p.first, p.second.c_str());
															#line 12106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		manifest += "</CompleteMultipartUpload>";
															#line 2169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2171 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		PacketWriter pw(part_list.getWriteBuffer(manifest.size()), nullptr, Unversioned());
															#line 2173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		pw.serializeBytes(manifest);
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("POST", resource, headers, &part_list, manifest.size(), { 200 });
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12124 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<FinishMultiPartUpload_implActor*>(this)));
															#line 12129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMultiPartUpload_implActor*>(this)->ActorCallback< FinishMultiPartUpload_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<FinishMultiPartUpload_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinishMultiPartUpload_implActorState(); static_cast<FinishMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<FinishMultiPartUpload_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinishMultiPartUpload_implActorState();
		static_cast<FinishMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<FinishMultiPartUpload_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FinishMultiPartUpload_implActorState(); static_cast<FinishMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<FinishMultiPartUpload_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FinishMultiPartUpload_implActorState();
		static_cast<FinishMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<FinishMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMultiPartUpload_implActor*>(this)->ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string uploadID;
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	S3BlobStoreEndpoint::MultiPartSetT parts;
															#line 2161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue part_list;
															#line 2169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2171 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 12324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via finishMultiPartUpload_impl()
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FinishMultiPartUpload_implActor final : public Actor<Void>, public ActorCallback< FinishMultiPartUpload_implActor, 0, Void >, public ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<FinishMultiPartUpload_implActor>, public FinishMultiPartUpload_implActorState<FinishMultiPartUpload_implActor> {
															#line 12329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FinishMultiPartUpload_implActor>::operator new;
	using FastAllocated<FinishMultiPartUpload_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5432880139287455488UL, 17912161297001473792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FinishMultiPartUpload_implActor, 0, Void >;
friend struct ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FinishMultiPartUpload_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID,S3BlobStoreEndpoint::MultiPartSetT const& parts) 
															#line 12347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   FinishMultiPartUpload_implActorState<FinishMultiPartUpload_implActor>(bstore, bucket, object, uploadID, parts),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(499877118889630720UL, 2657922083509416704UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("finishMultiPartUpload_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("finishMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FinishMultiPartUpload_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FinishMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> finishMultiPartUpload_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& uploadID, S3BlobStoreEndpoint::MultiPartSetT const& parts ) {
															#line 2156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new FinishMultiPartUpload_implActor(bstore, bucket, object, uploadID, parts));
															#line 12381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::finishMultiPartUpload(std::string const& bucket,
                                                        std::string const& object,
                                                        std::string const& uploadID,
                                                        MultiPartSetT const& parts) {
	return finishMultiPartUpload_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, uploadID, parts);
}

															#line 12393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via abortMultiPartUpload_impl()
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class AbortMultiPartUpload_implActor>
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class AbortMultiPartUpload_implActorState {
															#line 12400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	AbortMultiPartUpload_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID) 
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   uploadID(uploadID)
															#line 12413 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	~AbortMultiPartUpload_implActorState() 
	{
		fdb_probe_actor_destroy("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 1;
															#line 2194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*>(static_cast<AbortMultiPartUpload_implActor*>(this)));
															#line 12435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AbortMultiPartUpload_implActorState();
		static_cast<AbortMultiPartUpload_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string resource = bstore->constructResourcePath(bucket, object);
															#line 2197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		HTTP::Headers headers;
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("DELETE", resource, headers, nullptr, 0, { 200, 204 });
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12466 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<AbortMultiPartUpload_implActor*>(this)));
															#line 12471 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::string resource = bstore->constructResourcePath(bucket, object);
															#line 2197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += format("?uploadId=%s", uploadID.c_str());
															#line 2199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		HTTP::Headers headers;
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("DELETE", resource, headers, nullptr, 0, { 200, 204 });
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 2;
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<AbortMultiPartUpload_implActor*>(this)));
															#line 12493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<AbortMultiPartUpload_implActor*>(this)->ActorCallback< AbortMultiPartUpload_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<AbortMultiPartUpload_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AbortMultiPartUpload_implActorState(); static_cast<AbortMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<AbortMultiPartUpload_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AbortMultiPartUpload_implActorState();
		static_cast<AbortMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<AbortMultiPartUpload_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AbortMultiPartUpload_implActorState(); static_cast<AbortMultiPartUpload_implActor*>(this)->destroy(); return 0; }
															#line 12589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<AbortMultiPartUpload_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AbortMultiPartUpload_implActorState();
		static_cast<AbortMultiPartUpload_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state > 0) static_cast<AbortMultiPartUpload_implActor*>(this)->actor_wait_state = 0;
		static_cast<AbortMultiPartUpload_implActor*>(this)->ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string uploadID;
															#line 12680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via abortMultiPartUpload_impl()
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class AbortMultiPartUpload_implActor final : public Actor<Void>, public ActorCallback< AbortMultiPartUpload_implActor, 0, Void >, public ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<AbortMultiPartUpload_implActor>, public AbortMultiPartUpload_implActorState<AbortMultiPartUpload_implActor> {
															#line 12685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<AbortMultiPartUpload_implActor>::operator new;
	using FastAllocated<AbortMultiPartUpload_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15298077880272470528UL, 8576369422743963392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AbortMultiPartUpload_implActor, 0, Void >;
friend struct ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	AbortMultiPartUpload_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::string const& uploadID) 
															#line 12703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   AbortMultiPartUpload_implActorState<AbortMultiPartUpload_implActor>(bstore, bucket, object, uploadID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16979502986786407680UL, 15532907730366778880UL);
		ActorExecutionContextHelper __helper(static_cast<AbortMultiPartUpload_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("abortMultiPartUpload_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("abortMultiPartUpload_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AbortMultiPartUpload_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AbortMultiPartUpload_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> abortMultiPartUpload_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::string const& uploadID ) {
															#line 2190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new AbortMultiPartUpload_implActor(bstore, bucket, object, uploadID));
															#line 12737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::abortMultiPartUpload(std::string const& bucket,
                                                       std::string const& object,
                                                       std::string const& uploadID) {
	return abortMultiPartUpload_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, uploadID);
}

// Forward declarations
															#line 12749 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
[[nodiscard]] Future<std::map<std::string, std::string>> getObjectTags_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object );

#line 2215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 12754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via putObjectTags_impl()
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class PutObjectTags_implActor>
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class PutObjectTags_implActorState {
															#line 12761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PutObjectTags_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& tags) 
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object),
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   tags(tags),
															#line 2220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   packets()
															#line 12776 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("putObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	~PutObjectTags_implActorState() 
	{
		fdb_probe_actor_destroy("putObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateWrite->getAllowance(1);
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 1;
															#line 2221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 0, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 12798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PutObjectTags_implActorState();
		static_cast<PutObjectTags_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxRetries = 5;
															#line 2225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		retryCount = 0;
															#line 2226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = 1.0;
															#line 2227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxBackoff = 8.0;
															#line 2229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 12833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxRetries = 5;
															#line 2225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		retryCount = 0;
															#line 2226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = 1.0;
															#line 2227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		maxBackoff = 8.0;
															#line 2229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		;
															#line 12854 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string manifest = "<Tagging xmlns=\"http://s3.amazonaws.com/doc/2006-03-01/\"><TagSet>";
															#line 2232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(auto itr = tags.begin();itr != tags.end();++itr) {
															#line 2233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				manifest += "<Tag><Key>" + itr->first + "</Key><Value>" + itr->second + "</Value></Tag>";
															#line 12950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			manifest += "</TagSet></Tagging>";
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			PacketWriter pw(packets.getWriteBuffer(manifest.size()), nullptr, Unversioned());
															#line 2238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			pw.serializeBytes(manifest);
															#line 2240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			HTTP::Headers headers;
															#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-Type"] = "application/xml";
															#line 2242 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			headers["Content-Length"] = format("%d", manifest.size());
															#line 2244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("PUT", resource, headers, &packets, manifest.size(), { 200 });
															#line 2244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12968 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 2;
															#line 2244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 12973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 12995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 12999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStorePutTagsError") .error(e) .detail("Bucket", bucket) .detail("Object", object) .detail("RetryCount", retryCount);
															#line 2275 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (++retryCount >= maxRetries)
															#line 13005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 13009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(backoff * (0.9 + 0.2 * deterministicRandom()->random01()));
															#line 2280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 5;
															#line 2280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 4, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<std::map<std::string, std::string>> __when_expr_2 = getObjectTags_impl(bstore, bucket, object);
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 3;
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<std::map<std::string, std::string>> __when_expr_2 = getObjectTags_impl(bstore, bucket, object);
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 3;
															#line 2248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont4(std::map<std::string, std::string> const& verifyTags,int loopDepth) 
	{
															#line 2249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (verifyTags == tags)
															#line 13142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<PutObjectTags_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PutObjectTags_implActorState(); static_cast<PutObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<PutObjectTags_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PutObjectTags_implActorState();
			static_cast<PutObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (++retryCount >= maxRetries)
															#line 13154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStorePutTagsMaxRetriesExceeded") .detail("Bucket", bucket) .detail("Object", object);
															#line 2257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1cont1loopBody1Catch1(operation_failed(), loopDepth);
															#line 13160 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(backoff * (0.9 + 0.2 * deterministicRandom()->random01()));
															#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 4;
															#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 3, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13171 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(std::map<std::string, std::string> && verifyTags,int loopDepth) 
	{
															#line 2249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (verifyTags == tags)
															#line 13180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<PutObjectTags_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PutObjectTags_implActorState(); static_cast<PutObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13184 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<PutObjectTags_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~PutObjectTags_implActorState();
			static_cast<PutObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2253 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (++retryCount >= maxRetries)
															#line 13192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			TraceEvent(SevWarn, "S3BlobStorePutTagsMaxRetriesExceeded") .detail("Bucket", bucket) .detail("Object", object);
															#line 2257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			return a_body1cont1loopBody1Catch1(operation_failed(), loopDepth);
															#line 13198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(backoff * (0.9 + 0.2 * deterministicRandom()->random01()));
															#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 4;
															#line 2261 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PutObjectTags_implActor, 3, Void >*>(static_cast<PutObjectTags_implActor*>(this)));
															#line 13209 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(std::map<std::string, std::string> const& verifyTags,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(verifyTags, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(std::map<std::string, std::string> && verifyTags,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(verifyTags), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*,std::map<std::string, std::string> const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*,std::map<std::string, std::string> && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13293 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2262 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 2281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 2281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		backoff = std::min(backoff * 2, maxBackoff);
															#line 13408 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<PutObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<PutObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<PutObjectTags_implActor*>(this)->ActorCallback< PutObjectTags_implActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< PutObjectTags_implActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< PutObjectTags_implActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::map<std::string, std::string> tags;
															#line 2220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnsentPacketQueue packets;
															#line 2222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int maxRetries;
															#line 2225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	int retryCount;
															#line 2226 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double backoff;
															#line 2227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	double maxBackoff;
															#line 13508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via putObjectTags_impl()
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class PutObjectTags_implActor final : public Actor<Void>, public ActorCallback< PutObjectTags_implActor, 0, Void >, public ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >, public ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >, public ActorCallback< PutObjectTags_implActor, 3, Void >, public ActorCallback< PutObjectTags_implActor, 4, Void >, public FastAllocated<PutObjectTags_implActor>, public PutObjectTags_implActorState<PutObjectTags_implActor> {
															#line 13513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<PutObjectTags_implActor>::operator new;
	using FastAllocated<PutObjectTags_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12442033635318598656UL, 11326198365512458496UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PutObjectTags_implActor, 0, Void >;
friend struct ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >;
friend struct ActorCallback< PutObjectTags_implActor, 3, Void >;
friend struct ActorCallback< PutObjectTags_implActor, 4, Void >;
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	PutObjectTags_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& tags) 
															#line 13534 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   PutObjectTags_implActorState<PutObjectTags_implActor>(bstore, bucket, object, tags),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("putObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12648391953656857344UL, 8578338389149528320UL);
		ActorExecutionContextHelper __helper(static_cast<PutObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("putObjectTags_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("putObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 2, std::map<std::string, std::string> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< PutObjectTags_implActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<Void> putObjectTags_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& tags ) {
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new PutObjectTags_implActor(bstore, bucket, object, tags));
															#line 13571 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2285 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<Void> S3BlobStoreEndpoint::putObjectTags(std::string const& bucket,
                                                std::string const& object,
                                                std::map<std::string, std::string> const& tags) {
	return putObjectTags_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object, tags);
}

															#line 13582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via getObjectTags_impl()
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class GetObjectTags_implActor>
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class GetObjectTags_implActorState {
															#line 13589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	GetObjectTags_implActorState(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : bstore(bstore),
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   bucket(bucket),
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		   object(object)
															#line 13600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("getObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	~GetObjectTags_implActorState() 
	{
		fdb_probe_actor_destroy("getObjectTags_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bstore->requestRateRead->getAllowance(1);
															#line 2295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 1;
															#line 2295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetObjectTags_implActor, 0, Void >*>(static_cast<GetObjectTags_implActor*>(this)));
															#line 13622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetObjectTags_implActorState();
		static_cast<GetObjectTags_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", resource, headers, nullptr, 0, { 200 });
															#line 2301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 2;
															#line 2301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<GetObjectTags_implActor*>(this)));
															#line 13658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource = bstore->constructResourcePath(bucket, object);
															#line 2298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		resource += "?tagging";
															#line 2299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		headers = HTTP::Headers();
															#line 2301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_1 = bstore->doRequest("GET", resource, headers, nullptr, 0, { 200 });
															#line 2301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 2;
															#line 2301 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*>(static_cast<GetObjectTags_implActor*>(this)));
															#line 13680 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetObjectTags_implActor*>(this)->ActorCallback< GetObjectTags_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetObjectTags_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
															#line 2303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_document<> doc;
															#line 2304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		doc.parse<rapidxml::parse_default>((char*)r->data.content.c_str());
															#line 2306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::map<std::string, std::string> tags;
															#line 2309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_node<>* tagging = doc.first_node();
															#line 2310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for(;tagging && strcmp(tagging->name(), "Tagging") != 0;) {
															#line 2311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			tagging = tagging->next_sibling();
															#line 13774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (tagging)
															#line 13778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			rapidxml::xml_node<>* tagSet = tagging->first_node();
															#line 2317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;tagSet && strcmp(tagSet->name(), "TagSet") != 0;) {
															#line 2318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				tagSet = tagSet->next_sibling();
															#line 13786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (tagSet)
															#line 13790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(rapidxml::xml_node<>* tag = tagSet->first_node();tag;tag = tag->next_sibling()) {
															#line 2324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(tag->name(), "Tag") == 0)
															#line 13796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 2325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string key, value;
															#line 2327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						for(rapidxml::xml_node<>* node = tag->first_node();node;node = node->next_sibling()) {
															#line 2328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (strcmp(node->name(), "Key") == 0)
															#line 13804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 2329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								key = node->value();
															#line 13808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
							}
							else
							{
															#line 2330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								if (strcmp(node->name(), "Value") == 0)
															#line 13814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								{
															#line 2331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									value = node->value();
															#line 13818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								}
							}
						}
															#line 2334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (!key.empty())
															#line 13824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 2335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							tags[key] = value;
															#line 13828 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 2342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<GetObjectTags_implActor*>(this)->SAV<std::map<std::string, std::string>>::futures) { (void)(tags); this->~GetObjectTags_implActorState(); static_cast<GetObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<GetObjectTags_implActor*>(this)->SAV< std::map<std::string, std::string> >::value()) std::map<std::string, std::string>(tags);
		this->~GetObjectTags_implActorState();
		static_cast<GetObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
															#line 2303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_document<> doc;
															#line 2304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		doc.parse<rapidxml::parse_default>((char*)r->data.content.c_str());
															#line 2306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		std::map<std::string, std::string> tags;
															#line 2309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		rapidxml::xml_node<>* tagging = doc.first_node();
															#line 2310 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		for(;tagging && strcmp(tagging->name(), "Tagging") != 0;) {
															#line 2311 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			tagging = tagging->next_sibling();
															#line 13858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		}
															#line 2314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (tagging)
															#line 13862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		{
															#line 2316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			rapidxml::xml_node<>* tagSet = tagging->first_node();
															#line 2317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			for(;tagSet && strcmp(tagSet->name(), "TagSet") != 0;) {
															#line 2318 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				tagSet = tagSet->next_sibling();
															#line 13870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (tagSet)
															#line 13874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				for(rapidxml::xml_node<>* tag = tagSet->first_node();tag;tag = tag->next_sibling()) {
															#line 2324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
					if (strcmp(tag->name(), "Tag") == 0)
															#line 13880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
					{
															#line 2325 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						std::string key, value;
															#line 2327 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						for(rapidxml::xml_node<>* node = tag->first_node();node;node = node->next_sibling()) {
															#line 2328 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							if (strcmp(node->name(), "Key") == 0)
															#line 13888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
							{
															#line 2329 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								key = node->value();
															#line 13892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
							}
							else
							{
															#line 2330 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
								if (strcmp(node->name(), "Value") == 0)
															#line 13898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								{
															#line 2331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
									value = node->value();
															#line 13902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
								}
							}
						}
															#line 2334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
						if (!key.empty())
															#line 13908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						{
															#line 2335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
							tags[key] = value;
															#line 13912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
						}
					}
				}
			}
		}
															#line 2342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<GetObjectTags_implActor*>(this)->SAV<std::map<std::string, std::string>>::futures) { (void)(tags); this->~GetObjectTags_implActorState(); static_cast<GetObjectTags_implActor*>(this)->destroy(); return 0; }
															#line 13920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<GetObjectTags_implActor*>(this)->SAV< std::map<std::string, std::string> >::value()) std::map<std::string, std::string>(tags);
		this->~GetObjectTags_implActorState();
		static_cast<GetObjectTags_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> const& r,int loopDepth) 
	{
		loopDepth = a_body1cont2(r, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<HTTP::IncomingResponse> && r,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetObjectTags_implActor*>(this)->actor_wait_state > 0) static_cast<GetObjectTags_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetObjectTags_implActor*>(this)->ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	Reference<S3BlobStoreEndpoint> bstore;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string bucket;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string object;
															#line 2297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	std::string resource;
															#line 2299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	HTTP::Headers headers;
															#line 14013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via getObjectTags_impl()
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class GetObjectTags_implActor final : public Actor<std::map<std::string, std::string>>, public ActorCallback< GetObjectTags_implActor, 0, Void >, public ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >, public FastAllocated<GetObjectTags_implActor>, public GetObjectTags_implActorState<GetObjectTags_implActor> {
															#line 14018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<GetObjectTags_implActor>::operator new;
	using FastAllocated<GetObjectTags_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15126349093996463872UL, 12034220258860591872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::map<std::string, std::string>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetObjectTags_implActor, 0, Void >;
friend struct ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >;
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	GetObjectTags_implActor(Reference<S3BlobStoreEndpoint> const& bstore,std::string const& bucket,std::string const& object) 
															#line 14036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<std::map<std::string, std::string>>(),
		   GetObjectTags_implActorState<GetObjectTags_implActor>(bstore, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9536500618548032768UL, 11160301285233402112UL);
		ActorExecutionContextHelper __helper(static_cast<GetObjectTags_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getObjectTags_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getObjectTags_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetObjectTags_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetObjectTags_implActor, 1, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
[[nodiscard]] Future<std::map<std::string, std::string>> getObjectTags_impl( Reference<S3BlobStoreEndpoint> const& bstore, std::string const& bucket, std::string const& object ) {
															#line 2292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<std::map<std::string, std::string>>(new GetObjectTags_implActor(bstore, bucket, object));
															#line 14070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}

#line 2344 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

Future<std::map<std::string, std::string>> S3BlobStoreEndpoint::getObjectTags(std::string const& bucket,
                                                                              std::string const& object) {
	return getObjectTags_impl(Reference<S3BlobStoreEndpoint>::addRef(this), bucket, object);
}

															#line 14080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2350()
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FlowTestCase2350Actor>
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2350ActorState {
															#line 14087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2350ActorState(UnitTestParameters const& params) 
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : params(params)
															#line 14094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2350", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2350ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2350", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::Credentials creds{ "AKIAIOSFODNN7EXAMPLE", "wJalrXUtnFEMI/K7MDENG/bPxRfiCYEXAMPLEKEY", "" };
															#line 14109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			{
															#line 2354 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint s3("s3.amazonaws.com", "443", "amazonaws", "proxy", "port", creds);
															#line 2355 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string verb("GET");
															#line 2356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string resource("/test.txt");
															#line 2357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				HTTP::Headers headers;
															#line 2358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Host"] = "s3.amazonaws.com";
															#line 2359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3.setV4AuthHeaders(verb, resource, headers, "20130524T000000Z", "20130524");
															#line 2360 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Authorization"] == "AWS4-HMAC-SHA256 Credential=AKIAIOSFODNN7EXAMPLE/20130524/amazonaws/s3/aws4_request, " "SignedHeaders=host;x-amz-content-sha256;x-amz-date, " "Signature=c6037f4b174f2019d02d7085a611cef8adfe1efe583e220954dc85d59cd31ba3");
															#line 2364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["x-amz-date"] == "20130524T000000Z");
															#line 14127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			{
															#line 2369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint s3("s3.amazonaws.com", "443", "amazonaws", "proxy", "port", creds);
															#line 2370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string verb("GET");
															#line 2371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string resource("/test/examplebucket?Action=DescribeRegions&Version=2013-10-15");
															#line 2372 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				HTTP::Headers headers;
															#line 2373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Host"] = "s3.amazonaws.com";
															#line 2374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3.setV4AuthHeaders(verb, resource, headers, "20130524T000000Z", "20130524");
															#line 2375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Authorization"] == "AWS4-HMAC-SHA256 Credential=AKIAIOSFODNN7EXAMPLE/20130524/amazonaws/s3/aws4_request, " "SignedHeaders=host;x-amz-content-sha256;x-amz-date, " "Signature=426f04e71e191fbc30096c306fe1b11ce8f026a7be374541862bbee320cce71c");
															#line 2379 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["x-amz-date"] == "20130524T000000Z");
															#line 14146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
			{
															#line 2384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				S3BlobStoreEndpoint s3("s3.us-west-2.amazonaws.com", "443", "us-west-2", "proxy", "port", creds);
															#line 2385 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string verb("POST");
															#line 2386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				std::string resource("/simple.json");
															#line 2387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				HTTP::Headers headers;
															#line 2388 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Host"] = "s3.us-west-2.amazonaws.com";
															#line 2389 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				headers["Content-Type"] = "Application/x-amz-json-1.0";
															#line 2390 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3.setV4AuthHeaders(verb, resource, headers, "20130524T000000Z", "20130524");
															#line 2391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Authorization"] == "AWS4-HMAC-SHA256 Credential=AKIAIOSFODNN7EXAMPLE/20130524/us-west-2/s3/aws4_request, " "SignedHeaders=content-type;host;x-amz-content-sha256;x-amz-date, " "Signature=cf095e36bed9cd3139c2e8b3e20c296a79d8540987711bf3a0d816b19ae00314");
															#line 2395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["x-amz-date"] == "20130524T000000Z");
															#line 2396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Host"] == "s3.us-west-2.amazonaws.com");
															#line 2397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(headers["Content-Type"] == "Application/x-amz-json-1.0");
															#line 14171 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			}
															#line 2400 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			if (!static_cast<FlowTestCase2350Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2350ActorState(); static_cast<FlowTestCase2350Actor*>(this)->destroy(); return 0; }
															#line 14175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			new (&static_cast<FlowTestCase2350Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2350ActorState();
			static_cast<FlowTestCase2350Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2350ActorState();
		static_cast<FlowTestCase2350Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnitTestParameters params;
															#line 14199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2350()
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2350Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2350Actor>, public FlowTestCase2350ActorState<FlowTestCase2350Actor> {
															#line 14204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2350Actor>::operator new;
	using FastAllocated<FlowTestCase2350Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5466684368160510464UL, 16936660327898760960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2350Actor(UnitTestParameters const& params) 
															#line 14220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2350ActorState<FlowTestCase2350Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2350", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1786993027159534336UL, 6792992865354693120UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2350Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2350");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2350", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
static Future<Void> flowTestCase2350( UnitTestParameters const& params ) {
															#line 2350 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new FlowTestCase2350Actor(params));
															#line 14252 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2350, "/backup/s3/v4headers")

#line 2402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"

															#line 14258 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2403()
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
template <class FlowTestCase2403Actor>
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2403ActorState {
															#line 14265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2403ActorState(UnitTestParameters const& params) 
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		 : params(params)
															#line 14272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2403", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2403ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2403", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2404 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string url = "blobstore://s3.us-west-2.amazonaws.com/resource_name?bucket=bucket_name&sa=1";
															#line 2406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string resource;
															#line 2407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			std::string error;
															#line 2408 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			S3BlobStoreEndpoint::ParametersT parameters;
															#line 2409 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			Reference<S3BlobStoreEndpoint> s3 = S3BlobStoreEndpoint::fromString(url, {}, &resource, &error, &parameters);
															#line 2410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT(s3->getRegion() == "us-west-2");
															#line 2412 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			url = "blobstore://s3.us-west-2.amazonaws.com/resource_name?bucket=bucket_name&sc=922337203685477580700";
															#line 14299 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			try {
															#line 2414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				s3 = S3BlobStoreEndpoint::fromString(url, {}, &resource, &error, &parameters);
															#line 2415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
				ASSERT(false);
															#line 14305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2403ActorState();
		static_cast<FlowTestCase2403Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2420 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
		if (!static_cast<FlowTestCase2403Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2403ActorState(); static_cast<FlowTestCase2403Actor*>(this)->destroy(); return 0; }
															#line 14334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		new (&static_cast<FlowTestCase2403Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2403ActorState();
		static_cast<FlowTestCase2403Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2418 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
			ASSERT_EQ(e.code(), error_code_backup_invalid_url);
															#line 14347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	UnitTestParameters params;
															#line 14373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2403()
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
class FlowTestCase2403Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2403Actor>, public FlowTestCase2403ActorState<FlowTestCase2403Actor> {
															#line 14378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2403Actor>::operator new;
	using FastAllocated<FlowTestCase2403Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12948024778606109440UL, 4371791208597957888UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	FlowTestCase2403Actor(UnitTestParameters const& params) 
															#line 14394 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2403ActorState<FlowTestCase2403Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2403", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7446490383563358976UL, 7635971082708255744UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2403Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2403");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2403", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
static Future<Void> flowTestCase2403( UnitTestParameters const& params ) {
															#line 2403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
	return Future<Void>(new FlowTestCase2403Actor(params));
															#line 14426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/S3BlobStore.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2403, "/backup/s3/guess_region")

#line 2422 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/S3BlobStore.actor.cpp"
