#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
/*
 * NativeAPI.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/NativeAPI.actor.h"

#include <algorithm>
#include <cstdio>
#include <iterator>
#include <limits>
#include <memory>
#include <regex>
#include <string>
#include <unordered_set>
#include <tuple>
#include <utility>
#include <vector>

#include "boost/algorithm/string.hpp"

#include "fdbclient/Knobs.h"
#include "flow/CodeProbe.h"
#include "fmt/format.h"

#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbrpc/FailureMonitor.h"
#include "fdbrpc/MultiInterface.h"
#include "fdbrpc/TenantInfo.h"

#include "fdbclient/ActorLineageProfiler.h"
#include "fdbclient/AnnotateActor.h"
#include "fdbclient/Atomic.h"
#include "fdbclient/BlobGranuleCommon.h"
#include "fdbclient/BlobGranuleRequest.actor.h"
#include "fdbclient/ClusterInterface.h"
#include "fdbclient/ClusterConnectionFile.h"
#include "fdbclient/ClusterConnectionMemoryRecord.h"
#include "fdbclient/CoordinationInterface.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/GlobalConfig.actor.h"
#include "fdbclient/IKnobCollection.h"
#include "fdbclient/JsonBuilder.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/NameLineage.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/MonitorLeader.h"
#include "fdbclient/MutationList.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbclient/ParallelStream.actor.h"
#include "fdbclient/SpecialKeySpace.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/Tenant.h"
#include "fdbclient/TenantSpecialKeys.actor.h"
#include "fdbclient/TransactionLineage.h"
#include "fdbclient/versions.h"
#include "fdbrpc/WellKnownEndpoints.h"
#include "fdbrpc/LoadBalance.h"
#include "fdbrpc/Net2FileSystem.h"
#include "fdbrpc/simulator.h"
#include "fdbrpc/sim_validation.h"
#include "flow/Arena.h"
#include "flow/ActorCollection.h"
#include "flow/DeterministicRandom.h"
#include "flow/Error.h"
#include "flow/FastRef.h"
#include "flow/IRandom.h"
#include "flow/Trace.h"
#include "flow/ProtocolVersion.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/Knobs.h"
#include "flow/Platform.h"
#include "flow/SystemMonitor.h"
#include "flow/TLSConfig.actor.h"
#include "fdbclient/Tracing.h"
#include "flow/UnitTest.h"
#include "flow/network.h"
#include "flow/serialize.h"

#ifdef ADDRESS_SANITIZER
#include <sanitizer/lsan_interface.h>
#endif

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#undef min
#undef max
#else
#include <time.h>
#endif
#include "flow/actorcompiler.h" // This must be the last #include.

template class RequestStream<OpenDatabaseRequest, false>;
template struct NetNotifiedQueue<OpenDatabaseRequest, false>;

extern const char* getSourceVersion();

namespace {

TransactionLineageCollector transactionLineageCollector;
NameLineageCollector nameLineageCollector;

template <class Interface, class Request, bool P>
Future<REPLY_TYPE(Request)> loadBalance(
    DatabaseContext* ctx,
    const Reference<LocationInfo> alternatives,
    RequestStream<Request, P> Interface::* channel,
    const Request& request = Request(),
    TaskPriority taskID = TaskPriority::DefaultPromiseEndpoint,
    AtMostOnce atMostOnce =
        AtMostOnce::False, // if true, throws request_maybe_delivered() instead of retrying automatically
    QueueModel* model = nullptr,
    bool compareReplicas = false,
    int requiredReplicas = 0) {
	if (alternatives->hasCaches) {
		return loadBalance(
		    alternatives->locations(), channel, request, taskID, atMostOnce, model, compareReplicas, requiredReplicas);
	}
	return fmap(
	    [ctx](auto const& res) {
		    if (res.cached) {
			    ctx->updateCache.trigger();
		    }
		    return res;
	    },
	    loadBalance(
	        alternatives->locations(), channel, request, taskID, atMostOnce, model, compareReplicas, requiredReplicas));
}
} // namespace

FDB_BOOLEAN_PARAM(TransactionRecordLogInfo);

// Whether or not a request should include the tenant name
FDB_BOOLEAN_PARAM(UseTenant);

// Whether a blob granule request is a request for the mapping to read, or a request to get granule boundaries
FDB_BOOLEAN_PARAM(JustGranules);

NetworkOptions networkOptions;
TLSConfig tlsConfig(TLSEndpointType::CLIENT);

// The default values, TRACE_DEFAULT_ROLL_SIZE and TRACE_DEFAULT_MAX_LOGS_SIZE are located in Trace.h.
NetworkOptions::NetworkOptions()
  : traceRollSize(TRACE_DEFAULT_ROLL_SIZE), traceMaxLogsSize(TRACE_DEFAULT_MAX_LOGS_SIZE), traceLogGroup("default"),
    traceFormat("xml"), traceClockSource("now"), traceInitializeOnSetup(false),
    supportedVersions(new ReferencedObject<Standalone<VectorRef<ClientVersionRef>>>()), runLoopProfilingEnabled(false),
    primaryClient(true) {}

static const Key CLIENT_LATENCY_INFO_PREFIX = "client_latency/"_sr;
static const Key CLIENT_LATENCY_INFO_CTR_PREFIX = "client_latency_counter/"_sr;

void DatabaseContext::addTssMapping(StorageServerInterface const& ssi, StorageServerInterface const& tssi) {
	auto result = tssMapping.find(ssi.id());
	// Update tss endpoint mapping if ss isn't in mapping, or the interface it mapped to changed
	if (result == tssMapping.end() ||
	    result->second.getValue.getEndpoint().token.first() != tssi.getValue.getEndpoint().token.first()) {
		Reference<TSSMetrics> metrics;
		if (result == tssMapping.end()) {
			// new TSS pairing
			metrics = makeReference<TSSMetrics>();
			tssMetrics[tssi.id()] = metrics;
			tssMapping[ssi.id()] = tssi;
		} else {
			ASSERT(result->second.id() == tssi.id());
			metrics = tssMetrics[tssi.id()];
			result->second = tssi;
		}

		// data requests duplicated for load and data comparison
		queueModel.updateTssEndpoint(ssi.getValue.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getValue.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getKey.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getKey.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getKeyValues.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getKeyValues.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getMappedKeyValues.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getMappedKeyValues.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getKeyValuesStream.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getKeyValuesStream.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.changeFeedStream.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.changeFeedStream.getEndpoint(), metrics));

		// non-data requests duplicated for load
		queueModel.updateTssEndpoint(ssi.watchValue.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.watchValue.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.splitMetrics.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.splitMetrics.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getReadHotRanges.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getReadHotRanges.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getRangeSplitPoints.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getRangeSplitPoints.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.overlappingChangeFeeds.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.overlappingChangeFeeds.getEndpoint(), metrics));

		// duplicated to ensure feed data cleanup
		queueModel.updateTssEndpoint(ssi.changeFeedPop.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.changeFeedPop.getEndpoint(), metrics));
	}
}

void DatabaseContext::removeTssMapping(StorageServerInterface const& ssi) {
	auto result = tssMapping.find(ssi.id());
	if (result != tssMapping.end()) {
		tssMetrics.erase(ssi.id());
		tssMapping.erase(result);
		queueModel.removeTssEndpoint(ssi.getValue.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getKey.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getKeyValues.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getMappedKeyValues.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getKeyValuesStream.getEndpoint().token.first());

		queueModel.removeTssEndpoint(ssi.watchValue.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.splitMetrics.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getReadHotRanges.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getRangeSplitPoints.getEndpoint().token.first());
	}
}

void DatabaseContext::addSSIdTagMapping(const UID& uid, const Tag& tag) {
	ssidTagMapping[uid] = tag;
}

void DatabaseContext::getLatestCommitVersionForSSID(const UID& ssid, Tag& tag, Version& commitVersion) {
	// initialization
	tag = invalidTag;
	commitVersion = invalidVersion;

	auto iter = ssidTagMapping.find(ssid);
	if (iter != ssidTagMapping.end()) {
		tag = iter->second;

		if (ssVersionVectorCache.hasVersion(tag)) {
			commitVersion = ssVersionVectorCache.getVersion(tag);
		}
	}
}

void DatabaseContext::getLatestCommitVersion(const StorageServerInterface& ssi,
                                             Version readVersion,
                                             VersionVector& latestCommitVersion) {
	latestCommitVersion.clear();

	if (ssVersionVectorCache.getMaxVersion() == invalidVersion) {
		return;
	}

	// Error checking (based on the assumption that the read version was not obtained
	// from the client's grv cache).
	if (readVersion > ssVersionVectorCache.getMaxVersion()) {
		TraceEvent(SevError, "ReadVersionExceedsVersionVectorMax")
		    .detail("ReadVersion", readVersion)
		    .detail("VersionVector", ssVersionVectorCache.toString());
		if (g_network->isSimulated()) {
			ASSERT(false);
		} else {
			return; // Do not return a stale commit version in production.
		}
	}

	Tag tag = invalidTag;
	Version commitVersion = invalidVersion;
	getLatestCommitVersionForSSID(ssi.id(), tag, commitVersion);

	if (tag != invalidTag && commitVersion != invalidVersion && commitVersion < readVersion) {
		latestCommitVersion.setVersion(tag, commitVersion);
	}
}

void DatabaseContext::getLatestCommitVersions(const Reference<LocationInfo>& locationInfo,
                                              Reference<TransactionState> info,
                                              VersionVector& latestCommitVersions) {
	latestCommitVersions.clear();

	if (info->readOptions.present() && info->readOptions.get().debugID.present()) {
		g_traceBatch.addEvent(
		    "TransactionDebug", info->readOptions.get().debugID.get().first(), "NativeAPI.getLatestCommitVersions");
	}

	if (!info->readVersionObtainedFromGrvProxy) {
		return;
	}

	if (ssVersionVectorCache.getMaxVersion() == invalidVersion) {
		return;
	}

	if (info->readVersion() > ssVersionVectorCache.getMaxVersion()) {
		if (!CLIENT_KNOBS->FORCE_GRV_CACHE_OFF && !info->options.skipGrvCache && info->options.useGrvCache) {
			return;
		} else {
			TraceEvent(SevError, "GetLatestCommitVersions")
			    .detail("ReadVersion", info->readVersion())
			    .detail("VersionVector", ssVersionVectorCache.toString());
			ASSERT(false);
		}
	}

	std::map<Version, std::set<Tag>> versionMap; // order the versions to be returned
	for (int i = 0; i < locationInfo->locations()->size(); i++) {
		Tag tag = invalidTag;
		Version commitVersion = invalidVersion; // latest commit version
		getLatestCommitVersionForSSID(locationInfo->locations()->getId(i), tag, commitVersion);

		bool updatedVersionMap = false;
		if (tag != invalidTag && commitVersion != invalidVersion && commitVersion < info->readVersion()) {
			updatedVersionMap = true;
			versionMap[commitVersion].insert(tag);
		}

		// Do not log if commitVersion >= readVersion.
		if (!updatedVersionMap && commitVersion == invalidVersion) {
			TraceEvent(SevDebug, "CommitVersionNotFoundForSS")
			    .detail("InSSIDMap", tag != invalidTag ? 1 : 0)
			    .detail("Tag", tag)
			    .detail("CommitVersion", commitVersion)
			    .detail("ReadVersion", info->readVersion())
			    .detail("VersionVector", ssVersionVectorCache.toString())
			    .setMaxEventLength(11000)
			    .setMaxFieldLength(10000);
			++transactionCommitVersionNotFoundForSS;
		}
	}

	// insert the commit versions in the version vector.
	for (auto& iter : versionMap) {
		latestCommitVersions.setVersion(iter.second, iter.first);
	}
}

void updateCachedReadVersionShared(double t, Version v, DatabaseSharedState* p) {
	MutexHolder mutex(p->mutexLock);
	if (v >= p->grvCacheSpace.cachedReadVersion) {
		//TraceEvent(SevDebug, "CacheReadVersionUpdate")
		//    .detail("Version", v)
		//    .detail("CurTime", t)
		//    .detail("LastVersion", p->grvCacheSpace.cachedReadVersion)
		//    .detail("LastTime", p->grvCacheSpace.lastGrvTime);
		p->grvCacheSpace.cachedReadVersion = v;
		if (t > p->grvCacheSpace.lastGrvTime) {
			p->grvCacheSpace.lastGrvTime = t;
		}
	}
}

void DatabaseContext::updateCachedReadVersion(double t, Version v) {
	if (sharedStatePtr) {
		return updateCachedReadVersionShared(t, v, sharedStatePtr);
	}
	if (v >= cachedReadVersion) {
		//TraceEvent(SevDebug, "CachedReadVersionUpdate")
		//    .detail("Version", v)
		//    .detail("GrvStartTime", t)
		//    .detail("LastVersion", cachedReadVersion)
		//    .detail("LastTime", lastGrvTime);
		cachedReadVersion = v;
		// Since the time is based on the start of the request, it's possible that we
		// get a newer version with an older time.
		// (Request started earlier, but was latest to reach the proxy)
		// Only update time when strictly increasing (?)
		if (t > lastGrvTime) {
			lastGrvTime = t;
		}
	}
}

Version DatabaseContext::getCachedReadVersion() {
	if (sharedStatePtr) {
		MutexHolder mutex(sharedStatePtr->mutexLock);
		return sharedStatePtr->grvCacheSpace.cachedReadVersion;
	}
	return cachedReadVersion;
}

double DatabaseContext::getLastGrvTime() {
	if (sharedStatePtr) {
		MutexHolder mutex(sharedStatePtr->mutexLock);
		return sharedStatePtr->grvCacheSpace.lastGrvTime;
	}
	return lastGrvTime;
}

Reference<StorageServerInfo> StorageServerInfo::getInterface(DatabaseContext* cx,
                                                             StorageServerInterface const& ssi,
                                                             LocalityData const& locality) {
	auto it = cx->server_interf.find(ssi.id());
	if (it != cx->server_interf.end()) {
		if (it->second->interf.getValue.getEndpoint().token != ssi.getValue.getEndpoint().token) {
			if (it->second->interf.locality == ssi.locality) {
				// FIXME: load balance holds pointers to individual members of the interface, and this assignment will
				// swap out the object they are
				//       pointing to. This is technically correct, but is very unnatural. We may want to refactor load
				//       balance to take an AsyncVar<Reference<Interface>> so that it is notified when the interface
				//       changes.

				it->second->interf = ssi;
			} else {
				it->second->notifyContextDestroyed();
				Reference<StorageServerInfo> loc(new StorageServerInfo(cx, ssi, locality));
				cx->server_interf[ssi.id()] = loc.getPtr();
				return loc;
			}
		}

		return Reference<StorageServerInfo>::addRef(it->second);
	}

	Reference<StorageServerInfo> loc(new StorageServerInfo(cx, ssi, locality));
	cx->server_interf[ssi.id()] = loc.getPtr();
	return loc;
}

void StorageServerInfo::notifyContextDestroyed() {
	cx = nullptr;
}

StorageServerInfo::~StorageServerInfo() {
	if (cx) {
		auto it = cx->server_interf.find(interf.id());
		if (it != cx->server_interf.end())
			cx->server_interf.erase(it);
		cx = nullptr;
	}
}

std::string printable(const VectorRef<KeyValueRef>& val) {
	std::string s;
	for (int i = 0; i < val.size(); i++)
		s = s + printable(val[i].key) + format(":%d ", val[i].value.size());
	return s;
}

std::string printable(const KeyValueRef& val) {
	return printable(val.key) + format(":%d ", val.value.size());
}

std::string printable(const VectorRef<StringRef>& val) {
	std::string s;
	for (int i = 0; i < val.size(); i++)
		s = s + printable(val[i]) + " ";
	return s;
}

std::string printable(const StringRef& val) {
	return val.printable();
}
std::string printable(const std::string& str) {
	return StringRef(str).printable();
}

std::string printable(const KeyRangeRef& range) {
	return printable(range.begin) + " - " + printable(range.end);
}

std::string printable(const VectorRef<KeyRangeRef>& val) {
	std::string s;
	for (int i = 0; i < val.size(); i++)
		s = s + printable(val[i]) + " ";
	return s;
}

int unhex(char c) {
	if (c >= '0' && c <= '9')
		return c - '0';
	if (c >= 'a' && c <= 'f')
		return c - 'a' + 10;
	if (c >= 'A' && c <= 'F')
		return c - 'A' + 10;
	UNREACHABLE();
}

std::string unprintable(std::string const& val) {
	std::string s;
	for (int i = 0; i < val.size(); i++) {
		char c = val[i];
		if (c == '\\') {
			if (++i == val.size())
				ASSERT(false);
			if (val[i] == '\\') {
				s += '\\';
			} else if (val[i] == 'x') {
				if (i + 2 >= val.size())
					ASSERT(false);
				s += char((unhex(val[i + 1]) << 4) + unhex(val[i + 2]));
				i += 2;
			} else
				ASSERT(false);
		} else
			s += c;
	}
	return s;
}

void DatabaseContext::validateVersion(Version version) const {
	// Version could be 0 if the INITIALIZE_NEW_DATABASE option is set. In that case, it is illegal to perform any
	// reads. We throw client_invalid_operation because the caller didn't directly set the version, so the
	// version_invalid error might be confusing.
	if (version == 0) {
		throw client_invalid_operation();
	}
	if (switchable && version < minAcceptableReadVersion) {
		CODE_PROBE(true, "Attempted to read a version lower than any this client has seen from the current cluster");
		throw transaction_too_old();
	}

	ASSERT(version > 0 || version == latestVersion);
}

void validateOptionValuePresent(Optional<StringRef> value) {
	if (!value.present()) {
		throw invalid_option_value();
	}
}

void validateOptionValueNotPresent(Optional<StringRef> value) {
	if (value.present() && value.get().size() > 0) {
		throw invalid_option_value();
	}
}

void dumpMutations(const MutationListRef& mutations) {
	for (auto m = mutations.begin(); m; ++m) {
		switch (m->type) {
		case MutationRef::SetValue:
			printf("  '%s' := '%s'\n", printable(m->param1).c_str(), printable(m->param2).c_str());
			break;
		case MutationRef::AddValue:
			printf("  '%s' += '%s'", printable(m->param1).c_str(), printable(m->param2).c_str());
			break;
		case MutationRef::ClearRange:
			printf("  Clear ['%s','%s')\n", printable(m->param1).c_str(), printable(m->param2).c_str());
			break;
		default:
			printf("  Unknown mutation %d('%s','%s')\n",
			       m->type,
			       printable(m->param1).c_str(),
			       printable(m->param2).c_str());
			break;
		}
	}
}

template <>
void addref(DatabaseContext* ptr) {
	ptr->addref();
}
template <>
void delref(DatabaseContext* ptr) {
	ptr->delref();
}

void traceTSSErrors(const char* name, UID tssId, const std::unordered_map<int, uint64_t>& errorsByCode) {
	TraceEvent ev(name, tssId);
	for (auto& it : errorsByCode) {
		ev.detail("E" + std::to_string(it.first), it.second);
	}
}

/*
    For each request type, this will produce
    <Type>Count
    <Type>{SS,TSS}{Mean,P50,P90,P99}
    Example:
    GetValueLatencySSMean
*/
void traceSSOrTSSPercentiles(TraceEvent& ev, const std::string name, DDSketch<double>& sample) {
	ev.detail(name + "Mean", sample.mean());
	// don't log the larger percentiles unless we actually have enough samples to log the accurate percentile instead of
	// the largest sample in this window
	if (sample.getPopulationSize() >= 3) {
		ev.detail(name + "P50", sample.median());
	}
	if (sample.getPopulationSize() >= 10) {
		ev.detail(name + "P90", sample.percentile(0.90));
	}
	if (sample.getPopulationSize() >= 100) {
		ev.detail(name + "P99", sample.percentile(0.99));
	}
}

void traceTSSPercentiles(TraceEvent& ev,
                         const std::string name,
                         DDSketch<double>& ssSample,
                         DDSketch<double>& tssSample) {
	ASSERT(ssSample.getPopulationSize() == tssSample.getPopulationSize());
	ev.detail(name + "Count", ssSample.getPopulationSize());
	if (ssSample.getPopulationSize() > 0) {
		traceSSOrTSSPercentiles(ev, name + "SS", ssSample);
		traceSSOrTSSPercentiles(ev, name + "TSS", tssSample);
	}
}

															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via tssLogger()
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class TssLoggerActor>
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class TssLoggerActorState {
															#line 623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TssLoggerActorState(DatabaseContext* const& cx) 
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   lastLogged(0)
															#line 632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("tssLogger", reinterpret_cast<unsigned long>(this));

	}
	~TssLoggerActorState() 
	{
		fdb_probe_actor_destroy("tssLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TssLoggerActorState();
		static_cast<TssLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(CLIENT_KNOBS->TSS_METRICS_LOGGING_INTERVAL, TaskPriority::FlushTrace);
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<TssLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<TssLoggerActor*>(this)->actor_wait_state = 1;
															#line 617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TssLoggerActor, 0, Void >*>(static_cast<TssLoggerActor*>(this)));
															#line 684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& it : cx->tssMetrics ) {
															#line 621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second->detailedMismatches.size())
															#line 695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->tssMismatchStream.send( std::pair<UID, std::vector<DetailedTSSMismatch>>(it.first, it.second->detailedMismatches));
															#line 699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second->ssErrorsByCode.size())
															#line 703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				traceTSSErrors("TSS_SSErrors", it.first, it.second->ssErrorsByCode);
															#line 707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second->tssErrorsByCode.size())
															#line 711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				traceTSSErrors("TSS_TSSErrors", it.first, it.second->tssErrorsByCode);
															#line 715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 635 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent tssEv("TSSClientMetrics", cx->dbId);
															#line 636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tssEv.detail("TSSID", it.first) .detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Internal", cx->internal);
															#line 640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			it.second->cc.logToTraceEvent(tssEv);
															#line 642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles(tssEv, "GetValueLatency", it.second->SSgetValueLatency, it.second->TSSgetValueLatency);
															#line 643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles( tssEv, "GetKeyValuesLatency", it.second->SSgetKeyValuesLatency, it.second->TSSgetKeyValuesLatency);
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles(tssEv, "GetKeyLatency", it.second->SSgetKeyLatency, it.second->TSSgetKeyLatency);
															#line 646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles(tssEv, "GetMappedKeyValuesLatency", it.second->SSgetMappedKeyValuesLatency, it.second->TSSgetMappedKeyValuesLatency);
															#line 651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			it.second->clear();
															#line 733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastLogged = now();
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& it : cx->tssMetrics ) {
															#line 621 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second->detailedMismatches.size())
															#line 748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->tssMismatchStream.send( std::pair<UID, std::vector<DetailedTSSMismatch>>(it.first, it.second->detailedMismatches));
															#line 752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 627 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second->ssErrorsByCode.size())
															#line 756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				traceTSSErrors("TSS_SSErrors", it.first, it.second->ssErrorsByCode);
															#line 760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 631 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second->tssErrorsByCode.size())
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 632 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				traceTSSErrors("TSS_TSSErrors", it.first, it.second->tssErrorsByCode);
															#line 768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 635 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent tssEv("TSSClientMetrics", cx->dbId);
															#line 636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tssEv.detail("TSSID", it.first) .detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Internal", cx->internal);
															#line 640 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			it.second->cc.logToTraceEvent(tssEv);
															#line 642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles(tssEv, "GetValueLatency", it.second->SSgetValueLatency, it.second->TSSgetValueLatency);
															#line 643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles( tssEv, "GetKeyValuesLatency", it.second->SSgetKeyValuesLatency, it.second->TSSgetKeyValuesLatency);
															#line 645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles(tssEv, "GetKeyLatency", it.second->SSgetKeyLatency, it.second->TSSgetKeyLatency);
															#line 646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			traceTSSPercentiles(tssEv, "GetMappedKeyValuesLatency", it.second->SSgetMappedKeyValuesLatency, it.second->TSSgetMappedKeyValuesLatency);
															#line 651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			it.second->clear();
															#line 786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 654 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastLogged = now();
															#line 790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TssLoggerActor*>(this)->actor_wait_state > 0) static_cast<TssLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<TssLoggerActor*>(this)->ActorCallback< TssLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TssLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TssLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TssLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double lastLogged;
															#line 874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via tssLogger()
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class TssLoggerActor final : public Actor<Void>, public ActorCallback< TssLoggerActor, 0, Void >, public FastAllocated<TssLoggerActor>, public TssLoggerActorState<TssLoggerActor> {
															#line 879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<TssLoggerActor>::operator new;
	using FastAllocated<TssLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8375833998563038208UL, 11613643615644268544UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TssLoggerActor, 0, Void >;
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TssLoggerActor(DatabaseContext* const& cx) 
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   TssLoggerActorState<TssLoggerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9014176353709562112UL, 14021610819005956096UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tssLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TssLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> tssLogger( DatabaseContext* const& cx ) {
															#line 614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new TssLoggerActor(cx));
															#line 929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via databaseLogger()
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class DatabaseLoggerActor>
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class DatabaseLoggerActorState {
															#line 941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseLoggerActorState(DatabaseContext* const& cx) 
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   lastLogged(0)
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("databaseLogger", reinterpret_cast<unsigned long>(this));

	}
	~DatabaseLoggerActorState() 
	{
		fdb_probe_actor_destroy("databaseLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DatabaseLoggerActorState();
		static_cast<DatabaseLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(CLIENT_KNOBS->SYSTEM_MONITOR_INTERVAL, TaskPriority::FlushTrace);
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<DatabaseLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<DatabaseLoggerActor*>(this)->actor_wait_state = 1;
															#line 661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DatabaseLoggerActor, 0, Void >*>(static_cast<DatabaseLoggerActor*>(this)));
															#line 1002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool logTraces = !g_network->isSimulated() || BUGGIFY_WITH_PROB(0.01);
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (logTraces)
															#line 1013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent ev("TransactionMetrics", cx->dbId);
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ev.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->cc.logToTraceEvent(ev);
															#line 676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ev.detail("LocationCacheEntryCount", cx->locationCache.size());
															#line 677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ev.detail("MeanLatency", cx->latencies.mean()) .detail("MedianLatency", cx->latencies.median()) .detail("Latency90", cx->latencies.percentile(0.90)) .detail("Latency98", cx->latencies.percentile(0.98)) .detail("MaxLatency", cx->latencies.max()) .detail("MeanRowReadLatency", cx->readLatencies.mean()) .detail("MedianRowReadLatency", cx->readLatencies.median()) .detail("MaxRowReadLatency", cx->readLatencies.max()) .detail("MeanGRVLatency", cx->GRVLatencies.mean()) .detail("MedianGRVLatency", cx->GRVLatencies.median()) .detail("MaxGRVLatency", cx->GRVLatencies.max()) .detail("MeanCommitLatency", cx->commitLatencies.mean()) .detail("MedianCommitLatency", cx->commitLatencies.median()) .detail("MaxCommitLatency", cx->commitLatencies.max()) .detail("MeanMutationsPerCommit", cx->mutationsPerCommit.mean()) .detail("MedianMutationsPerCommit", cx->mutationsPerCommit.median()) .detail("MaxMutationsPerCommit", cx->mutationsPerCommit.max()) .detail("MeanBytesPerCommit", cx->bytesPerCommit.mean()) .detail("MedianBytesPerCommit", cx->bytesPerCommit.median()) .detail("MaxBytesPerCommit", cx->bytesPerCommit.max()) .detail("NumLocalityCacheEntries", cx->locationCache.size());
															#line 1025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (cx->usedAnyChangeFeeds && logTraces)
															#line 1029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent feedEv("ChangeFeedClientMetrics", cx->dbId);
															#line 703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			feedEv.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->ccFeed.logToTraceEvent(feedEv);
															#line 1037 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (cx->anyBGReads && logTraces)
															#line 1041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent bgReadEv("BlobGranuleReadMetrics", cx->dbId);
															#line 716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			bgReadEv.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->ccBG.logToTraceEvent(bgReadEv);
															#line 727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			bgReadEv.detail("MeanBGLatency", cx->bgLatencies.mean()) .detail("MedianBGLatency", cx->bgLatencies.median()) .detail("MaxBGLatency", cx->bgLatencies.max()) .detail("MeanBGGranulesPerRequest", cx->bgGranulesPerRequest.mean()) .detail("MedianBGGranulesPerRequest", cx->bgGranulesPerRequest.median()) .detail("MaxBGGranulesPerRequest", cx->bgGranulesPerRequest.max());
															#line 1051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->latencies.clear();
															#line 736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->readLatencies.clear();
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->GRVLatencies.clear();
															#line 738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->commitLatencies.clear();
															#line 739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->mutationsPerCommit.clear();
															#line 740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->bytesPerCommit.clear();
															#line 741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->bgLatencies.clear();
															#line 742 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->bgGranulesPerRequest.clear();
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastLogged = now();
															#line 1071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool logTraces = !g_network->isSimulated() || BUGGIFY_WITH_PROB(0.01);
															#line 664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (logTraces)
															#line 1082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent ev("TransactionMetrics", cx->dbId);
															#line 667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ev.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 674 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->cc.logToTraceEvent(ev);
															#line 676 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ev.detail("LocationCacheEntryCount", cx->locationCache.size());
															#line 677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ev.detail("MeanLatency", cx->latencies.mean()) .detail("MedianLatency", cx->latencies.median()) .detail("Latency90", cx->latencies.percentile(0.90)) .detail("Latency98", cx->latencies.percentile(0.98)) .detail("MaxLatency", cx->latencies.max()) .detail("MeanRowReadLatency", cx->readLatencies.mean()) .detail("MedianRowReadLatency", cx->readLatencies.median()) .detail("MaxRowReadLatency", cx->readLatencies.max()) .detail("MeanGRVLatency", cx->GRVLatencies.mean()) .detail("MedianGRVLatency", cx->GRVLatencies.median()) .detail("MaxGRVLatency", cx->GRVLatencies.max()) .detail("MeanCommitLatency", cx->commitLatencies.mean()) .detail("MedianCommitLatency", cx->commitLatencies.median()) .detail("MaxCommitLatency", cx->commitLatencies.max()) .detail("MeanMutationsPerCommit", cx->mutationsPerCommit.mean()) .detail("MedianMutationsPerCommit", cx->mutationsPerCommit.median()) .detail("MaxMutationsPerCommit", cx->mutationsPerCommit.max()) .detail("MeanBytesPerCommit", cx->bytesPerCommit.mean()) .detail("MedianBytesPerCommit", cx->bytesPerCommit.median()) .detail("MaxBytesPerCommit", cx->bytesPerCommit.max()) .detail("NumLocalityCacheEntries", cx->locationCache.size());
															#line 1094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (cx->usedAnyChangeFeeds && logTraces)
															#line 1098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent feedEv("ChangeFeedClientMetrics", cx->dbId);
															#line 703 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			feedEv.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->ccFeed.logToTraceEvent(feedEv);
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (cx->anyBGReads && logTraces)
															#line 1110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent bgReadEv("BlobGranuleReadMetrics", cx->dbId);
															#line 716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			bgReadEv.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->ccBG.logToTraceEvent(bgReadEv);
															#line 727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			bgReadEv.detail("MeanBGLatency", cx->bgLatencies.mean()) .detail("MedianBGLatency", cx->bgLatencies.median()) .detail("MaxBGLatency", cx->bgLatencies.max()) .detail("MeanBGGranulesPerRequest", cx->bgGranulesPerRequest.mean()) .detail("MedianBGGranulesPerRequest", cx->bgGranulesPerRequest.median()) .detail("MaxBGGranulesPerRequest", cx->bgGranulesPerRequest.max());
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->latencies.clear();
															#line 736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->readLatencies.clear();
															#line 737 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->GRVLatencies.clear();
															#line 738 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->commitLatencies.clear();
															#line 739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->mutationsPerCommit.clear();
															#line 740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->bytesPerCommit.clear();
															#line 741 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->bgLatencies.clear();
															#line 742 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->bgGranulesPerRequest.clear();
															#line 744 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastLogged = now();
															#line 1140 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DatabaseLoggerActor*>(this)->actor_wait_state > 0) static_cast<DatabaseLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<DatabaseLoggerActor*>(this)->ActorCallback< DatabaseLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DatabaseLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DatabaseLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DatabaseLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double lastLogged;
															#line 1224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via databaseLogger()
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class DatabaseLoggerActor final : public Actor<Void>, public ActorCallback< DatabaseLoggerActor, 0, Void >, public FastAllocated<DatabaseLoggerActor>, public DatabaseLoggerActorState<DatabaseLoggerActor> {
															#line 1229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<DatabaseLoggerActor>::operator new;
	using FastAllocated<DatabaseLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10039735589468325120UL, 5338402242409730560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DatabaseLoggerActor, 0, Void >;
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseLoggerActor(DatabaseContext* const& cx) 
															#line 1246 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   DatabaseLoggerActorState<DatabaseLoggerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13968243465581786624UL, 631654202351916032UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("databaseLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DatabaseLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> databaseLogger( DatabaseContext* const& cx ) {
															#line 658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new DatabaseLoggerActor(cx));
															#line 1279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

struct TrInfoChunk {
	ValueRef value;
	Key key;
};

															#line 1289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via transactionInfoCommitActor()
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class TransactionInfoCommitActorActor>
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class TransactionInfoCommitActorActorState {
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TransactionInfoCommitActorActorState(Transaction* const& tr,std::vector<TrInfoChunk>* const& chunks) 
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : tr(tr),
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   chunks(chunks),
															#line 754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientLatencyAtomicCtr(CLIENT_LATENCY_INFO_CTR_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin)),
															#line 755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   retryCount(0)
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this));

	}
	~TransactionInfoCommitActorActorState() 
	{
		fdb_probe_actor_destroy("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 1324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TransactionInfoCommitActorActorState();
		static_cast<TransactionInfoCommitActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 758 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->reset();
															#line 759 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 760 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			vstamp = tr->getVersionstamp();
															#line 762 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			int64_t numCommitBytes = 0;
															#line 763 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			for( auto& chunk : *chunks ) {
															#line 764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tr->atomicOp(chunk.key, chunk.value, MutationRef::SetVersionstampedKey);
															#line 765 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				numCommitBytes += chunk.key.size() + chunk.value.size() - 4;
															#line 1369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 768 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->atomicOp(clientLatencyAtomicCtr, StringRef((uint8_t*)&numCommitBytes, 8), MutationRef::AddValue);
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr->commit();
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1377 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 1;
															#line 769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TransactionInfoCommitActorActor, 0, Void >*>(static_cast<TransactionInfoCommitActorActor*>(this)));
															#line 1382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 772 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			retryCount++;
															#line 773 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (retryCount == 10)
															#line 1406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 774 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 1410 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr->onError(e);
															#line 775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1416 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 2;
															#line 775 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TransactionInfoCommitActorActor, 1, Void >*>(static_cast<TransactionInfoCommitActorActor*>(this)));
															#line 1421 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<TransactionInfoCommitActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TransactionInfoCommitActorActorState(); static_cast<TransactionInfoCommitActorActor*>(this)->destroy(); return 0; }
															#line 1436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<TransactionInfoCommitActorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TransactionInfoCommitActorActorState();
		static_cast<TransactionInfoCommitActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<TransactionInfoCommitActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TransactionInfoCommitActorActorState(); static_cast<TransactionInfoCommitActorActor*>(this)->destroy(); return 0; }
															#line 1448 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<TransactionInfoCommitActorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TransactionInfoCommitActorActorState();
		static_cast<TransactionInfoCommitActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state > 0) static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionInfoCommitActorActor*>(this)->ActorCallback< TransactionInfoCommitActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TransactionInfoCommitActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state > 0) static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionInfoCommitActorActor*>(this)->ActorCallback< TransactionInfoCommitActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TransactionInfoCommitActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction* tr;
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<TrInfoChunk>* chunks;
															#line 754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	const Key clientLatencyAtomicCtr;
															#line 755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int retryCount;
															#line 761 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Future<Standalone<StringRef>> vstamp;
															#line 1628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via transactionInfoCommitActor()
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class TransactionInfoCommitActorActor final : public Actor<Void>, public ActorCallback< TransactionInfoCommitActorActor, 0, Void >, public ActorCallback< TransactionInfoCommitActorActor, 1, Void >, public FastAllocated<TransactionInfoCommitActorActor>, public TransactionInfoCommitActorActorState<TransactionInfoCommitActorActor> {
															#line 1633 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<TransactionInfoCommitActorActor>::operator new;
	using FastAllocated<TransactionInfoCommitActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6118403463054976000UL, 12303674325751309824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TransactionInfoCommitActorActor, 0, Void >;
friend struct ActorCallback< TransactionInfoCommitActorActor, 1, Void >;
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TransactionInfoCommitActorActor(Transaction* const& tr,std::vector<TrInfoChunk>* const& chunks) 
															#line 1651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   TransactionInfoCommitActorActorState<TransactionInfoCommitActorActor>(tr, chunks),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10077237674783052032UL, 3830813435306680320UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("transactionInfoCommitActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TransactionInfoCommitActorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TransactionInfoCommitActorActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> transactionInfoCommitActor( Transaction* const& tr, std::vector<TrInfoChunk>* const& chunks ) {
															#line 753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new TransactionInfoCommitActorActor(tr, chunks));
															#line 1685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 1690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via delExcessClntTxnEntriesActor()
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class DelExcessClntTxnEntriesActorActor>
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class DelExcessClntTxnEntriesActorActorState {
															#line 1697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DelExcessClntTxnEntriesActorActorState(Transaction* const& tr,int64_t const& clientTxInfoSizeLimit) 
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : tr(tr),
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientTxInfoSizeLimit(clientTxInfoSizeLimit),
															#line 781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientLatencyName(CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin)),
															#line 782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientLatencyAtomicCtr(CLIENT_LATENCY_INFO_CTR_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin))
															#line 1710 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this));

	}
	~DelExcessClntTxnEntriesActorActorState() 
	{
		fdb_probe_actor_destroy("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "DelExcessClntTxnEntriesCalled").log();
															#line 784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 1727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DelExcessClntTxnEntriesActorActorState();
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->reset();
															#line 787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(KeyRef(clientLatencyAtomicCtr), Snapshot::True);
															#line 789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 1;
															#line 789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 1771 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr->onError(e);
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 4;
															#line 824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 1800 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& ctrValue,int loopDepth) 
	{
															#line 790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!ctrValue.present())
															#line 1815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "NumClntTxnEntriesNotFound").log();
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 1821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txInfoSize = 0;
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(ctrValue.get().size() == sizeof(int64_t));
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		memcpy(&txInfoSize, ctrValue.get().begin(), ctrValue.get().size());
															#line 797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (txInfoSize < clientTxInfoSizeLimit)
															#line 1835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 1839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		int getRangeByteLimit = (txInfoSize - clientTxInfoSizeLimit) < CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT ? (txInfoSize - clientTxInfoSizeLimit) : CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		GetRangeLimits limit(GetRangeLimits::ROW_LIMIT_UNLIMITED, getRangeByteLimit);
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr->getRange(KeyRangeRef(clientLatencyName, strinc(clientLatencyName)), limit);
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 2;
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 1858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && ctrValue,int loopDepth) 
	{
															#line 790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!ctrValue.present())
															#line 1867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "NumClntTxnEntriesNotFound").log();
															#line 792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 1873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txInfoSize = 0;
															#line 795 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(ctrValue.get().size() == sizeof(int64_t));
															#line 796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		memcpy(&txInfoSize, ctrValue.get().begin(), ctrValue.get().size());
															#line 797 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (txInfoSize < clientTxInfoSizeLimit)
															#line 1887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 1891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 799 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		int getRangeByteLimit = (txInfoSize - clientTxInfoSizeLimit) < CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT ? (txInfoSize - clientTxInfoSizeLimit) : CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		GetRangeLimits limit(GetRangeLimits::ROW_LIMIT_UNLIMITED, getRangeByteLimit);
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr->getRange(KeyRangeRef(clientLatencyName, strinc(clientLatencyName)), limit);
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 2;
															#line 803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 1910 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& ctrValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(ctrValue, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && ctrValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(ctrValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& txEntries,int loopDepth) 
	{
															#line 805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		numBytesToDel = 0;
															#line 806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		KeyRef endKey;
															#line 807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( auto& kv : txEntries ) {
															#line 808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			endKey = kv.key;
															#line 809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			numBytesToDel += kv.key.size() + kv.value.size();
															#line 810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (txInfoSize - numBytesToDel <= clientTxInfoSizeLimit)
															#line 2004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
				break;
			}
		}
															#line 813 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (numBytesToDel)
															#line 2011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->clear(KeyRangeRef(txEntries[0].key, strinc(endKey)));
															#line 815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "DeletingExcessCntTxnEntries").detail("BytesToBeDeleted", numBytesToDel);
															#line 816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			int64_t bytesDel = -numBytesToDel;
															#line 818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->atomicOp(clientLatencyAtomicCtr, StringRef((uint8_t*)&bytesDel, 8), MutationRef::AddValue);
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 3;
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && txEntries,int loopDepth) 
	{
															#line 805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		numBytesToDel = 0;
															#line 806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		KeyRef endKey;
															#line 807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( auto& kv : txEntries ) {
															#line 808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			endKey = kv.key;
															#line 809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			numBytesToDel += kv.key.size() + kv.value.size();
															#line 810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (txInfoSize - numBytesToDel <= clientTxInfoSizeLimit)
															#line 2054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
				break;
			}
		}
															#line 813 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (numBytesToDel)
															#line 2061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 814 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->clear(KeyRangeRef(txEntries[0].key, strinc(endKey)));
															#line 815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "DeletingExcessCntTxnEntries").detail("BytesToBeDeleted", numBytesToDel);
															#line 816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			int64_t bytesDel = -numBytesToDel;
															#line 818 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->atomicOp(clientLatencyAtomicCtr, StringRef((uint8_t*)&bytesDel, 8), MutationRef::AddValue);
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 3;
															#line 819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& txEntries,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(txEntries, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && txEntries,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(txEntries), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 821 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (txInfoSize - numBytesToDel <= clientTxInfoSizeLimit)
															#line 2169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 822 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 2173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction* tr;
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int64_t clientTxInfoSizeLimit;
															#line 781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	const Key clientLatencyName;
															#line 782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	const Key clientLatencyAtomicCtr;
															#line 794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int64_t txInfoSize;
															#line 805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int64_t numBytesToDel;
															#line 2382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via delExcessClntTxnEntriesActor()
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class DelExcessClntTxnEntriesActorActor final : public Actor<Void>, public ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >, public ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >, public ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >, public ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >, public FastAllocated<DelExcessClntTxnEntriesActorActor>, public DelExcessClntTxnEntriesActorActorState<DelExcessClntTxnEntriesActorActor> {
															#line 2387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<DelExcessClntTxnEntriesActorActor>::operator new;
	using FastAllocated<DelExcessClntTxnEntriesActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3791583495363600128UL, 10270169944007207424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >;
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >;
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >;
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >;
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DelExcessClntTxnEntriesActorActor(Transaction* const& tr,int64_t const& clientTxInfoSizeLimit) 
															#line 2407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   DelExcessClntTxnEntriesActorActorState<DelExcessClntTxnEntriesActorActor>(tr, clientTxInfoSizeLimit),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12258001942667996416UL, 1429686287236455680UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("delExcessClntTxnEntriesActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> delExcessClntTxnEntriesActor( Transaction* const& tr, int64_t const& clientTxInfoSizeLimit ) {
															#line 780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new DelExcessClntTxnEntriesActorActor(tr, clientTxInfoSizeLimit));
															#line 2443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 828 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// Delref and addref self to give self a chance to get destroyed.
															#line 2449 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via refreshTransaction()
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class RefreshTransactionActor>
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class RefreshTransactionActorState {
															#line 2456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	RefreshTransactionActorState(DatabaseContext* const& self,Transaction* const& tr) 
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : self(self),
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tr(tr)
															#line 2465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("refreshTransaction", reinterpret_cast<unsigned long>(this));

	}
	~RefreshTransactionActorState() 
	{
		fdb_probe_actor_destroy("refreshTransaction", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			*tr = Transaction();
															#line 832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0);
															#line 832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<RefreshTransactionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2484 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RefreshTransactionActor*>(this)->actor_wait_state = 1;
															#line 832 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RefreshTransactionActor, 0, Void >*>(static_cast<RefreshTransactionActor*>(this)));
															#line 2489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RefreshTransactionActorState();
		static_cast<RefreshTransactionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		*tr = Transaction(Database(Reference<DatabaseContext>::addRef(self)));
															#line 834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<RefreshTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RefreshTransactionActorState(); static_cast<RefreshTransactionActor*>(this)->destroy(); return 0; }
															#line 2514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<RefreshTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RefreshTransactionActorState();
		static_cast<RefreshTransactionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 833 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		*tr = Transaction(Database(Reference<DatabaseContext>::addRef(self)));
															#line 834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<RefreshTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RefreshTransactionActorState(); static_cast<RefreshTransactionActor*>(this)->destroy(); return 0; }
															#line 2528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<RefreshTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RefreshTransactionActorState();
		static_cast<RefreshTransactionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RefreshTransactionActor*>(this)->actor_wait_state > 0) static_cast<RefreshTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshTransactionActor*>(this)->ActorCallback< RefreshTransactionActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshTransactionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RefreshTransactionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RefreshTransactionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* self;
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction* tr;
															#line 2615 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via refreshTransaction()
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class RefreshTransactionActor final : public Actor<Void>, public ActorCallback< RefreshTransactionActor, 0, Void >, public FastAllocated<RefreshTransactionActor>, public RefreshTransactionActorState<RefreshTransactionActor> {
															#line 2620 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<RefreshTransactionActor>::operator new;
	using FastAllocated<RefreshTransactionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2708099383033008640UL, 3953214711615123200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RefreshTransactionActor, 0, Void >;
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	RefreshTransactionActor(DatabaseContext* const& self,Transaction* const& tr) 
															#line 2637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   RefreshTransactionActorState<RefreshTransactionActor>(self, tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("refreshTransaction", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4345345362439766272UL, 3425069992022049280UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("refreshTransaction");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("refreshTransaction", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RefreshTransactionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> refreshTransaction( DatabaseContext* const& self, Transaction* const& tr ) {
															#line 830 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new RefreshTransactionActor(self, tr));
															#line 2670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// The reason for getting a pointer to DatabaseContext instead of a reference counted object is because reference
// counting will increment reference count for DatabaseContext which holds the future of this actor. This creates a
// cyclic reference and hence this actor and Database object will not be destroyed at all.
															#line 2678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via clientStatusUpdateActor()
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class ClientStatusUpdateActorActor>
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class ClientStatusUpdateActorActorState {
															#line 2685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	ClientStatusUpdateActorActorState(DatabaseContext* const& cx) 
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 841 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientLatencyName(CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin).toString()),
															#line 843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tr(),
															#line 844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   commitQ(),
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   txBytes(0)
															#line 2700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this));

	}
	~ClientStatusUpdateActorActorState() 
	{
		fdb_probe_actor_destroy("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 2715 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ClientStatusUpdateActorActorState();
		static_cast<ClientStatusUpdateActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = refreshTransaction(cx, &tr);
															#line 850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2747 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 1;
															#line 850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 0, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 2752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(cx->clientStatusUpdater.outStatusQ.empty());
															#line 853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->clientStatusUpdater.inStatusQ.swap(cx->clientStatusUpdater.outStatusQ);
															#line 855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			trChunksQ = std::vector<TrInfoChunk>();
															#line 856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			for( auto& entry : cx->clientStatusUpdater.outStatusQ ) {
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				auto& bw = entry.second;
															#line 858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				int64_t value_size_limit = BUGGIFY ? deterministicRandom()->randomInt(1e3, CLIENT_KNOBS->VALUE_SIZE_LIMIT) : CLIENT_KNOBS->VALUE_SIZE_LIMIT;
															#line 861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				int num_chunks = (bw.getLength() + value_size_limit - 1) / value_size_limit;
															#line 862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				std::string random_id = deterministicRandom()->randomAlphaNumeric(16);
															#line 863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				std::string user_provided_id = entry.first.size() ? entry.first + "/" : "";
															#line 864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				for(int i = 0;i < num_chunks;i++) {
															#line 865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					TrInfoChunk chunk;
															#line 866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					BinaryWriter chunkBW(Unversioned());
															#line 867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					chunkBW << bigEndian32(i + 1) << bigEndian32(num_chunks);
															#line 868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					chunk.key = KeyRef(clientLatencyName + std::string(10, '\x00') + "/" + random_id + "/" + chunkBW.toValue().toString() + "/" + user_provided_id + std::string(4, '\x00'));
															#line 870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					int32_t pos = littleEndian32(clientLatencyName.size());
															#line 871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					memcpy(mutateString(chunk.key) + chunk.key.size() - sizeof(int32_t), &pos, sizeof(int32_t));
															#line 872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (i == num_chunks - 1)
															#line 2794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), bw.getLength() - (i * value_size_limit));
															#line 2798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
					else
					{
															#line 876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), value_size_limit);
															#line 2804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
															#line 879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					trChunksQ.push_back(std::move(chunk));
															#line 2808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
			}
															#line 884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			dataSizeLimit = BUGGIFY ? deterministicRandom()->randomInt(200e3, 1.5 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT) : 0.8 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracking_iter = trChunksQ.begin();
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(commitQ.empty() && (txBytes == 0));
															#line 889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 2819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 852 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(cx->clientStatusUpdater.outStatusQ.empty());
															#line 853 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->clientStatusUpdater.inStatusQ.swap(cx->clientStatusUpdater.outStatusQ);
															#line 855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			trChunksQ = std::vector<TrInfoChunk>();
															#line 856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			for( auto& entry : cx->clientStatusUpdater.outStatusQ ) {
															#line 857 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				auto& bw = entry.second;
															#line 858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				int64_t value_size_limit = BUGGIFY ? deterministicRandom()->randomInt(1e3, CLIENT_KNOBS->VALUE_SIZE_LIMIT) : CLIENT_KNOBS->VALUE_SIZE_LIMIT;
															#line 861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				int num_chunks = (bw.getLength() + value_size_limit - 1) / value_size_limit;
															#line 862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				std::string random_id = deterministicRandom()->randomAlphaNumeric(16);
															#line 863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				std::string user_provided_id = entry.first.size() ? entry.first + "/" : "";
															#line 864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				for(int i = 0;i < num_chunks;i++) {
															#line 865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					TrInfoChunk chunk;
															#line 866 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					BinaryWriter chunkBW(Unversioned());
															#line 867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					chunkBW << bigEndian32(i + 1) << bigEndian32(num_chunks);
															#line 868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					chunk.key = KeyRef(clientLatencyName + std::string(10, '\x00') + "/" + random_id + "/" + chunkBW.toValue().toString() + "/" + user_provided_id + std::string(4, '\x00'));
															#line 870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					int32_t pos = littleEndian32(clientLatencyName.size());
															#line 871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					memcpy(mutateString(chunk.key) + chunk.key.size() - sizeof(int32_t), &pos, sizeof(int32_t));
															#line 872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (i == num_chunks - 1)
															#line 2867 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), bw.getLength() - (i * value_size_limit));
															#line 2871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
					else
					{
															#line 876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), value_size_limit);
															#line 2877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
															#line 879 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					trChunksQ.push_back(std::move(chunk));
															#line 2881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
			}
															#line 884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			dataSizeLimit = BUGGIFY ? deterministicRandom()->randomInt(200e3, 1.5 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT) : 0.8 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracking_iter = trChunksQ.begin();
															#line 888 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(commitQ.empty() && (txBytes == 0));
															#line 889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 2892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 2989 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 2993 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 939 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->clientStatusUpdater.outStatusQ.clear();
															#line 940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevWarnAlways, "UnableToWriteClientStatus").error(e);
															#line 941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_6 = delay(10.0);
															#line 941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_6.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 7;
															#line 941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 6, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->clientStatusUpdater.outStatusQ.clear();
															#line 924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = cx->globalConfig->onInitialized();
															#line 924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3027 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 4;
															#line 924 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 3, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		iter = tracking_iter;
															#line 891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txBytes = 0;
															#line 892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		commitQ.clear();
															#line 3052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		try {
															#line 894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 3056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_transaction_too_large)
															#line 3091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				dataSizeLimit /= 2;
															#line 914 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				ASSERT(dataSizeLimit >= CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->KEY_SIZE_LIMIT);
															#line 3097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				TraceEvent(SevWarnAlways, "ClientTrInfoErrorCommit").error(e).detail("TxBytes", txBytes);
															#line 917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				commitQ.clear();
															#line 918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				txBytes = 0;
															#line 919 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1loopBody1cont1Catch1(e, std::max(0, loopDepth - 1));
															#line 3109 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			loopDepth = a_body1loopBody1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont2(int loopDepth) 
	{
															#line 905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!commitQ.empty())
															#line 3125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = transactionInfoCommitActor(&tr, &commitQ);
															#line 906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3131 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 3;
															#line 906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 2, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3136 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1(int loopDepth) 
	{
															#line 894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!(iter != trChunksQ.end()))
															#line 3157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			return a_body1loopBody1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (iter->value.size() + iter->key.size() + txBytes > dataSizeLimit)
															#line 3163 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = transactionInfoCommitActor(&tr, &commitQ);
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 2;
															#line 896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 1, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		commitQ.push_back(*iter);
															#line 902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txBytes += iter->value.size() + iter->key.size();
															#line 903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++iter;
															#line 3205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tracking_iter = iter;
															#line 898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		commitQ.clear();
															#line 899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txBytes = 0;
															#line 3218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tracking_iter = iter;
															#line 898 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		commitQ.clear();
															#line 899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txBytes = 0;
															#line 3231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont1loopBody1cont3(int loopDepth) 
	{
		return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		commitQ.clear();
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txBytes = 0;
															#line 3323 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		commitQ.clear();
															#line 908 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		txBytes = 0;
															#line 3334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double sampleRate = cx->globalConfig->get<double>(fdbClientInfoTxnSampleRate, std::numeric_limits<double>::infinity());
															#line 927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double clientSamplingProbability = std::isinf(sampleRate) ? CLIENT_KNOBS->CSI_SAMPLING_PROBABILITY : sampleRate;
															#line 929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		int64_t sizeLimit = cx->globalConfig->get<int64_t>(fdbClientInfoTxnSizeLimit, -1);
															#line 930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		int64_t clientTxnInfoSizeLimit = sizeLimit == -1 ? CLIENT_KNOBS->CSI_SIZE_LIMIT : sizeLimit;
															#line 931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!trChunksQ.empty() && deterministicRandom()->random01() < clientSamplingProbability)
															#line 3426 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delExcessClntTxnEntriesActor(&tr, clientTxnInfoSizeLimit);
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3432 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_4.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 5;
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 4, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3437 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double sampleRate = cx->globalConfig->get<double>(fdbClientInfoTxnSampleRate, std::numeric_limits<double>::infinity());
															#line 927 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double clientSamplingProbability = std::isinf(sampleRate) ? CLIENT_KNOBS->CSI_SAMPLING_PROBABILITY : sampleRate;
															#line 929 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		int64_t sizeLimit = cx->globalConfig->get<int64_t>(fdbClientInfoTxnSizeLimit, -1);
															#line 930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		int64_t clientTxnInfoSizeLimit = sizeLimit == -1 ? CLIENT_KNOBS->CSI_SIZE_LIMIT : sizeLimit;
															#line 931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!trChunksQ.empty() && deterministicRandom()->random01() < clientSamplingProbability)
															#line 3459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delExcessClntTxnEntriesActor(&tr, clientTxnInfoSizeLimit);
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3465 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_4.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 5;
															#line 932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 4, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = delay(CLIENT_KNOBS->CSI_STATUS_DELAY);
															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1cont1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_5.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 6;
															#line 934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 5, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont11cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 841 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	const std::string clientLatencyName;
															#line 843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction tr;
															#line 844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<TrInfoChunk> commitQ;
															#line 845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int txBytes;
															#line 855 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<TrInfoChunk> trChunksQ;
															#line 884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int64_t dataSizeLimit;
															#line 887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<TrInfoChunk>::iterator tracking_iter;
															#line 890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<TrInfoChunk>::iterator iter;
															#line 3863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via clientStatusUpdateActor()
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class ClientStatusUpdateActorActor final : public Actor<Void>, public ActorCallback< ClientStatusUpdateActorActor, 0, Void >, public ActorCallback< ClientStatusUpdateActorActor, 1, Void >, public ActorCallback< ClientStatusUpdateActorActor, 2, Void >, public ActorCallback< ClientStatusUpdateActorActor, 3, Void >, public ActorCallback< ClientStatusUpdateActorActor, 4, Void >, public ActorCallback< ClientStatusUpdateActorActor, 5, Void >, public ActorCallback< ClientStatusUpdateActorActor, 6, Void >, public FastAllocated<ClientStatusUpdateActorActor>, public ClientStatusUpdateActorActorState<ClientStatusUpdateActorActor> {
															#line 3868 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<ClientStatusUpdateActorActor>::operator new;
	using FastAllocated<ClientStatusUpdateActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8162996771772256512UL, 2335885788935913216UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ClientStatusUpdateActorActor, 0, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 1, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 2, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 3, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 4, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 5, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 6, Void >;
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	ClientStatusUpdateActorActor(DatabaseContext* const& cx) 
															#line 3891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   ClientStatusUpdateActorActorState<ClientStatusUpdateActorActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5908733088513914368UL, 12349308750647825408UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clientStatusUpdateActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> clientStatusUpdateActor( DatabaseContext* const& cx ) {
															#line 840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new ClientStatusUpdateActorActor(cx));
															#line 3930 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 3935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via assertFailure()
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class AssertFailureActor>
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class AssertFailureActorState {
															#line 3942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	AssertFailureActorState(GrvProxyInterface const& remote,Future<ErrorOr<GetReadVersionReply>> const& reply) 
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : remote(remote),
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   reply(reply)
															#line 3951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("assertFailure", reinterpret_cast<unsigned long>(this));

	}
	~AssertFailureActorState() 
	{
		fdb_probe_actor_destroy("assertFailure", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<ErrorOr<GetReadVersionReply>> __when_expr_0 = reply;
															#line 948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<AssertFailureActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<AssertFailureActor*>(this)->actor_wait_state = 1;
															#line 948 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*>(static_cast<AssertFailureActor*>(this)));
															#line 3974 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AssertFailureActorState();
		static_cast<AssertFailureActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<AssertFailureActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AssertFailureActorState(); static_cast<AssertFailureActor*>(this)->destroy(); return 0; }
															#line 4003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<AssertFailureActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AssertFailureActorState();
		static_cast<AssertFailureActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetReadVersionReply> const& res,int loopDepth) 
	{
															#line 949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!res.isError())
															#line 4036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevError, "GotStaleReadVersion") .detail("Remote", remote.getConsistentReadVersion.getEndpoint().addresses.address.toString()) .detail("Provisional", remote.provisional) .detail("ReadVersion", res.get().version);
															#line 954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 4042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetReadVersionReply> && res,int loopDepth) 
	{
															#line 949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!res.isError())
															#line 4052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevError, "GotStaleReadVersion") .detail("Remote", remote.getConsistentReadVersion.getEndpoint().addresses.address.toString()) .detail("Provisional", remote.provisional) .detail("ReadVersion", res.get().version);
															#line 954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 4058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when1(ErrorOr<GetReadVersionReply> const& res,int loopDepth) 
	{
		loopDepth = a_body1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(ErrorOr<GetReadVersionReply> && res,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AssertFailureActor*>(this)->actor_wait_state > 0) static_cast<AssertFailureActor*>(this)->actor_wait_state = 0;
		static_cast<AssertFailureActor*>(this)->ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >::remove();

	}
	void a_callback_fire(ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*,ErrorOr<GetReadVersionReply> const& value) 
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*,ErrorOr<GetReadVersionReply> && value) 
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*,Error err) 
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GrvProxyInterface remote;
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Future<ErrorOr<GetReadVersionReply>> reply;
															#line 4156 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via assertFailure()
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class AssertFailureActor final : public Actor<Void>, public ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >, public FastAllocated<AssertFailureActor>, public AssertFailureActorState<AssertFailureActor> {
															#line 4161 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<AssertFailureActor>::operator new;
	using FastAllocated<AssertFailureActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13279298725143748096UL, 18093091821433421312UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >;
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	AssertFailureActor(GrvProxyInterface const& remote,Future<ErrorOr<GetReadVersionReply>> const& reply) 
															#line 4178 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   AssertFailureActorState<AssertFailureActor>(remote, reply),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9499482857341526272UL, 8100088389272420864UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("assertFailure");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> assertFailure( GrvProxyInterface const& remote, Future<ErrorOr<GetReadVersionReply>> const& reply ) {
															#line 946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new AssertFailureActor(remote, reply));
															#line 4211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

Future<Void> attemptGRVFromOldProxies(std::vector<GrvProxyInterface> oldProxies,
                                      std::vector<GrvProxyInterface> newProxies) {
	auto debugID = nondeterministicRandom()->randomUniqueID();
	g_traceBatch.addEvent("AttemptGRVFromOldProxyDebug", debugID.first(), "NativeAPI.attemptGRVFromOldProxies.Start");
	Span span("NAPI:VerifyCausalReadRisky"_loc);
	std::vector<Future<Void>> replies;
	replies.reserve(oldProxies.size());
	GetReadVersionRequest req(
	    span.context, 1, TransactionPriority::IMMEDIATE, GetReadVersionRequest::FLAG_CAUSAL_READ_RISKY);
	TraceEvent evt("AttemptGRVFromOldProxies");
	evt.detail("NumOldProxies", oldProxies.size()).detail("NumNewProxies", newProxies.size());
	auto traceProxies = [&](std::vector<GrvProxyInterface>& proxies, std::string const& key) {
		for (int i = 0; i < proxies.size(); ++i) {
			auto k = key + std::to_string(i);
			evt.detail(k.c_str(), proxies[i].id());
		}
	};
	traceProxies(oldProxies, "OldProxy"s);
	traceProxies(newProxies, "NewProxy"s);
	evt.log();
	for (auto& i : oldProxies) {
		req.reply = ReplyPromise<GetReadVersionReply>();
		replies.push_back(assertFailure(i, i.getConsistentReadVersion.tryGetReply(req)));
	}
	return waitForAll(replies);
}

															#line 4243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorClientDBInfoChange()
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class MonitorClientDBInfoChangeActor>
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class MonitorClientDBInfoChangeActorState {
															#line 4250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	MonitorClientDBInfoChangeActorState(DatabaseContext* const& cx,Reference<AsyncVar<ClientDBInfo> const> const& clientDBInfo,AsyncTrigger* const& proxiesChangeTrigger) 
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientDBInfo(clientDBInfo),
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   proxiesChangeTrigger(proxiesChangeTrigger),
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   curCommitProxies(),
															#line 996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   curGrvProxies(),
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   actors(false),
															#line 998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   clientDBInfoOnChange(clientDBInfo->onChange())
															#line 4269 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this));

	}
	~MonitorClientDBInfoChangeActorState() 
	{
		fdb_probe_actor_destroy("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			curCommitProxies = clientDBInfo->get().commitProxies;
															#line 1000 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			curGrvProxies = clientDBInfo->get().grvProxies;
															#line 1002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 4288 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorClientDBInfoChangeActorState();
		static_cast<MonitorClientDBInfoChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = clientDBInfoOnChange;
															#line 1003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 1028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = actors.getResult();
															#line 4324 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state = 1;
															#line 1004 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*>(static_cast<MonitorClientDBInfoChangeActor*>(this)));
															#line 1028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*>(static_cast<MonitorClientDBInfoChangeActor*>(this)));
															#line 4331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 1005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		clientDBInfoOnChange = clientDBInfo->onChange();
															#line 1006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (clientDBInfo->get().commitProxies != curCommitProxies || clientDBInfo->get().grvProxies != curGrvProxies)
															#line 4348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (deterministicRandom()->random01() < cx->verifyCausalReadsProp && !curGrvProxies.empty() && !clientDBInfo->get().grvProxies.empty() && !clientDBInfo->get().grvProxies[0].provisional)
															#line 4352 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				actors.add(attemptGRVFromOldProxies(curGrvProxies, clientDBInfo->get().grvProxies));
															#line 4356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 1020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			curCommitProxies = clientDBInfo->get().commitProxies;
															#line 1021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			curGrvProxies = clientDBInfo->get().grvProxies;
															#line 1024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->ssVersionVectorCache.clear();
															#line 1025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			proxiesChangeTrigger->trigger();
															#line 4366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 1005 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		clientDBInfoOnChange = clientDBInfo->onChange();
															#line 1006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (clientDBInfo->get().commitProxies != curCommitProxies || clientDBInfo->get().grvProxies != curGrvProxies)
															#line 4378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (deterministicRandom()->random01() < cx->verifyCausalReadsProp && !curGrvProxies.empty() && !clientDBInfo->get().grvProxies.empty() && !clientDBInfo->get().grvProxies[0].provisional)
															#line 4382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1018 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				actors.add(attemptGRVFromOldProxies(curGrvProxies, clientDBInfo->get().grvProxies));
															#line 4386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 1020 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			curCommitProxies = clientDBInfo->get().commitProxies;
															#line 1021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			curGrvProxies = clientDBInfo->get().grvProxies;
															#line 1024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->ssVersionVectorCache.clear();
															#line 1025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			proxiesChangeTrigger->trigger();
															#line 4396 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 1029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		UNSTOPPABLE_ASSERT(false);
															#line 4406 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 1029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		UNSTOPPABLE_ASSERT(false);
															#line 4415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientDBInfoChangeActor*>(this)->ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >::remove();
		static_cast<MonitorClientDBInfoChangeActor*>(this)->ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<AsyncVar<ClientDBInfo> const> clientDBInfo;
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	AsyncTrigger* proxiesChangeTrigger;
															#line 995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<CommitProxyInterface> curCommitProxies;
															#line 996 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<GrvProxyInterface> curGrvProxies;
															#line 997 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	ActorCollection actors;
															#line 998 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Future<Void> clientDBInfoOnChange;
															#line 4555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via monitorClientDBInfoChange()
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class MonitorClientDBInfoChangeActor final : public Actor<Void>, public ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >, public ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >, public FastAllocated<MonitorClientDBInfoChangeActor>, public MonitorClientDBInfoChangeActorState<MonitorClientDBInfoChangeActor> {
															#line 4560 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<MonitorClientDBInfoChangeActor>::operator new;
	using FastAllocated<MonitorClientDBInfoChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10374102417319970816UL, 14681946826576335104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >;
friend struct ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >;
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	MonitorClientDBInfoChangeActor(DatabaseContext* const& cx,Reference<AsyncVar<ClientDBInfo> const> const& clientDBInfo,AsyncTrigger* const& proxiesChangeTrigger) 
															#line 4578 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorClientDBInfoChangeActorState<MonitorClientDBInfoChangeActor>(cx, clientDBInfo, proxiesChangeTrigger),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8226933834932854784UL, 6929653946600028160UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorClientDBInfoChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> monitorClientDBInfoChange( DatabaseContext* const& cx, Reference<AsyncVar<ClientDBInfo> const> const& clientDBInfo, AsyncTrigger* const& proxiesChangeTrigger ) {
															#line 992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new MonitorClientDBInfoChangeActor(cx, clientDBInfo, proxiesChangeTrigger));
															#line 4611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1034 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

void updateLocationCacheWithCaches(DatabaseContext* self,
                                   const std::map<UID, StorageServerInterface>& removed,
                                   const std::map<UID, StorageServerInterface>& added) {
	// TODO: this needs to be more clever in the future
	auto ranges = self->locationCache.ranges();
	for (auto iter = ranges.begin(); iter != ranges.end(); ++iter) {
		if (iter->value() && iter->value()->hasCaches) {
			auto& val = iter->value();
			std::vector<Reference<ReferencedInterface<StorageServerInterface>>> interfaces;
			interfaces.reserve(val->size() - removed.size() + added.size());
			for (int i = 0; i < val->size(); ++i) {
				const auto& interf = (*val)[i];
				if (removed.count(interf->interf.id()) == 0) {
					interfaces.emplace_back(interf);
				}
			}
			for (const auto& p : added) {
				interfaces.push_back(makeReference<ReferencedInterface<StorageServerInterface>>(p.second));
			}
			iter->value() = makeReference<LocationInfo>(interfaces, true);
		}
	}
}

Reference<LocationInfo> addCaches(const Reference<LocationInfo>& loc,
                                  const std::vector<Reference<ReferencedInterface<StorageServerInterface>>>& other) {
	std::vector<Reference<ReferencedInterface<StorageServerInterface>>> interfaces;
	interfaces.reserve(loc->size() + other.size());
	for (int i = 0; i < loc->size(); ++i) {
		interfaces.emplace_back((*loc)[i]);
	}
	interfaces.insert(interfaces.end(), other.begin(), other.end());
	return makeReference<LocationInfo>(interfaces, true);
}

															#line 4651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via updateCachedRanges()
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class UpdateCachedRangesActor>
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class UpdateCachedRangesActorState {
															#line 4658 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UpdateCachedRangesActorState(DatabaseContext* const& self,std::map<UID, StorageServerInterface>* const& cacheServers) 
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : self(self),
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   cacheServers(cacheServers),
															#line 1071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tr(),
															#line 1072 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   trueValue(storageCacheValue(std::vector<uint16_t>{ 0 })),
															#line 1073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   falseValue(storageCacheValue(std::vector<uint16_t>{}))
															#line 4673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("updateCachedRanges", reinterpret_cast<unsigned long>(this));

	}
	~UpdateCachedRangesActorState() 
	{
		fdb_probe_actor_destroy("updateCachedRanges", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				;
															#line 4689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateCachedRangesActorState();
		static_cast<UpdateCachedRangesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1142 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevError, "UpdateCachedRangesFailed").error(e);
															#line 1143 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1078 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr = Transaction();
															#line 1079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(0);
															#line 1079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4746 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 1;
															#line 1079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 0, Void >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 4751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = brokenPromiseToNever(self->updateCache.onTrigger());
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4762 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 2;
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 1, Void >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 4767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = brokenPromiseToNever(self->updateCache.onTrigger());
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4778 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 2;
															#line 1080 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 1, Void >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 4783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state > 0) static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCachedRangesActor*>(this)->ActorCallback< UpdateCachedRangesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateCachedRangesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr = Transaction(Database(Reference<DatabaseContext>::addRef(self)));
															#line 1083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 4871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		try {
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_2 = tr.getRange(storageCacheKeys, CLIENT_KNOBS->TOO_MANY);
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 3;
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 2, RangeResult >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 4882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1082 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr = Transaction(Database(Reference<DatabaseContext>::addRef(self)));
															#line 1083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1084 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 4901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		try {
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_2 = tr.getRange(storageCacheKeys, CLIENT_KNOBS->TOO_MANY);
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 3;
															#line 1086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 2, RangeResult >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 4912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state > 0) static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCachedRangesActor*>(this)->ActorCallback< UpdateCachedRangesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< UpdateCachedRangesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 5;
															#line 1138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 4, Void >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 5016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4(RangeResult const& range,int loopDepth) 
	{
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(!range.more);
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::vector<Reference<ReferencedInterface<StorageServerInterface>>> cacheInterfaces;
															#line 1089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cacheInterfaces.reserve(cacheServers->size());
															#line 1090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& p : *cacheServers ) {
															#line 1091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cacheInterfaces.push_back(makeReference<ReferencedInterface<StorageServerInterface>>(p.second));
															#line 5039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool currCached = false;
															#line 1094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		KeyRef begin, end;
															#line 1095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& kv : range ) {
															#line 1097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(currCached == (kv.value == falseValue));
															#line 1098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (kv.value == trueValue)
															#line 5051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1099 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				begin = kv.key.substr(storageCacheKeys.begin.size());
															#line 1100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				currCached = true;
															#line 5057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 1102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				currCached = false;
															#line 1103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				end = kv.key.substr(storageCacheKeys.begin.size());
															#line 1104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				KeyRangeRef cachedRange{ begin, end };
															#line 1105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				auto ranges = self->locationCache.containedRanges(cachedRange);
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				KeyRef containedRangesBegin, containedRangesEnd, prevKey;
															#line 1107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (!ranges.empty())
															#line 5073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 1108 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					containedRangesBegin = ranges.begin().range().begin;
															#line 5077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
															#line 1110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				for(auto iter = ranges.begin();iter != ranges.end();++iter) {
															#line 1111 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					containedRangesEnd = iter->range().end;
															#line 1112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (iter->value() && !iter->value()->hasCaches)
															#line 5085 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 1113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						iter->value() = addCaches(iter->value(), cacheInterfaces);
															#line 5089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
				}
															#line 1116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				auto iter = self->locationCache.rangeContaining(begin);
															#line 1117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (iter->value() && !iter->value()->hasCaches)
															#line 5096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 1118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (end >= iter->range().end)
															#line 5100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 1119 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						Key endCopy = iter->range().end;
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						self->locationCache.insert(KeyRangeRef{ begin, endCopy }, addCaches(iter->value(), cacheInterfaces));
															#line 5106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
					else
					{
															#line 1123 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						self->locationCache.insert(KeyRangeRef{ begin, end }, addCaches(iter->value(), cacheInterfaces));
															#line 5112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
				}
															#line 1127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				iter = self->locationCache.rangeContainingKeyBefore(end);
															#line 1128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (iter->value() && !iter->value()->hasCaches)
															#line 5119 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 1129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					Key beginCopy = iter->range().begin;
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					self->locationCache.insert(KeyRangeRef{ beginCopy, end }, addCaches(iter->value(), cacheInterfaces));
															#line 5125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
			}
		}
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(2.0);
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 4;
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 3, Void >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 5138 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(RangeResult && range,int loopDepth) 
	{
															#line 1087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(!range.more);
															#line 1088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::vector<Reference<ReferencedInterface<StorageServerInterface>>> cacheInterfaces;
															#line 1089 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cacheInterfaces.reserve(cacheServers->size());
															#line 1090 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& p : *cacheServers ) {
															#line 1091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cacheInterfaces.push_back(makeReference<ReferencedInterface<StorageServerInterface>>(p.second));
															#line 5155 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool currCached = false;
															#line 1094 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		KeyRef begin, end;
															#line 1095 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& kv : range ) {
															#line 1097 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(currCached == (kv.value == falseValue));
															#line 1098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (kv.value == trueValue)
															#line 5167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1099 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				begin = kv.key.substr(storageCacheKeys.begin.size());
															#line 1100 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				currCached = true;
															#line 5173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 1102 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				currCached = false;
															#line 1103 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				end = kv.key.substr(storageCacheKeys.begin.size());
															#line 1104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				KeyRangeRef cachedRange{ begin, end };
															#line 1105 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				auto ranges = self->locationCache.containedRanges(cachedRange);
															#line 1106 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				KeyRef containedRangesBegin, containedRangesEnd, prevKey;
															#line 1107 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (!ranges.empty())
															#line 5189 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 1108 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					containedRangesBegin = ranges.begin().range().begin;
															#line 5193 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
															#line 1110 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				for(auto iter = ranges.begin();iter != ranges.end();++iter) {
															#line 1111 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					containedRangesEnd = iter->range().end;
															#line 1112 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (iter->value() && !iter->value()->hasCaches)
															#line 5201 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 1113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						iter->value() = addCaches(iter->value(), cacheInterfaces);
															#line 5205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
				}
															#line 1116 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				auto iter = self->locationCache.rangeContaining(begin);
															#line 1117 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (iter->value() && !iter->value()->hasCaches)
															#line 5212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 1118 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (end >= iter->range().end)
															#line 5216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 1119 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						Key endCopy = iter->range().end;
															#line 1120 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						self->locationCache.insert(KeyRangeRef{ begin, endCopy }, addCaches(iter->value(), cacheInterfaces));
															#line 5222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
					else
					{
															#line 1123 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						self->locationCache.insert(KeyRangeRef{ begin, end }, addCaches(iter->value(), cacheInterfaces));
															#line 5228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					}
				}
															#line 1127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				iter = self->locationCache.rangeContainingKeyBefore(end);
															#line 1128 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (iter->value() && !iter->value()->hasCaches)
															#line 5235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 1129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					Key beginCopy = iter->range().begin;
															#line 1130 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					self->locationCache.insert(KeyRangeRef{ beginCopy, end }, addCaches(iter->value(), cacheInterfaces));
															#line 5241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
			}
		}
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(2.0);
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont2Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 4;
															#line 1135 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< UpdateCachedRangesActor, 3, Void >*>(static_cast<UpdateCachedRangesActor*>(this)));
															#line 5254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& range,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(range, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && range,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(range), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state > 0) static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCachedRangesActor*>(this)->ActorCallback< UpdateCachedRangesActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< UpdateCachedRangesActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state > 0) static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCachedRangesActor*>(this)->ActorCallback< UpdateCachedRangesActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< UpdateCachedRangesActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont17(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state > 0) static_cast<UpdateCachedRangesActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateCachedRangesActor*>(this)->ActorCallback< UpdateCachedRangesActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< UpdateCachedRangesActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< UpdateCachedRangesActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* self;
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::map<UID, StorageServerInterface>* cacheServers;
															#line 1071 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction tr;
															#line 1072 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Value trueValue;
															#line 1073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Value falseValue;
															#line 5531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via updateCachedRanges()
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class UpdateCachedRangesActor final : public Actor<Void>, public ActorCallback< UpdateCachedRangesActor, 0, Void >, public ActorCallback< UpdateCachedRangesActor, 1, Void >, public ActorCallback< UpdateCachedRangesActor, 2, RangeResult >, public ActorCallback< UpdateCachedRangesActor, 3, Void >, public ActorCallback< UpdateCachedRangesActor, 4, Void >, public FastAllocated<UpdateCachedRangesActor>, public UpdateCachedRangesActorState<UpdateCachedRangesActor> {
															#line 5536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<UpdateCachedRangesActor>::operator new;
	using FastAllocated<UpdateCachedRangesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5022857381931918592UL, 9168462342376629248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateCachedRangesActor, 0, Void >;
friend struct ActorCallback< UpdateCachedRangesActor, 1, Void >;
friend struct ActorCallback< UpdateCachedRangesActor, 2, RangeResult >;
friend struct ActorCallback< UpdateCachedRangesActor, 3, Void >;
friend struct ActorCallback< UpdateCachedRangesActor, 4, Void >;
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UpdateCachedRangesActor(DatabaseContext* const& self,std::map<UID, StorageServerInterface>* const& cacheServers) 
															#line 5557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateCachedRangesActorState<UpdateCachedRangesActor>(self, cacheServers),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateCachedRanges", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9308568107035644416UL, 14468141351058818816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateCachedRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateCachedRanges");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateCachedRanges", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateCachedRangesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< UpdateCachedRangesActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< UpdateCachedRangesActor, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< UpdateCachedRangesActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< UpdateCachedRangesActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> updateCachedRanges( DatabaseContext* const& self, std::map<UID, StorageServerInterface>* const& cacheServers ) {
															#line 1070 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new UpdateCachedRangesActor(self, cacheServers));
															#line 5594 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1146 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// The reason for getting a pointer to DatabaseContext instead of a reference counted object is because reference
// counting will increment reference count for DatabaseContext which holds the future of this actor. This creates a
// cyclic reference and hence this actor and Database object will not be destroyed at all.
															#line 5602 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorCacheList()
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class MonitorCacheListActor>
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class MonitorCacheListActorState {
															#line 5609 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	MonitorCacheListActorState(DatabaseContext* const& self) 
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : self(self),
															#line 1151 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tr(),
															#line 1152 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   cacheServerMap(),
															#line 1153 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   updateRanges(updateCachedRanges(self, &cacheServerMap)),
															#line 1154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   backoff()
															#line 5624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorCacheList", reinterpret_cast<unsigned long>(this));

	}
	~MonitorCacheListActorState() 
	{
		fdb_probe_actor_destroy("monitorCacheList", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->updateCache.onTrigger();
															#line 1157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5641 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 1;
															#line 1157 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 0, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 5646 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorCacheListActorState();
		static_cast<MonitorCacheListActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 1159 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 5670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 1159 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 5686 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state > 0) static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorCacheListActor*>(this)->ActorCallback< MonitorCacheListActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorCacheListActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevError, "MonitorCacheListFailed").error(e);
															#line 1200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 5779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = refreshTransaction(self, &tr);
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5802 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 2;
															#line 1162 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 1, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 5807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		try {
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_2 = tr.getRange(storageCacheServerKeys, CLIENT_KNOBS->TOO_MANY);
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 5819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 3;
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 2, RangeResult >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 5824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		try {
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<RangeResult> __when_expr_2 = tr.getRange(storageCacheServerKeys, CLIENT_KNOBS->TOO_MANY);
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 5842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
			static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 3;
															#line 1164 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 2, RangeResult >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 5847 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state > 0) static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorCacheListActor*>(this)->ActorCallback< MonitorCacheListActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorCacheListActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 1194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 5;
															#line 1194 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 4, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 5951 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(RangeResult const& cacheList,int loopDepth) 
	{
															#line 1165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(!cacheList.more);
															#line 1166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool hasChanges = false;
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::map<UID, StorageServerInterface> allCacheServers;
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( auto kv : cacheList ) {
															#line 1169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			auto ssi = BinaryReader::fromStringRef<StorageServerInterface>(kv.value, IncludeVersion());
															#line 1170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			allCacheServers.emplace(ssi.id(), ssi);
															#line 5976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::map<UID, StorageServerInterface> newCacheServers;
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::map<UID, StorageServerInterface> deletedCacheServers;
															#line 1174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::set_difference(allCacheServers.begin(), allCacheServers.end(), cacheServerMap.begin(), cacheServerMap.end(), std::insert_iterator<std::map<UID, StorageServerInterface>>( newCacheServers, newCacheServers.begin()));
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::set_difference(cacheServerMap.begin(), cacheServerMap.end(), allCacheServers.begin(), allCacheServers.end(), std::insert_iterator<std::map<UID, StorageServerInterface>>( deletedCacheServers, deletedCacheServers.begin()));
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		hasChanges = !(newCacheServers.empty() && deletedCacheServers.empty());
															#line 1187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (hasChanges)
															#line 5990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			updateLocationCacheWithCaches(self, deletedCacheServers, newCacheServers);
															#line 5994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cacheServerMap = std::move(allCacheServers);
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(5.0);
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 6002 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 4;
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 3, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 6007 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(RangeResult && cacheList,int loopDepth) 
	{
															#line 1165 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(!cacheList.more);
															#line 1166 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool hasChanges = false;
															#line 1167 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::map<UID, StorageServerInterface> allCacheServers;
															#line 1168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( auto kv : cacheList ) {
															#line 1169 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			auto ssi = BinaryReader::fromStringRef<StorageServerInterface>(kv.value, IncludeVersion());
															#line 1170 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			allCacheServers.emplace(ssi.id(), ssi);
															#line 6026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::map<UID, StorageServerInterface> newCacheServers;
															#line 1173 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::map<UID, StorageServerInterface> deletedCacheServers;
															#line 1174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::set_difference(allCacheServers.begin(), allCacheServers.end(), cacheServerMap.begin(), cacheServerMap.end(), std::insert_iterator<std::map<UID, StorageServerInterface>>( newCacheServers, newCacheServers.begin()));
															#line 1180 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::set_difference(cacheServerMap.begin(), cacheServerMap.end(), allCacheServers.begin(), allCacheServers.end(), std::insert_iterator<std::map<UID, StorageServerInterface>>( deletedCacheServers, deletedCacheServers.begin()));
															#line 1186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		hasChanges = !(newCacheServers.empty() && deletedCacheServers.empty());
															#line 1187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (hasChanges)
															#line 6040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1188 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			updateLocationCacheWithCaches(self, deletedCacheServers, newCacheServers);
															#line 6044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1190 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cacheServerMap = std::move(allCacheServers);
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(5.0);
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 6052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 4;
															#line 1191 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 3, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 6057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(RangeResult const& cacheList,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(cacheList, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(RangeResult && cacheList,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(cacheList), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state > 0) static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorCacheListActor*>(this)->ActorCallback< MonitorCacheListActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MonitorCacheListActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		backoff = Backoff();
															#line 6141 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		backoff = Backoff();
															#line 6150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state > 0) static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorCacheListActor*>(this)->ActorCallback< MonitorCacheListActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MonitorCacheListActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = backoff.onError();
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6249 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1Catch1cont1when1(__when_expr_5.get(), loopDepth); };
		static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 6;
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 5, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 6254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_5 = backoff.onError();
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6265 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1Catch1cont1when1(__when_expr_5.get(), loopDepth); };
		static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 6;
															#line 1195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MonitorCacheListActor, 5, Void >*>(static_cast<MonitorCacheListActor*>(this)));
															#line 6270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state > 0) static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorCacheListActor*>(this)->ActorCallback< MonitorCacheListActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MonitorCacheListActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont1Catch1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MonitorCacheListActor*>(this)->actor_wait_state > 0) static_cast<MonitorCacheListActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorCacheListActor*>(this)->ActorCallback< MonitorCacheListActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont1Catch1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MonitorCacheListActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont1Catch1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MonitorCacheListActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* self;
															#line 1151 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction tr;
															#line 1152 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::map<UID, StorageServerInterface> cacheServerMap;
															#line 1153 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Future<Void> updateRanges;
															#line 1154 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Backoff backoff;
															#line 6447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via monitorCacheList()
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class MonitorCacheListActor final : public Actor<Void>, public ActorCallback< MonitorCacheListActor, 0, Void >, public ActorCallback< MonitorCacheListActor, 1, Void >, public ActorCallback< MonitorCacheListActor, 2, RangeResult >, public ActorCallback< MonitorCacheListActor, 3, Void >, public ActorCallback< MonitorCacheListActor, 4, Void >, public ActorCallback< MonitorCacheListActor, 5, Void >, public FastAllocated<MonitorCacheListActor>, public MonitorCacheListActorState<MonitorCacheListActor> {
															#line 6452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<MonitorCacheListActor>::operator new;
	using FastAllocated<MonitorCacheListActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8910420353490767616UL, 14095367643551329280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorCacheListActor, 0, Void >;
friend struct ActorCallback< MonitorCacheListActor, 1, Void >;
friend struct ActorCallback< MonitorCacheListActor, 2, RangeResult >;
friend struct ActorCallback< MonitorCacheListActor, 3, Void >;
friend struct ActorCallback< MonitorCacheListActor, 4, Void >;
friend struct ActorCallback< MonitorCacheListActor, 5, Void >;
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	MonitorCacheListActor(DatabaseContext* const& self) 
															#line 6474 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorCacheListActorState<MonitorCacheListActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorCacheList", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7862087477804658688UL, 12761429796255660032UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorCacheListActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorCacheList");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorCacheList", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorCacheListActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MonitorCacheListActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MonitorCacheListActor, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MonitorCacheListActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MonitorCacheListActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MonitorCacheListActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> monitorCacheList( DatabaseContext* const& self ) {
															#line 1150 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new MonitorCacheListActor(self));
															#line 6512 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 6517 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via handleTssMismatches()
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class HandleTssMismatchesActor>
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class HandleTssMismatchesActorState {
															#line 6524 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	HandleTssMismatchesActorState(DatabaseContext* const& cx) 
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 1205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tr(),
															#line 1206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tssMapDB(KeyBackedMap<UID, UID>(tssMappingKeys.begin)),
															#line 1207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tssMismatchDB(KeyBackedMap<Tuple, std::string>(tssMismatchKeys.begin))
															#line 6537 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("handleTssMismatches", reinterpret_cast<unsigned long>(this));

	}
	~HandleTssMismatchesActorState() 
	{
		fdb_probe_actor_destroy("handleTssMismatches", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 6552 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleTssMismatchesActorState();
		static_cast<HandleTssMismatchesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		auto __when_expr_0 = cx->tssMismatchStream.getFuture();
															#line 6582 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<std::pair<UID, std::vector<DetailedTSSMismatch>>>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<std::pair<UID, std::vector<DetailedTSSMismatch>>>>::value, "invalid type");
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
		static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 1;
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 6591 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6602 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 2;
															#line 1212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleTssMismatchesActor, 1, Void >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 6607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(std::pair<UID, std::vector<DetailedTSSMismatch>> const& __data,int loopDepth) 
	{
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		data = __data;
															#line 6616 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::pair<UID, std::vector<DetailedTSSMismatch>> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >::remove();

	}
	void a_callback_fire(ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >*,std::pair<UID, std::vector<DetailedTSSMismatch>> const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >*,std::pair<UID, std::vector<DetailedTSSMismatch>> && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tssPairID = UID();
															#line 1215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool found = false;
															#line 1216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& it : cx->tssMapping ) {
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second.id() == data.first)
															#line 6701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tssPairID = it.first;
															#line 1219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				found = true;
															#line 6707 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				break;
			}
		}
															#line 1223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (found)
															#line 6713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			quarantine = CLIENT_KNOBS->QUARANTINE_TSS_ON_MISMATCH;
															#line 1225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevWarnAlways, quarantine ? "TSS_QuarantineMismatch" : "TSS_KillMismatch") .detail("TSSID", data.first.toString());
															#line 1227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(quarantine, "Quarantining TSS because it got mismatch");
															#line 1228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(!quarantine, "Killing TSS because it got mismatch");
															#line 1230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr = makeReference<ReadYourWritesTransaction>(Database(Reference<DatabaseContext>::addRef(cx)));
															#line 1231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tries = 0;
															#line 1232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 6729 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);
		}
		else
		{
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(true, "Not handling TSS with mismatch because it's already gone");
															#line 6736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tssPairID = UID();
															#line 1215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		bool found = false;
															#line 1216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( const auto& it : cx->tssMapping ) {
															#line 1217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (it.second.id() == data.first)
															#line 6752 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tssPairID = it.first;
															#line 1219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				found = true;
															#line 6758 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				break;
			}
		}
															#line 1223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (found)
															#line 6764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			quarantine = CLIENT_KNOBS->QUARANTINE_TSS_ON_MISMATCH;
															#line 1225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevWarnAlways, quarantine ? "TSS_QuarantineMismatch" : "TSS_KillMismatch") .detail("TSSID", data.first.toString());
															#line 1227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(quarantine, "Quarantining TSS because it got mismatch");
															#line 1228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(!quarantine, "Killing TSS because it got mismatch");
															#line 1230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr = makeReference<ReadYourWritesTransaction>(Database(Reference<DatabaseContext>::addRef(cx)));
															#line 1231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tries = 0;
															#line 1232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 6780 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);
		}
		else
		{
															#line 1266 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(true, "Not handling TSS with mismatch because it's already gone");
															#line 6787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorCallback< HandleTssMismatchesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleTssMismatchesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 1264 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>();
															#line 6878 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1235 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (quarantine)
															#line 6899 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 1237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tr->set(tssQuarantineKeyFor(data.first), ""_sr);
															#line 6903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 1239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tr->clear(serverTagKeyFor(data.first));
															#line 6909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 1241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tssMapDB.erase(tr, tssPairID);
															#line 1243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			for( const DetailedTSSMismatch& d : data.second ) {
															#line 1245 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tssMismatchDB.set(tr, Tuple::makeTuple(data.first.toString(), d.timestamp, d.mismatchId.toString()), d.traceString);
															#line 6917 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 1250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 1250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 3;
															#line 1250 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< HandleTssMismatchesActor, 2, Void >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 6928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1256 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tries++;
															#line 1257 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (tries > 10)
															#line 6958 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1259 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent("TSS_MismatchGaveUp").detail("TSSID", data.first.toString());
															#line 6962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr->onError(e);
															#line 1254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 6976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 4;
															#line 1254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< HandleTssMismatchesActor, 3, Void >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 6981 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorCallback< HandleTssMismatchesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< HandleTssMismatchesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont2loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorCallback< HandleTssMismatchesActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< HandleTssMismatchesActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 1205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyBackedMap<UID, UID> tssMapDB;
															#line 1207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyBackedMap<Tuple, std::string> tssMismatchDB;
															#line 1210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::pair<UID, std::vector<DetailedTSSMismatch>> data;
															#line 1214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UID tssPairID;
															#line 1224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	bool quarantine;
															#line 1231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int tries;
															#line 7182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via handleTssMismatches()
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class HandleTssMismatchesActor final : public Actor<Void>, public ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >, public ActorCallback< HandleTssMismatchesActor, 1, Void >, public ActorCallback< HandleTssMismatchesActor, 2, Void >, public ActorCallback< HandleTssMismatchesActor, 3, Void >, public FastAllocated<HandleTssMismatchesActor>, public HandleTssMismatchesActorState<HandleTssMismatchesActor> {
															#line 7187 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<HandleTssMismatchesActor>::operator new;
	using FastAllocated<HandleTssMismatchesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11072762611816720640UL, 15063571074838241280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >;
friend struct ActorCallback< HandleTssMismatchesActor, 1, Void >;
friend struct ActorCallback< HandleTssMismatchesActor, 2, Void >;
friend struct ActorCallback< HandleTssMismatchesActor, 3, Void >;
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	HandleTssMismatchesActor(DatabaseContext* const& cx) 
															#line 7207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   HandleTssMismatchesActorState<HandleTssMismatchesActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6231205013535006464UL, 15374262790278653696UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleTssMismatches");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< HandleTssMismatchesActor, 0, std::pair<UID, std::vector<DetailedTSSMismatch>> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleTssMismatchesActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< HandleTssMismatchesActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< HandleTssMismatchesActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> handleTssMismatches( DatabaseContext* const& cx ) {
															#line 1204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new HandleTssMismatchesActor(cx));
															#line 7243 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1270 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 7248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via backgroundGrvUpdater()
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class BackgroundGrvUpdaterActor>
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class BackgroundGrvUpdaterActorState {
															#line 7255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	BackgroundGrvUpdaterActorState(DatabaseContext* const& cx) 
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 1272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tr(),
															#line 1273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   grvDelay(0.001),
															#line 1274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   backoff()
															#line 7268 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this));

	}
	~BackgroundGrvUpdaterActorState() 
	{
		fdb_probe_actor_destroy("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1276 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				;
															#line 7284 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~BackgroundGrvUpdaterActorState();
		static_cast<BackgroundGrvUpdaterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "BackgroundGrvUpdaterFailed").errorUnsuppressed(e);
															#line 1317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1277 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (CLIENT_KNOBS->FORCE_GRV_CACHE_OFF)
															#line 7337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1278 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<BackgroundGrvUpdaterActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~BackgroundGrvUpdaterActorState(); static_cast<BackgroundGrvUpdaterActor*>(this)->destroy(); return 0; }
															#line 7341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<BackgroundGrvUpdaterActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~BackgroundGrvUpdaterActorState();
			static_cast<BackgroundGrvUpdaterActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = refreshTransaction(cx, &tr);
															#line 1279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7351 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 1;
															#line 1279 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 0, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		curTime = now();
															#line 1281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastTime = cx->getLastGrvTime();
															#line 1282 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastProxyTime = cx->lastProxyRequestTime;
															#line 1283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		TraceEvent(SevDebug, "BackgroundGrvUpdaterBefore") .detail("CurTime", curTime) .detail("LastTime", lastTime) .detail("GrvDelay", grvDelay) .detail("CachedReadVersion", cx->getCachedReadVersion()) .detail("CachedTime", cx->getLastGrvTime()) .detail("Gap", curTime - lastTime) .detail("Bound", CLIENT_KNOBS->MAX_VERSION_CACHE_LAG - grvDelay);
															#line 1291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (curTime - lastTime >= (CLIENT_KNOBS->MAX_VERSION_CACHE_LAG - grvDelay) || curTime - lastProxyTime > CLIENT_KNOBS->MAX_PROXY_CONTACT_LAG)
															#line 7373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			try {
															#line 1294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tr.setOption(FDBTransactionOptions::SKIP_GRV_CACHE);
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_1 = success(tr.getReadVersion());
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7382 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 2;
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 1, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(std::max(0.001, std::min(CLIENT_KNOBS->MAX_PROXY_CONTACT_LAG - (curTime - lastProxyTime), (CLIENT_KNOBS->MAX_VERSION_CACHE_LAG - grvDelay) - (curTime - lastTime))));
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when2(__when_expr_4.get(), loopDepth); };
			static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 5;
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 4, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		curTime = now();
															#line 1281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastTime = cx->getLastGrvTime();
															#line 1282 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		lastProxyTime = cx->lastProxyRequestTime;
															#line 1283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		TraceEvent(SevDebug, "BackgroundGrvUpdaterBefore") .detail("CurTime", curTime) .detail("LastTime", lastTime) .detail("GrvDelay", grvDelay) .detail("CachedReadVersion", cx->getCachedReadVersion()) .detail("CachedTime", cx->getLastGrvTime()) .detail("Gap", curTime - lastTime) .detail("Bound", CLIENT_KNOBS->MAX_VERSION_CACHE_LAG - grvDelay);
															#line 1291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (curTime - lastTime >= (CLIENT_KNOBS->MAX_VERSION_CACHE_LAG - grvDelay) || curTime - lastProxyTime > CLIENT_KNOBS->MAX_PROXY_CONTACT_LAG)
															#line 7425 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			try {
															#line 1294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				tr.setOption(FDBTransactionOptions::SKIP_GRV_CACHE);
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_1 = success(tr.getReadVersion());
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7434 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 2;
															#line 1295 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 1, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7439 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(std::max(0.001, std::min(CLIENT_KNOBS->MAX_PROXY_CONTACT_LAG - (curTime - lastProxyTime), (CLIENT_KNOBS->MAX_VERSION_CACHE_LAG - grvDelay) - (curTime - lastTime))));
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when2(__when_expr_4.get(), loopDepth); };
			static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 5;
															#line 1309 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 4, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state > 0) static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 0;
		static_cast<BackgroundGrvUpdaterActor*>(this)->ActorCallback< BackgroundGrvUpdaterActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< BackgroundGrvUpdaterActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent(SevInfo, "BackgroundGrvUpdaterTxnError").errorUnsuppressed(e);
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr.onError(e);
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 3;
															#line 1305 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 2, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7566 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->lastProxyRequestTime = curTime;
															#line 1297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		grvDelay = (grvDelay + (now() - curTime)) / 2.0;
															#line 1298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		TraceEvent(SevDebug, "BackgroundGrvUpdaterSuccess") .detail("GrvDelay", grvDelay) .detail("CachedReadVersion", cx->getCachedReadVersion()) .detail("CachedTime", cx->getLastGrvTime());
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		backoff = Backoff();
															#line 7587 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->lastProxyRequestTime = curTime;
															#line 1297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		grvDelay = (grvDelay + (now() - curTime)) / 2.0;
															#line 1298 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		TraceEvent(SevDebug, "BackgroundGrvUpdaterSuccess") .detail("GrvDelay", grvDelay) .detail("CachedReadVersion", cx->getCachedReadVersion()) .detail("CachedTime", cx->getLastGrvTime());
															#line 1302 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		backoff = Backoff();
															#line 7602 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state > 0) static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 0;
		static_cast<BackgroundGrvUpdaterActor*>(this)->ActorCallback< BackgroundGrvUpdaterActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< BackgroundGrvUpdaterActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = backoff.onError();
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7701 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 4;
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 3, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7706 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = backoff.onError();
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 4;
															#line 1306 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< BackgroundGrvUpdaterActor, 3, Void >*>(static_cast<BackgroundGrvUpdaterActor*>(this)));
															#line 7722 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state > 0) static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 0;
		static_cast<BackgroundGrvUpdaterActor*>(this)->ActorCallback< BackgroundGrvUpdaterActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< BackgroundGrvUpdaterActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont1Catch1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1Catch1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state > 0) static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 0;
		static_cast<BackgroundGrvUpdaterActor*>(this)->ActorCallback< BackgroundGrvUpdaterActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont1Catch1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< BackgroundGrvUpdaterActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state > 0) static_cast<BackgroundGrvUpdaterActor*>(this)->actor_wait_state = 0;
		static_cast<BackgroundGrvUpdaterActor*>(this)->ActorCallback< BackgroundGrvUpdaterActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< BackgroundGrvUpdaterActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< BackgroundGrvUpdaterActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 1272 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction tr;
															#line 1273 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double grvDelay;
															#line 1274 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Backoff backoff;
															#line 1280 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double curTime;
															#line 1281 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double lastTime;
															#line 1282 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double lastProxyTime;
															#line 7990 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via backgroundGrvUpdater()
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class BackgroundGrvUpdaterActor final : public Actor<Void>, public ActorCallback< BackgroundGrvUpdaterActor, 0, Void >, public ActorCallback< BackgroundGrvUpdaterActor, 1, Void >, public ActorCallback< BackgroundGrvUpdaterActor, 2, Void >, public ActorCallback< BackgroundGrvUpdaterActor, 3, Void >, public ActorCallback< BackgroundGrvUpdaterActor, 4, Void >, public FastAllocated<BackgroundGrvUpdaterActor>, public BackgroundGrvUpdaterActorState<BackgroundGrvUpdaterActor> {
															#line 7995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<BackgroundGrvUpdaterActor>::operator new;
	using FastAllocated<BackgroundGrvUpdaterActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5305836225764766464UL, 15263879326927881472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< BackgroundGrvUpdaterActor, 0, Void >;
friend struct ActorCallback< BackgroundGrvUpdaterActor, 1, Void >;
friend struct ActorCallback< BackgroundGrvUpdaterActor, 2, Void >;
friend struct ActorCallback< BackgroundGrvUpdaterActor, 3, Void >;
friend struct ActorCallback< BackgroundGrvUpdaterActor, 4, Void >;
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	BackgroundGrvUpdaterActor(DatabaseContext* const& cx) 
															#line 8016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   BackgroundGrvUpdaterActorState<BackgroundGrvUpdaterActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14021273295369828864UL, 1677349736430853888UL);
		ActorExecutionContextHelper __helper(static_cast<BackgroundGrvUpdaterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("backgroundGrvUpdater");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("backgroundGrvUpdater", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< BackgroundGrvUpdaterActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< BackgroundGrvUpdaterActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< BackgroundGrvUpdaterActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< BackgroundGrvUpdaterActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< BackgroundGrvUpdaterActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> backgroundGrvUpdater( DatabaseContext* const& cx ) {
															#line 1271 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new BackgroundGrvUpdaterActor(cx));
															#line 8053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1320 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

inline HealthMetrics populateHealthMetrics(const HealthMetrics& detailedMetrics, bool detailedOutput) {
	if (detailedOutput) {
		return detailedMetrics;
	} else {
		HealthMetrics result;
		result.update(detailedMetrics, false, false);
		return result;
	}
}

															#line 8068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getHealthMetricsActor()
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetHealthMetricsActorActor>
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetHealthMetricsActorActorState {
															#line 8075 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetHealthMetricsActorActorState(DatabaseContext* const& cx,bool const& detailed,bool const& sendDetailedRequest) 
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   detailed(detailed),
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   sendDetailedRequest(sendDetailedRequest)
															#line 8086 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getHealthMetricsActor", reinterpret_cast<unsigned long>(this));

	}
	~GetHealthMetricsActorActorState() 
	{
		fdb_probe_actor_destroy("getHealthMetricsActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 8101 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetHealthMetricsActorActorState();
		static_cast<GetHealthMetricsActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = cx->onProxiesChanged();
															#line 1333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8133 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 1335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetHealthMetricsReply> __when_expr_1 = basicLoadBalance(cx->getGrvProxies(UseProvisionalProxies::False), &GrvProxyInterface::getHealthMetrics, GetHealthMetricsRequest(sendDetailedRequest));
															#line 8137 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state = 1;
															#line 1334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetHealthMetricsActorActor, 0, Void >*>(static_cast<GetHealthMetricsActorActor*>(this)));
															#line 1335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*>(static_cast<GetHealthMetricsActorActor*>(this)));
															#line 8144 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(GetHealthMetricsReply const& rep,int loopDepth) 
	{
															#line 1338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->healthMetrics.update(rep.healthMetrics, sendDetailedRequest, true);
															#line 1339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->healthMetricsLastUpdated = now();
															#line 1340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (sendDetailedRequest)
															#line 8175 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->detailedHealthMetricsLastUpdated = now();
															#line 8179 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetHealthMetricsActorActor*>(this)->SAV<HealthMetrics>::futures) { (void)(populateHealthMetrics(cx->healthMetrics, detailed)); this->~GetHealthMetricsActorActorState(); static_cast<GetHealthMetricsActorActor*>(this)->destroy(); return 0; }
															#line 8183 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetHealthMetricsActorActor*>(this)->SAV< HealthMetrics >::value()) HealthMetrics(populateHealthMetrics(cx->healthMetrics, detailed));
		this->~GetHealthMetricsActorActorState();
		static_cast<GetHealthMetricsActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(GetHealthMetricsReply && rep,int loopDepth) 
	{
															#line 1338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->healthMetrics.update(rep.healthMetrics, sendDetailedRequest, true);
															#line 1339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->healthMetricsLastUpdated = now();
															#line 1340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (sendDetailedRequest)
															#line 8199 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 1341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->detailedHealthMetricsLastUpdated = now();
															#line 8203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 1343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetHealthMetricsActorActor*>(this)->SAV<HealthMetrics>::futures) { (void)(populateHealthMetrics(cx->healthMetrics, detailed)); this->~GetHealthMetricsActorActorState(); static_cast<GetHealthMetricsActorActor*>(this)->destroy(); return 0; }
															#line 8207 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetHealthMetricsActorActor*>(this)->SAV< HealthMetrics >::value()) HealthMetrics(populateHealthMetrics(cx->healthMetrics, detailed));
		this->~GetHealthMetricsActorActorState();
		static_cast<GetHealthMetricsActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state > 0) static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state = 0;
		static_cast<GetHealthMetricsActorActor*>(this)->ActorCallback< GetHealthMetricsActorActor, 0, Void >::remove();
		static_cast<GetHealthMetricsActorActor*>(this)->ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetHealthMetricsActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*,GetHealthMetricsReply const& value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*,GetHealthMetricsReply && value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*,Error err) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	bool detailed;
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	bool sendDetailedRequest;
															#line 8342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getHealthMetricsActor()
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetHealthMetricsActorActor final : public Actor<HealthMetrics>, public ActorCallback< GetHealthMetricsActorActor, 0, Void >, public ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >, public FastAllocated<GetHealthMetricsActorActor>, public GetHealthMetricsActorActorState<GetHealthMetricsActorActor> {
															#line 8347 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetHealthMetricsActorActor>::operator new;
	using FastAllocated<GetHealthMetricsActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15581619752514267136UL, 8658678672999153152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<HealthMetrics>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetHealthMetricsActorActor, 0, Void >;
friend struct ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >;
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetHealthMetricsActorActor(DatabaseContext* const& cx,bool const& detailed,bool const& sendDetailedRequest) 
															#line 8365 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<HealthMetrics>(),
		   GetHealthMetricsActorActorState<GetHealthMetricsActorActor>(cx, detailed, sendDetailedRequest),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13925261518677660672UL, 14865277337925754624UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getHealthMetricsActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetHealthMetricsActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<HealthMetrics> getHealthMetricsActor( DatabaseContext* const& cx, bool const& detailed, bool const& sendDetailedRequest ) {
															#line 1331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<HealthMetrics>(new GetHealthMetricsActorActor(cx, detailed, sendDetailedRequest));
															#line 8398 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1348 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

Future<HealthMetrics> DatabaseContext::getHealthMetrics(bool detailed = false) {
	if (now() - healthMetricsLastUpdated < CLIENT_KNOBS->AGGREGATE_HEALTH_METRICS_MAX_STALENESS) {
		return populateHealthMetrics(healthMetrics, detailed);
	}
	bool sendDetailedRequest =
	    detailed && now() - detailedHealthMetricsLastUpdated > CLIENT_KNOBS->DETAILED_HEALTH_METRICS_MAX_STALENESS;
	return getHealthMetricsActor(this, detailed, sendDetailedRequest);
}

Future<Optional<HealthMetrics::StorageStats>> DatabaseContext::getStorageStats(const UID& id, double maxStaleness) {
	if (now() - detailedHealthMetricsLastUpdated < maxStaleness) {
		auto it = healthMetrics.storageStats.find(id);
		return it == healthMetrics.storageStats.end() ? Optional<HealthMetrics::StorageStats>() : it->second;
	}

	return map(getHealthMetricsActor(this, true, true), [&id](auto metrics) -> Optional<HealthMetrics::StorageStats> {
		auto it = metrics.storageStats.find(id);
		return it == metrics.storageStats.end() ? Optional<HealthMetrics::StorageStats>() : it->second;
	});
}

// register a special key(s) implementation under the specified module
void DatabaseContext::registerSpecialKeysImpl(SpecialKeySpace::MODULE module,
                                              SpecialKeySpace::IMPLTYPE type,
                                              std::unique_ptr<SpecialKeyRangeReadImpl>&& impl,
                                              int deprecatedVersion) {
	// if deprecated, add the implementation when the api version is less than the deprecated version
	if (deprecatedVersion == -1 || apiVersion.version() < deprecatedVersion) {
		specialKeySpace->registerKeyRange(module, type, impl->getKeyRange(), impl.get());
		specialKeySpaceModules.push_back(std::move(impl));
	}
}

															#line 8436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
[[nodiscard]] Future<RangeResult> getWorkerInterfaces( Reference<IClusterConnectionRecord> const& clusterRecord );

#line 1383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 8440 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
[[nodiscard]] Future<Optional<Value>> getJSON( Database const& db, std::string const& jsonField = "" );

#line 1384 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

struct SingleSpecialKeyImpl : SpecialKeyRangeReadImpl {
	Future<RangeResult> getRange(ReadYourWritesTransaction* ryw,
	                             KeyRangeRef kr,
	                             GetRangeLimits limitsHint) const override {
		ASSERT(kr.contains(k));
		return map(f(ryw), [k = k](Optional<Value> v) {
			RangeResult result;
			if (v.present()) {
				result.push_back_deep(result.arena(), KeyValueRef(k, v.get()));
			}
			return result;
		});
	}

	SingleSpecialKeyImpl(KeyRef k,
	                     const std::function<Future<Optional<Value>>(ReadYourWritesTransaction*)>& f,
	                     bool supportsTenants = false)
	  : SpecialKeyRangeReadImpl(singleKeyRange(k)), k(k), f(f), tenantSupport(supportsTenants) {}

	bool supportsTenants() const override {
		CODE_PROBE(tenantSupport, "Single special key in tenant");
		return tenantSupport;
	};

private:
	Key k;
	std::function<Future<Optional<Value>>(ReadYourWritesTransaction*)> f;
	bool tenantSupport;
};

class HealthMetricsRangeImpl : public SpecialKeyRangeAsyncImpl {
public:
	explicit HealthMetricsRangeImpl(KeyRangeRef kr);
	Future<RangeResult> getRange(ReadYourWritesTransaction* ryw,
	                             KeyRangeRef kr,
	                             GetRangeLimits limitsHint) const override;
};

static RangeResult healthMetricsToKVPairs(const HealthMetrics& metrics, KeyRangeRef kr) {
	RangeResult result;
	if (CLIENT_BUGGIFY)
		return result;
	if (kr.contains("\xff\xff/metrics/health/aggregate"_sr) && metrics.worstStorageDurabilityLag != 0) {
		json_spirit::mObject statsObj;
		statsObj["batch_limited"] = metrics.batchLimited;
		statsObj["tps_limit"] = metrics.tpsLimit;
		statsObj["worst_storage_durability_lag"] = metrics.worstStorageDurabilityLag;
		statsObj["limiting_storage_durability_lag"] = metrics.limitingStorageDurabilityLag;
		statsObj["worst_storage_queue"] = metrics.worstStorageQueue;
		statsObj["limiting_storage_queue"] = metrics.limitingStorageQueue;
		statsObj["worst_log_queue"] = metrics.worstTLogQueue;
		std::string statsString =
		    json_spirit::write_string(json_spirit::mValue(statsObj), json_spirit::Output_options::raw_utf8);
		ValueRef bytes(result.arena(), statsString);
		result.push_back(result.arena(), KeyValueRef("\xff\xff/metrics/health/aggregate"_sr, bytes));
	}
	// tlog stats
	{
		int phase = 0; // Avoid comparing twice per loop iteration
		for (const auto& [uid, logStats] : metrics.tLogQueue) {
			StringRef k{ StringRef(uid.toString()).withPrefix("\xff\xff/metrics/health/log/"_sr, result.arena()) };
			if (phase == 0 && k >= kr.begin) {
				phase = 1;
			}
			if (phase == 1) {
				if (k < kr.end) {
					json_spirit::mObject statsObj;
					statsObj["log_queue"] = logStats;
					std::string statsString =
					    json_spirit::write_string(json_spirit::mValue(statsObj), json_spirit::Output_options::raw_utf8);
					ValueRef bytes(result.arena(), statsString);
					result.push_back(result.arena(), KeyValueRef(k, bytes));
				} else {
					break;
				}
			}
		}
	}
	// Storage stats
	{
		int phase = 0; // Avoid comparing twice per loop iteration
		for (const auto& [uid, storageStats] : metrics.storageStats) {
			StringRef k{ StringRef(uid.toString()).withPrefix("\xff\xff/metrics/health/storage/"_sr, result.arena()) };
			if (phase == 0 && k >= kr.begin) {
				phase = 1;
			}
			if (phase == 1) {
				if (k < kr.end) {
					json_spirit::mObject statsObj;
					statsObj["storage_durability_lag"] = storageStats.storageDurabilityLag;
					statsObj["storage_queue"] = storageStats.storageQueue;
					statsObj["cpu_usage"] = storageStats.cpuUsage;
					statsObj["disk_usage"] = storageStats.diskUsage;
					std::string statsString =
					    json_spirit::write_string(json_spirit::mValue(statsObj), json_spirit::Output_options::raw_utf8);
					ValueRef bytes(result.arena(), statsString);
					result.push_back(result.arena(), KeyValueRef(k, bytes));
				} else {
					break;
				}
			}
		}
	}
	return result;
}

															#line 8551 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via healthMetricsGetRangeActor()
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class HealthMetricsGetRangeActorActor>
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class HealthMetricsGetRangeActorActorState {
															#line 8558 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	HealthMetricsGetRangeActorActorState(ReadYourWritesTransaction* const& ryw,KeyRangeRef const& kr) 
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : ryw(ryw),
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   kr(kr)
															#line 8567 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this));

	}
	~HealthMetricsGetRangeActorActorState() 
	{
		fdb_probe_actor_destroy("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<HealthMetrics> __when_expr_0 = ryw->getDatabase()->getHealthMetrics( kr.intersects( KeyRangeRef("\xff\xff/metrics/health/storage/"_sr, "\xff\xff/metrics/health/storage0"_sr)) || kr.intersects(KeyRangeRef("\xff\xff/metrics/health/log/"_sr, "\xff\xff/metrics/health/log0"_sr)));
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8584 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state = 1;
															#line 1492 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*>(static_cast<HealthMetricsGetRangeActorActor*>(this)));
															#line 8589 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HealthMetricsGetRangeActorActorState();
		static_cast<HealthMetricsGetRangeActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(HealthMetrics const& metrics,int loopDepth) 
	{
															#line 1496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV<RangeResult>::futures) { (void)(healthMetricsToKVPairs(metrics, kr)); this->~HealthMetricsGetRangeActorActorState(); static_cast<HealthMetricsGetRangeActorActor*>(this)->destroy(); return 0; }
															#line 8612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV< RangeResult >::value()) RangeResult(healthMetricsToKVPairs(metrics, kr));
		this->~HealthMetricsGetRangeActorActorState();
		static_cast<HealthMetricsGetRangeActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(HealthMetrics && metrics,int loopDepth) 
	{
															#line 1496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV<RangeResult>::futures) { (void)(healthMetricsToKVPairs(metrics, kr)); this->~HealthMetricsGetRangeActorActorState(); static_cast<HealthMetricsGetRangeActorActor*>(this)->destroy(); return 0; }
															#line 8624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV< RangeResult >::value()) RangeResult(healthMetricsToKVPairs(metrics, kr));
		this->~HealthMetricsGetRangeActorActorState();
		static_cast<HealthMetricsGetRangeActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(HealthMetrics const& metrics,int loopDepth) 
	{
		loopDepth = a_body1cont1(metrics, loopDepth);

		return loopDepth;
	}
	int a_body1when1(HealthMetrics && metrics,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(metrics), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state > 0) static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state = 0;
		static_cast<HealthMetricsGetRangeActorActor*>(this)->ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >::remove();

	}
	void a_callback_fire(ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*,HealthMetrics const& value) 
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*,HealthMetrics && value) 
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*,Error err) 
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	ReadYourWritesTransaction* ryw;
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyRangeRef kr;
															#line 8711 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via healthMetricsGetRangeActor()
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class HealthMetricsGetRangeActorActor final : public Actor<RangeResult>, public ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >, public FastAllocated<HealthMetricsGetRangeActorActor>, public HealthMetricsGetRangeActorActorState<HealthMetricsGetRangeActorActor> {
															#line 8716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<HealthMetricsGetRangeActorActor>::operator new;
	using FastAllocated<HealthMetricsGetRangeActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3894731485281919488UL, 17751967074142897408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<RangeResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >;
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	HealthMetricsGetRangeActorActor(ReadYourWritesTransaction* const& ryw,KeyRangeRef const& kr) 
															#line 8733 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<RangeResult>(),
		   HealthMetricsGetRangeActorActorState<HealthMetricsGetRangeActorActor>(ryw, kr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6540468110473879552UL, 10808560399808058880UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("healthMetricsGetRangeActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<RangeResult> healthMetricsGetRangeActor( ReadYourWritesTransaction* const& ryw, KeyRangeRef const& kr ) {
															#line 1491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<RangeResult>(new HealthMetricsGetRangeActorActor(ryw, kr));
															#line 8766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

HealthMetricsRangeImpl::HealthMetricsRangeImpl(KeyRangeRef kr) : SpecialKeyRangeAsyncImpl(kr) {}

Future<RangeResult> HealthMetricsRangeImpl::getRange(ReadYourWritesTransaction* ryw,
                                                     KeyRangeRef kr,
                                                     GetRangeLimits limitsHint) const {
	return healthMetricsGetRangeActor(ryw, kr);
}

															#line 8779 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getClusterId()
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetClusterIdActor>
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetClusterIdActorState {
															#line 8786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetClusterIdActorState(Database const& db) 
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : db(db)
															#line 8793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getClusterId", reinterpret_cast<unsigned long>(this));

	}
	~GetClusterIdActorState() 
	{
		fdb_probe_actor_destroy("getClusterId", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 8808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetClusterIdActorState();
		static_cast<GetClusterIdActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1511 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetClusterIdActor*>(this)->SAV<UID>::futures) { (void)(db->clientInfo->get().clusterId); this->~GetClusterIdActorState(); static_cast<GetClusterIdActor*>(this)->destroy(); return 0; }
															#line 8831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetClusterIdActor*>(this)->SAV< UID >::value()) UID(db->clientInfo->get().clusterId);
		this->~GetClusterIdActorState();
		static_cast<GetClusterIdActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!(!db->clientInfo->get().clusterId.isValid()))
															#line 8850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->clientInfo->onChange();
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetClusterIdActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetClusterIdActor*>(this)->actor_wait_state = 1;
															#line 1509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetClusterIdActor, 0, Void >*>(static_cast<GetClusterIdActor*>(this)));
															#line 8863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetClusterIdActor*>(this)->actor_wait_state > 0) static_cast<GetClusterIdActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterIdActor*>(this)->ActorCallback< GetClusterIdActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterIdActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetClusterIdActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetClusterIdActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Database db;
															#line 8970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getClusterId()
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetClusterIdActor final : public Actor<UID>, public ActorCallback< GetClusterIdActor, 0, Void >, public FastAllocated<GetClusterIdActor>, public GetClusterIdActorState<GetClusterIdActor> {
															#line 8975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetClusterIdActor>::operator new;
	using FastAllocated<GetClusterIdActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9377887585830726656UL, 6649186260217430784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<UID>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetClusterIdActor, 0, Void >;
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetClusterIdActor(Database const& db) 
															#line 8992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<UID>(),
		   GetClusterIdActorState<GetClusterIdActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2565818813149026048UL, 5883691562570913024UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getClusterId");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetClusterIdActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<UID> getClusterId( Database const& db ) {
															#line 1507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<UID>(new GetClusterIdActor(db));
															#line 9025 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 1513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

void DatabaseContext::initializeSpecialCounters() {
	specialCounter(cc, "OutstandingWatches", [this] { return outstandingWatches; });
	specialCounter(cc, "WatchMapSize", [this] { return watchMap.size(); });
}

DatabaseContext::DatabaseContext(Reference<AsyncVar<Reference<IClusterConnectionRecord>>> connectionRecord,
                                 Reference<AsyncVar<ClientDBInfo>> clientInfo,
                                 Reference<AsyncVar<Optional<ClientLeaderRegInterface>> const> coordinator,
                                 Future<Void> clientInfoMonitor,
                                 TaskPriority taskID,
                                 LocalityData const& clientLocality,
                                 EnableLocalityLoadBalance enableLocalityLoadBalance,
                                 LockAware lockAware,
                                 IsInternal internal,
                                 int _apiVersion,
                                 IsSwitchable switchable,
                                 Optional<TenantName> defaultTenant)
  : dbId(deterministicRandom()->randomUniqueID()), lockAware(lockAware), switchable(switchable),
    connectionRecord(connectionRecord), proxyProvisional(false), clientLocality(clientLocality),
    enableLocalityLoadBalance(enableLocalityLoadBalance), defaultTenant(defaultTenant), internal(internal),
    cc("TransactionMetrics", dbId.toString()), transactionReadVersions("ReadVersions", cc),
    transactionReadVersionsThrottled("ReadVersionsThrottled", cc),
    transactionReadVersionsCompleted("ReadVersionsCompleted", cc),
    transactionReadVersionBatches("ReadVersionBatches", cc),
    transactionBatchReadVersions("BatchPriorityReadVersions", cc),
    transactionDefaultReadVersions("DefaultPriorityReadVersions", cc),
    transactionImmediateReadVersions("ImmediatePriorityReadVersions", cc),
    transactionBatchReadVersionsCompleted("BatchPriorityReadVersionsCompleted", cc),
    transactionDefaultReadVersionsCompleted("DefaultPriorityReadVersionsCompleted", cc),
    transactionImmediateReadVersionsCompleted("ImmediatePriorityReadVersionsCompleted", cc),
    transactionLogicalReads("LogicalUncachedReads", cc), transactionPhysicalReads("PhysicalReadRequests", cc),
    transactionPhysicalReadsCompleted("PhysicalReadRequestsCompleted", cc),
    transactionGetKeyRequests("GetKeyRequests", cc), transactionGetValueRequests("GetValueRequests", cc),
    transactionGetRangeRequests("GetRangeRequests", cc),
    transactionGetMappedRangeRequests("GetMappedRangeRequests", cc),
    transactionGetRangeStreamRequests("GetRangeStreamRequests", cc), transactionWatchRequests("WatchRequests", cc),
    transactionGetAddressesForKeyRequests("GetAddressesForKeyRequests", cc), transactionBytesRead("BytesRead", cc),
    transactionKeysRead("KeysRead", cc), transactionMetadataVersionReads("MetadataVersionReads", cc),
    transactionCommittedMutations("CommittedMutations", cc),
    transactionCommittedMutationBytes("CommittedMutationBytes", cc), transactionSetMutations("SetMutations", cc),
    transactionClearMutations("ClearMutations", cc), transactionAtomicMutations("AtomicMutations", cc),
    transactionsCommitStarted("CommitStarted", cc), transactionsCommitCompleted("CommitCompleted", cc),
    transactionKeyServerLocationRequests("KeyServerLocationRequests", cc),
    transactionKeyServerLocationRequestsCompleted("KeyServerLocationRequestsCompleted", cc),
    transactionBlobGranuleLocationRequests("BlobGranuleLocationRequests", cc),
    transactionBlobGranuleLocationRequestsCompleted("BlobGranuleLocationRequestsCompleted", cc),
    transactionStatusRequests("StatusRequests", cc), transactionTenantLookupRequests("TenantLookupRequests", cc),
    transactionTenantLookupRequestsCompleted("TenantLookupRequestsCompleted", cc), transactionsTooOld("TooOld", cc),
    transactionsFutureVersions("FutureVersions", cc), transactionsNotCommitted("NotCommitted", cc),
    transactionsMaybeCommitted("MaybeCommitted", cc), transactionsResourceConstrained("ResourceConstrained", cc),
    transactionsProcessBehind("ProcessBehind", cc), transactionsThrottled("Throttled", cc),
    transactionsLockRejected("LockRejected", cc),
    transactionsExpensiveClearCostEstCount("ExpensiveClearCostEstCount", cc),
    transactionGrvFullBatches("NumGrvFullBatches", cc), transactionGrvTimedOutBatches("NumGrvTimedOutBatches", cc),
    transactionCommitVersionNotFoundForSS("CommitVersionNotFoundForSS", cc), anyBGReads(false),
    ccBG("BlobGranuleReadMetrics", dbId.toString()), bgReadInputBytes("BGReadInputBytes", ccBG),
    bgReadOutputBytes("BGReadOutputBytes", ccBG), bgReadSnapshotRows("BGReadSnapshotRows", ccBG),
    bgReadRowsCleared("BGReadRowsCleared", ccBG), bgReadRowsInserted("BGReadRowsInserted", ccBG),
    bgReadRowsUpdated("BGReadRowsUpdated", ccBG), bgLatencies(), bgGranulesPerRequest(), usedAnyChangeFeeds(false),
    ccFeed("ChangeFeedClientMetrics", dbId.toString()), feedStreamStarts("FeedStreamStarts", ccFeed),
    feedMergeStreamStarts("FeedMergeStreamStarts", ccFeed), feedErrors("FeedErrors", ccFeed),
    feedNonRetriableErrors("FeedNonRetriableErrors", ccFeed), feedPops("FeedPops", ccFeed),
    feedPopsFallback("FeedPopsFallback", ccFeed), latencies(), readLatencies(), commitLatencies(), GRVLatencies(),
    mutationsPerCommit(), bytesPerCommit(), outstandingWatches(0), sharedStatePtr(nullptr), lastGrvTime(0.0),
    cachedReadVersion(0), lastRkBatchThrottleTime(0.0), lastRkDefaultThrottleTime(0.0), lastProxyRequestTime(0.0),
    transactionTracingSample(false), taskID(taskID), clientInfo(clientInfo), clientInfoMonitor(clientInfoMonitor),
    coordinator(coordinator), apiVersion(_apiVersion), mvCacheInsertLocation(0), healthMetricsLastUpdated(0),
    detailedHealthMetricsLastUpdated(0), smoothMidShardSize(CLIENT_KNOBS->SHARD_STAT_SMOOTH_AMOUNT),
    specialKeySpace(std::make_unique<SpecialKeySpace>(specialKeys.begin, specialKeys.end, /* test */ false)),
    connectToDatabaseEventCacheHolder(format("ConnectToDatabase/%s", dbId.toString().c_str())) {

	DisabledTraceEvent("DatabaseContextCreated", dbId).backtrace();

	connected = (clientInfo->get().commitProxies.size() && clientInfo->get().grvProxies.size())
	                ? Void()
	                : clientInfo->onChange();

	metadataVersionCache.resize(CLIENT_KNOBS->METADATA_VERSION_CACHE_SIZE);
	maxOutstandingWatches = CLIENT_KNOBS->DEFAULT_MAX_OUTSTANDING_WATCHES;

	snapshotRywEnabled = apiVersion.hasSnapshotRYW() ? 1 : 0;

	logger = databaseLogger(this) && tssLogger(this);
	locationCacheSize = g_network->isSimulated() ? CLIENT_KNOBS->LOCATION_CACHE_EVICTION_SIZE_SIM
	                                             : CLIENT_KNOBS->LOCATION_CACHE_EVICTION_SIZE;

	getValueSubmitted.init("NativeAPI.GetValueSubmitted"_sr);
	getValueCompleted.init("NativeAPI.GetValueCompleted"_sr);

	clientDBInfoMonitor = monitorClientDBInfoChange(this, clientInfo, &proxiesChangeTrigger);
	tssMismatchHandler = handleTssMismatches(this);
	clientStatusUpdater.actor = clientStatusUpdateActor(this);
	cacheListMonitor = monitorCacheList(this);

	smoothMidShardSize.reset(CLIENT_KNOBS->INIT_MID_SHARD_BYTES);
	globalConfig = std::make_unique<GlobalConfig>(this);

	if (apiVersion.version() >= 740) {
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::METRICS,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<FaultToleranceMetricsImpl>(
		        singleKeyRange("fault_tolerance_metrics_json"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::METRICS).begin)));
	}

	if (apiVersion.version() >= 700) {
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::ERRORMSG,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<SingleSpecialKeyImpl>(
		                            SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ERRORMSG).begin,
		                            [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			                            if (ryw->getSpecialKeySpaceErrorMsg().present())
				                            return Optional<Value>(ryw->getSpecialKeySpaceErrorMsg().get());
			                            else
				                            return Optional<Value>();
		                            },
		                            true));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ManagementCommandsOptionsImpl>(
		        KeyRangeRef("options/"_sr, "options0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ExcludeServersRangeImpl>(SpecialKeySpace::getManagementApiCommandRange("exclude")));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<FailedServersRangeImpl>(SpecialKeySpace::getManagementApiCommandRange("failed")));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::MANAGEMENT,
		                        SpecialKeySpace::IMPLTYPE::READWRITE,
		                        std::make_unique<ExcludedLocalitiesRangeImpl>(
		                            SpecialKeySpace::getManagementApiCommandRange("excludedlocality")));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::MANAGEMENT,
		                        SpecialKeySpace::IMPLTYPE::READWRITE,
		                        std::make_unique<FailedLocalitiesRangeImpl>(
		                            SpecialKeySpace::getManagementApiCommandRange("failedlocality")));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<ExclusionInProgressRangeImpl>(
		        KeyRangeRef("in_progress_exclusion/"_sr, "in_progress_exclusion0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONFIGURATION,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ProcessClassRangeImpl>(
		        KeyRangeRef("process/class_type/"_sr, "process/class_type0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::CONFIGURATION).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONFIGURATION,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<ProcessClassSourceRangeImpl>(
		        KeyRangeRef("process/class_source/"_sr, "process/class_source0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::CONFIGURATION).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<LockDatabaseImpl>(
		        singleKeyRange("db_locked"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ConsistencyCheckImpl>(
		        singleKeyRange("consistency_check_suspended"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::GLOBALCONFIG,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<GlobalConfigImpl>(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::GLOBALCONFIG)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::TRACING,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<TracingOptionsImpl>(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::TRACING)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONFIGURATION,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<CoordinatorsImpl>(
		        KeyRangeRef("coordinators/"_sr, "coordinators0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::CONFIGURATION).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<CoordinatorsAutoImpl>(
		        singleKeyRange("auto_coordinators"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<AdvanceVersionImpl>(
		        singleKeyRange("min_required_commit_version"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<VersionEpochImpl>(
		        singleKeyRange("version_epoch"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ClientProfilingImpl>(
		        KeyRangeRef("profiling/"_sr, "profiling0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)),
		    /* deprecated */ ApiVersion::withClientProfilingDeprecated().version());
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<MaintenanceImpl>(
		        KeyRangeRef("maintenance/"_sr, "maintenance0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<DataDistributionImpl>(
		        KeyRangeRef("data_distribution/"_sr, "data_distribution0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::ACTORLINEAGE,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<ActorLineageImpl>(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ACTORLINEAGE)));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::ACTOR_PROFILER_CONF,
		                        SpecialKeySpace::IMPLTYPE::READWRITE,
		                        std::make_unique<ActorProfilerConf>(
		                            SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ACTOR_PROFILER_CONF)));
	}
	if (apiVersion.version() >= 630) {
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::TRANSACTION,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<ConflictingKeysImpl>(conflictingKeysRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::TRANSACTION,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<ReadConflictRangeImpl>(readConflictRangeKeysRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::TRANSACTION,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<WriteConflictRangeImpl>(writeConflictRangeKeysRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::METRICS,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<DDStatsRangeImpl>(ddStatsRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::METRICS,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<HealthMetricsRangeImpl>(
		                            KeyRangeRef("\xff\xff/metrics/health/"_sr, "\xff\xff/metrics/health0"_sr)));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::WORKERINTERFACE,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<WorkerInterfacesSpecialKeyImpl>(
		                            KeyRangeRef("\xff\xff/worker_interfaces/"_sr, "\xff\xff/worker_interfaces0"_sr)));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::STATUSJSON,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<SingleSpecialKeyImpl>(
		                            "\xff\xff/status/json"_sr,
		                            [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			                            if (ryw->getDatabase().getPtr() && ryw->getDatabase()->getConnectionRecord()) {
				                            ++ryw->getDatabase()->transactionStatusRequests;
				                            return getJSON(ryw->getDatabase());
			                            } else {
				                            return Optional<Value>();
			                            }
		                            },
		                            true));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::CLUSTERFILEPATH,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<SingleSpecialKeyImpl>(
		                            "\xff\xff/cluster_file_path"_sr,
		                            [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			                            try {
				                            if (ryw->getDatabase().getPtr() &&
				                                ryw->getDatabase()->getConnectionRecord()) {
					                            Optional<Value> output =
					                                StringRef(ryw->getDatabase()->getConnectionRecord()->getLocation());
					                            return output;
				                            }
			                            } catch (Error& e) {
				                            return e;
			                            }
			                            return Optional<Value>();
		                            },
		                            true));

		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONNECTIONSTRING,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<SingleSpecialKeyImpl>(
		        "\xff\xff/connection_string"_sr,
		        [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			        try {
				        if (ryw->getDatabase().getPtr() && ryw->getDatabase()->getConnectionRecord()) {
					        Reference<IClusterConnectionRecord> f = ryw->getDatabase()->getConnectionRecord();
					        Optional<Value> output = StringRef(f->getConnectionString().toString());
					        return output;
				        }
			        } catch (Error& e) {
				        return e;
			        }
			        return Optional<Value>();
		        },
		        true));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::CLUSTERID,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<SingleSpecialKeyImpl>(
		                            "\xff\xff/cluster_id"_sr,
		                            [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			                            try {
				                            if (ryw->getDatabase().getPtr()) {
					                            return map(getClusterId(ryw->getDatabase()), [](UID id) {
						                            return Optional<Value>(StringRef(id.toString()));
					                            });
				                            }
			                            } catch (Error& e) {
				                            return e;
			                            }
			                            return Optional<Value>();
		                            },
		                            true));

		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<TenantRangeImpl>(SpecialKeySpace::getManagementApiCommandRange("tenant")));
	}
	throttleExpirer = recurring([this]() { expireThrottles(); }, CLIENT_KNOBS->TAG_THROTTLE_EXPIRATION_INTERVAL);

	if (BUGGIFY) {
		DatabaseContext::debugUseTags = true;
	}

	initializeSpecialCounters();
}

DatabaseContext::DatabaseContext(const Error& err)
  : deferredError(err), internal(IsInternal::False), cc("TransactionMetrics"),
    transactionReadVersions("ReadVersions", cc), transactionReadVersionsThrottled("ReadVersionsThrottled", cc),
    transactionReadVersionsCompleted("ReadVersionsCompleted", cc),
    transactionReadVersionBatches("ReadVersionBatches", cc),
    transactionBatchReadVersions("BatchPriorityReadVersions", cc),
    transactionDefaultReadVersions("DefaultPriorityReadVersions", cc),
    transactionImmediateReadVersions("ImmediatePriorityReadVersions", cc),
    transactionBatchReadVersionsCompleted("BatchPriorityReadVersionsCompleted", cc),
    transactionDefaultReadVersionsCompleted("DefaultPriorityReadVersionsCompleted", cc),
    transactionImmediateReadVersionsCompleted("ImmediatePriorityReadVersionsCompleted", cc),
    transactionLogicalReads("LogicalUncachedReads", cc), transactionPhysicalReads("PhysicalReadRequests", cc),
    transactionPhysicalReadsCompleted("PhysicalReadRequestsCompleted", cc),
    transactionGetKeyRequests("GetKeyRequests", cc), transactionGetValueRequests("GetValueRequests", cc),
    transactionGetRangeRequests("GetRangeRequests", cc),
    transactionGetMappedRangeRequests("GetMappedRangeRequests", cc),
    transactionGetRangeStreamRequests("GetRangeStreamRequests", cc), transactionWatchRequests("WatchRequests", cc),
    transactionGetAddressesForKeyRequests("GetAddressesForKeyRequests", cc), transactionBytesRead("BytesRead", cc),
    transactionKeysRead("KeysRead", cc), transactionMetadataVersionReads("MetadataVersionReads", cc),
    transactionCommittedMutations("CommittedMutations", cc),
    transactionCommittedMutationBytes("CommittedMutationBytes", cc), transactionSetMutations("SetMutations", cc),
    transactionClearMutations("ClearMutations", cc), transactionAtomicMutations("AtomicMutations", cc),
    transactionsCommitStarted("CommitStarted", cc), transactionsCommitCompleted("CommitCompleted", cc),
    transactionKeyServerLocationRequests("KeyServerLocationRequests", cc),
    transactionKeyServerLocationRequestsCompleted("KeyServerLocationRequestsCompleted", cc),
    transactionBlobGranuleLocationRequests("BlobGranuleLocationRequests", cc),
    transactionBlobGranuleLocationRequestsCompleted("BlobGranuleLocationRequestsCompleted", cc),
    transactionStatusRequests("StatusRequests", cc), transactionTenantLookupRequests("TenantLookupRequests", cc),
    transactionTenantLookupRequestsCompleted("TenantLookupRequestsCompleted", cc), transactionsTooOld("TooOld", cc),
    transactionsFutureVersions("FutureVersions", cc), transactionsNotCommitted("NotCommitted", cc),
    transactionsMaybeCommitted("MaybeCommitted", cc), transactionsResourceConstrained("ResourceConstrained", cc),
    transactionsProcessBehind("ProcessBehind", cc), transactionsThrottled("Throttled", cc),
    transactionsLockRejected("LockRejected", cc),
    transactionsExpensiveClearCostEstCount("ExpensiveClearCostEstCount", cc),
    transactionGrvFullBatches("NumGrvFullBatches", cc), transactionGrvTimedOutBatches("NumGrvTimedOutBatches", cc),
    transactionCommitVersionNotFoundForSS("CommitVersionNotFoundForSS", cc), anyBGReads(false),
    ccBG("BlobGranuleReadMetrics"), bgReadInputBytes("BGReadInputBytes", ccBG),
    bgReadOutputBytes("BGReadOutputBytes", ccBG), bgReadSnapshotRows("BGReadSnapshotRows", ccBG),
    bgReadRowsCleared("BGReadRowsCleared", ccBG), bgReadRowsInserted("BGReadRowsInserted", ccBG),
    bgReadRowsUpdated("BGReadRowsUpdated", ccBG), bgLatencies(), bgGranulesPerRequest(), usedAnyChangeFeeds(false),
    ccFeed("ChangeFeedClientMetrics"), feedStreamStarts("FeedStreamStarts", ccFeed),
    feedMergeStreamStarts("FeedMergeStreamStarts", ccFeed), feedErrors("FeedErrors", ccFeed),
    feedNonRetriableErrors("FeedNonRetriableErrors", ccFeed), feedPops("FeedPops", ccFeed),
    feedPopsFallback("FeedPopsFallback", ccFeed), latencies(), readLatencies(), commitLatencies(), GRVLatencies(),
    mutationsPerCommit(), bytesPerCommit(), sharedStatePtr(nullptr), transactionTracingSample(false),
    smoothMidShardSize(CLIENT_KNOBS->SHARD_STAT_SMOOTH_AMOUNT),
    connectToDatabaseEventCacheHolder(format("ConnectToDatabase/%s", dbId.toString().c_str())), outstandingWatches(0) {
	initializeSpecialCounters();
}

// Static constructor used by server processes to create a DatabaseContext
// For internal (fdbserver) use only
Database DatabaseContext::create(Reference<AsyncVar<ClientDBInfo>> clientInfo,
                                 Future<Void> clientInfoMonitor,
                                 LocalityData clientLocality,
                                 EnableLocalityLoadBalance enableLocalityLoadBalance,
                                 TaskPriority taskID,
                                 LockAware lockAware,
                                 int apiVersion,
                                 IsSwitchable switchable) {
	return Database(new DatabaseContext(Reference<AsyncVar<Reference<IClusterConnectionRecord>>>(),
	                                    clientInfo,
	                                    makeReference<AsyncVar<Optional<ClientLeaderRegInterface>>>(),
	                                    clientInfoMonitor,
	                                    taskID,
	                                    clientLocality,
	                                    enableLocalityLoadBalance,
	                                    lockAware,
	                                    IsInternal::True,
	                                    apiVersion,
	                                    switchable));
}

DatabaseContext::~DatabaseContext() {
	cacheListMonitor.cancel();
	clientDBInfoMonitor.cancel();
	monitorTssInfoChange.cancel();
	tssMismatchHandler.cancel();
	initializeChangeFeedCache = Void();
	storage = nullptr;
	changeFeedStorageCommitter = Void();
	if (grvUpdateHandler.isValid()) {
		grvUpdateHandler.cancel();
	}
	if (sharedStatePtr) {
		sharedStatePtr->delRef(sharedStatePtr);
	}
	for (auto it = server_interf.begin(); it != server_interf.end(); it = server_interf.erase(it))
		it->second->notifyContextDestroyed();
	ASSERT_ABORT(server_interf.empty());
	locationCache.insert(allKeys, Reference<LocationInfo>());
	for (auto& it : notAtLatestChangeFeeds) {
		it.second->context = nullptr;
	}
	for (auto& it : changeFeedUpdaters) {
		it.second->context = nullptr;
	}

	DisabledTraceEvent("DatabaseContextDestructed", dbId).backtrace();
}

Optional<KeyRangeLocationInfo> DatabaseContext::getCachedLocation(const TenantInfo& tenant,
                                                                  const KeyRef& key,
                                                                  Reverse isBackward) {
	Arena arena;
	KeyRef resolvedKey = key;

	if (tenant.hasTenant()) {
		CODE_PROBE(true, "Database context get cached location with tenant");
		resolvedKey = resolvedKey.withPrefix(tenant.prefix.get(), arena);
	}

	auto range =
	    isBackward ? locationCache.rangeContainingKeyBefore(resolvedKey) : locationCache.rangeContaining(resolvedKey);
	if (range->value()) {
		return KeyRangeLocationInfo(toPrefixRelativeRange(range->range(), tenant.prefix), range->value());
	}

	return Optional<KeyRangeLocationInfo>();
}

bool DatabaseContext::getCachedLocations(const TenantInfo& tenant,
                                         const KeyRangeRef& range,
                                         std::vector<KeyRangeLocationInfo>& result,
                                         int limit,
                                         Reverse reverse) {
	result.clear();

	Arena arena;
	KeyRangeRef resolvedRange = range;

	if (tenant.hasTenant()) {
		CODE_PROBE(true, "Database context get cached locations with tenant");
		resolvedRange = resolvedRange.withPrefix(tenant.prefix.get(), arena);
	}

	auto begin = locationCache.rangeContaining(resolvedRange.begin);
	auto end = locationCache.rangeContainingKeyBefore(resolvedRange.end);

	loop {
		auto r = reverse ? end : begin;
		if (!r->value()) {
			CODE_PROBE(result.size(), "had some but not all cached locations");
			result.clear();
			return false;
		}
		result.emplace_back(toPrefixRelativeRange(r->range() & resolvedRange, tenant.prefix), r->value());
		if (result.size() == limit || begin == end) {
			break;
		}

		if (reverse)
			--end;
		else
			++begin;
	}

	return true;
}

Reference<LocationInfo> DatabaseContext::setCachedLocation(const KeyRangeRef& absoluteKeys,
                                                           const std::vector<StorageServerInterface>& servers) {
	std::vector<Reference<ReferencedInterface<StorageServerInterface>>> serverRefs;
	serverRefs.reserve(servers.size());
	for (const auto& interf : servers) {
		serverRefs.push_back(StorageServerInfo::getInterface(this, interf, clientLocality));
	}

	int maxEvictionAttempts = 100, attempts = 0;
	auto loc = makeReference<LocationInfo>(serverRefs);
	while (locationCache.size() > locationCacheSize && attempts < maxEvictionAttempts) {
		CODE_PROBE(true, "NativeAPI storage server locationCache entry evicted");
		attempts++;
		auto r = locationCache.randomRange();
		Key begin = r.begin(), end = r.end(); // insert invalidates r, so can't be passed a mere reference into it
		locationCache.insert(KeyRangeRef(begin, end), Reference<LocationInfo>());
	}
	locationCache.insert(absoluteKeys, loc);
	return loc;
}

void DatabaseContext::invalidateCache(const Optional<KeyRef>& tenantPrefix, const KeyRef& key, Reverse isBackward) {
	Arena arena;
	KeyRef resolvedKey = key;
	if (tenantPrefix.present() && !tenantPrefix.get().empty()) {
		CODE_PROBE(true, "Database context invalidate cache for tenant key");
		resolvedKey = resolvedKey.withPrefix(tenantPrefix.get(), arena);
	}

	if (isBackward) {
		locationCache.rangeContainingKeyBefore(resolvedKey)->value() = Reference<LocationInfo>();
	} else {
		locationCache.rangeContaining(resolvedKey)->value() = Reference<LocationInfo>();
	}
}

void DatabaseContext::invalidateCache(const Optional<KeyRef>& tenantPrefix, const KeyRangeRef& keys) {
	Arena arena;
	KeyRangeRef resolvedKeys = keys;
	if (tenantPrefix.present() && !tenantPrefix.get().empty()) {
		CODE_PROBE(true, "Database context invalidate cache for tenant range");
		resolvedKeys = resolvedKeys.withPrefix(tenantPrefix.get(), arena);
	}

	auto rs = locationCache.intersectingRanges(resolvedKeys);
	Key begin = rs.begin().begin(),
	    end = rs.end().begin(); // insert invalidates rs, so can't be passed a mere reference into it
	locationCache.insert(KeyRangeRef(begin, end), Reference<LocationInfo>());
}

void DatabaseContext::setFailedEndpointOnHealthyServer(const Endpoint& endpoint) {
	if (failedEndpointsOnHealthyServersInfo.find(endpoint) == failedEndpointsOnHealthyServersInfo.end()) {
		failedEndpointsOnHealthyServersInfo[endpoint] =
		    EndpointFailureInfo{ .startTime = now(), .lastRefreshTime = now() };
	}
}

void DatabaseContext::updateFailedEndpointRefreshTime(const Endpoint& endpoint) {
	if (failedEndpointsOnHealthyServersInfo.find(endpoint) == failedEndpointsOnHealthyServersInfo.end()) {
		// The endpoint is not failed. Nothing to update.
		return;
	}
	failedEndpointsOnHealthyServersInfo[endpoint].lastRefreshTime = now();
}

Optional<EndpointFailureInfo> DatabaseContext::getEndpointFailureInfo(const Endpoint& endpoint) {
	if (failedEndpointsOnHealthyServersInfo.find(endpoint) == failedEndpointsOnHealthyServersInfo.end()) {
		return Optional<EndpointFailureInfo>();
	}
	return failedEndpointsOnHealthyServersInfo[endpoint];
}

void DatabaseContext::clearFailedEndpointOnHealthyServer(const Endpoint& endpoint) {
	failedEndpointsOnHealthyServersInfo.erase(endpoint);
}

Future<Void> DatabaseContext::onProxiesChanged() {
	backoffDelay = 0.0;
	return this->proxiesChangeTrigger.onTrigger();
}

bool DatabaseContext::sampleReadTags() const {
	double sampleRate = globalConfig->get(transactionTagSampleRate, CLIENT_KNOBS->READ_TAG_SAMPLE_RATE);
	return sampleRate > 0 && deterministicRandom()->random01() <= sampleRate;
}

bool DatabaseContext::sampleOnCost(uint64_t cost) const {
	double sampleCost = globalConfig->get<double>(transactionTagSampleCost, CLIENT_KNOBS->COMMIT_SAMPLE_COST);
	if (sampleCost <= 0)
		return false;
	return deterministicRandom()->random01() <= (double)cost / sampleCost;
}

int64_t extractIntOption(Optional<StringRef> value, int64_t minValue, int64_t maxValue) {
	validateOptionValuePresent(value);
	if (value.get().size() != 8) {
		throw invalid_option_value();
	}

	int64_t passed = *((int64_t*)(value.get().begin()));
	if (passed > maxValue || passed < minValue) {
		throw invalid_option_value();
	}

	return passed;
}

uint64_t extractHexOption(StringRef value) {
	char* end;
	uint64_t id = strtoull(value.toString().c_str(), &end, 16);
	if (*end)
		throw invalid_option_value();
	return id;
}

void DatabaseContext::setOption(FDBDatabaseOptions::Option option, Optional<StringRef> value) {
	int defaultFor = FDBDatabaseOptions::optionInfo.getMustExist(option).defaultFor;
	if (defaultFor >= 0) {
		ASSERT(FDBTransactionOptions::optionInfo.find((FDBTransactionOptions::Option)defaultFor) !=
		       FDBTransactionOptions::optionInfo.end());
		TraceEvent(SevDebug, "DatabaseContextSetPersistentOption").detail("Option", option).detail("Value", value);
		transactionDefaults.addOption((FDBTransactionOptions::Option)defaultFor, value.castTo<Standalone<StringRef>>());
	} else {
		switch (option) {
		case FDBDatabaseOptions::LOCATION_CACHE_SIZE:
			locationCacheSize = (int)extractIntOption(value, 0, std::numeric_limits<int>::max());
			break;
		case FDBDatabaseOptions::MACHINE_ID:
			clientLocality =
			    LocalityData(clientLocality.processId(),
			                 value.present() ? Standalone<StringRef>(value.get()) : Optional<Standalone<StringRef>>(),
			                 clientLocality.machineId(),
			                 clientLocality.dcId());
			if (clientInfo->get().commitProxies.size())
				commitProxies = makeReference<CommitProxyInfo>(clientInfo->get().commitProxies);
			if (clientInfo->get().grvProxies.size())
				grvProxies = makeReference<GrvProxyInfo>(clientInfo->get().grvProxies, BalanceOnRequests::True);
			server_interf.clear();
			locationCache.insert(allKeys, Reference<LocationInfo>());
			break;
		case FDBDatabaseOptions::MAX_WATCHES:
			maxOutstandingWatches = (int)extractIntOption(value, 0, CLIENT_KNOBS->ABSOLUTE_MAX_WATCHES);
			break;
		case FDBDatabaseOptions::DATACENTER_ID:
			clientLocality =
			    LocalityData(clientLocality.processId(),
			                 clientLocality.zoneId(),
			                 clientLocality.machineId(),
			                 value.present() ? Standalone<StringRef>(value.get()) : Optional<Standalone<StringRef>>());
			if (clientInfo->get().commitProxies.size())
				commitProxies = makeReference<CommitProxyInfo>(clientInfo->get().commitProxies);
			if (clientInfo->get().grvProxies.size())
				grvProxies = makeReference<GrvProxyInfo>(clientInfo->get().grvProxies, BalanceOnRequests::True);
			server_interf.clear();
			locationCache.insert(allKeys, Reference<LocationInfo>());
			break;
		case FDBDatabaseOptions::SNAPSHOT_RYW_ENABLE:
			validateOptionValueNotPresent(value);
			snapshotRywEnabled++;
			break;
		case FDBDatabaseOptions::SNAPSHOT_RYW_DISABLE:
			validateOptionValueNotPresent(value);
			snapshotRywEnabled--;
			break;
		case FDBDatabaseOptions::USE_CONFIG_DATABASE:
			validateOptionValueNotPresent(value);
			useConfigDatabase = true;
			break;
		case FDBDatabaseOptions::TEST_CAUSAL_READ_RISKY:
			verifyCausalReadsProp = double(extractIntOption(value, 0, 100)) / 100.0;
			break;
		default:
			break;
		}
	}
}

void DatabaseContext::increaseWatchCounter() {
	if (outstandingWatches >= maxOutstandingWatches)
		throw too_many_watches();

	++outstandingWatches;
}

void DatabaseContext::decreaseWatchCounter() {
	--outstandingWatches;
	ASSERT(outstandingWatches >= 0);
}

Future<Void> DatabaseContext::onConnected() const {
	return connected;
}

															#line 9716 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via switchConnectionRecordImpl()
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class SwitchConnectionRecordImplActor>
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class SwitchConnectionRecordImplActorState {
															#line 9723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	SwitchConnectionRecordImplActorState(Reference<IClusterConnectionRecord> const& connRecord,DatabaseContext* const& self) 
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : connRecord(connRecord),
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   self(self)
															#line 9732 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this));

	}
	~SwitchConnectionRecordImplActorState() 
	{
		fdb_probe_actor_destroy("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(true, "Switch connection file");
															#line 2203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent("SwitchConnectionRecord") .detail("ClusterFile", connRecord->toString()) .detail("ConnectionString", connRecord->getConnectionString().toString());
															#line 2208 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->commitProxies.clear();
															#line 2209 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->grvProxies.clear();
															#line 2210 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->minAcceptableReadVersion = std::numeric_limits<Version>::max();
															#line 2211 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->invalidateCache({}, allKeys);
															#line 2213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->ssVersionVectorCache.clear();
															#line 2215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			auto clearedClientInfo = self->clientInfo->get();
															#line 2216 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			clearedClientInfo.commitProxies.clear();
															#line 2217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			clearedClientInfo.grvProxies.clear();
															#line 2218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			clearedClientInfo.id = deterministicRandom()->randomUniqueID();
															#line 2219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->clientInfo->set(clearedClientInfo);
															#line 2220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			self->connectionRecord->set(connRecord);
															#line 2222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			db = Database(Reference<DatabaseContext>::addRef(self));
															#line 2223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr = Transaction(db);
															#line 2224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 9777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SwitchConnectionRecordImplActorState();
		static_cast<SwitchConnectionRecordImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2225 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_LOCK_AWARE);
															#line 9807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		try {
															#line 2227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent("SwitchConnectionRecordAttemptingGRV").log();
															#line 2228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Version> __when_expr_0 = tr.getReadVersion();
															#line 2228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state = 1;
															#line 2228 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SwitchConnectionRecordImplActor, 0, Version >*>(static_cast<SwitchConnectionRecordImplActor*>(this)));
															#line 9820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			TraceEvent("SwitchConnectionRecordError").detail("Error", e.what());
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state = 2;
															#line 2237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SwitchConnectionRecordImplActor, 1, Void >*>(static_cast<SwitchConnectionRecordImplActor*>(this)));
															#line 9851 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version const& v,int loopDepth) 
	{
															#line 2229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		TraceEvent("SwitchConnectionRecordGotRV") .detail("ReadVersion", v) .detail("MinAcceptableReadVersion", self->minAcceptableReadVersion);
															#line 2232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(self->minAcceptableReadVersion != std::numeric_limits<Version>::max());
															#line 2233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		self->connectionFileChangedTrigger.trigger();
															#line 2234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<SwitchConnectionRecordImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SwitchConnectionRecordImplActorState(); static_cast<SwitchConnectionRecordImplActor*>(this)->destroy(); return 0; }
															#line 9872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<SwitchConnectionRecordImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SwitchConnectionRecordImplActorState();
		static_cast<SwitchConnectionRecordImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Version && v,int loopDepth) 
	{
															#line 2229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		TraceEvent("SwitchConnectionRecordGotRV") .detail("ReadVersion", v) .detail("MinAcceptableReadVersion", self->minAcceptableReadVersion);
															#line 2232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(self->minAcceptableReadVersion != std::numeric_limits<Version>::max());
															#line 2233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		self->connectionFileChangedTrigger.trigger();
															#line 2234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<SwitchConnectionRecordImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SwitchConnectionRecordImplActorState(); static_cast<SwitchConnectionRecordImplActor*>(this)->destroy(); return 0; }
															#line 9890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<SwitchConnectionRecordImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SwitchConnectionRecordImplActorState();
		static_cast<SwitchConnectionRecordImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& v,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(v, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && v,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state > 0) static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state = 0;
		static_cast<SwitchConnectionRecordImplActor*>(this)->ActorCallback< SwitchConnectionRecordImplActor, 0, Version >::remove();

	}
	void a_callback_fire(ActorCallback< SwitchConnectionRecordImplActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SwitchConnectionRecordImplActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SwitchConnectionRecordImplActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state > 0) static_cast<SwitchConnectionRecordImplActor*>(this)->actor_wait_state = 0;
		static_cast<SwitchConnectionRecordImplActor*>(this)->ActorCallback< SwitchConnectionRecordImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SwitchConnectionRecordImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SwitchConnectionRecordImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SwitchConnectionRecordImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<IClusterConnectionRecord> connRecord;
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* self;
															#line 2222 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Database db;
															#line 2223 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction tr;
															#line 10068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via switchConnectionRecordImpl()
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class SwitchConnectionRecordImplActor final : public Actor<Void>, public ActorCallback< SwitchConnectionRecordImplActor, 0, Version >, public ActorCallback< SwitchConnectionRecordImplActor, 1, Void >, public FastAllocated<SwitchConnectionRecordImplActor>, public SwitchConnectionRecordImplActorState<SwitchConnectionRecordImplActor> {
															#line 10073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<SwitchConnectionRecordImplActor>::operator new;
	using FastAllocated<SwitchConnectionRecordImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12843648440376135424UL, 4723058308837381632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SwitchConnectionRecordImplActor, 0, Version >;
friend struct ActorCallback< SwitchConnectionRecordImplActor, 1, Void >;
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	SwitchConnectionRecordImplActor(Reference<IClusterConnectionRecord> const& connRecord,DatabaseContext* const& self) 
															#line 10091 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   SwitchConnectionRecordImplActorState<SwitchConnectionRecordImplActor>(connRecord, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7312653141277564928UL, 2692131377335283456UL);
		ActorExecutionContextHelper __helper(static_cast<SwitchConnectionRecordImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("switchConnectionRecordImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("switchConnectionRecordImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SwitchConnectionRecordImplActor, 0, Version >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SwitchConnectionRecordImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] static Future<Void> switchConnectionRecordImpl( Reference<IClusterConnectionRecord> const& connRecord, DatabaseContext* const& self ) {
															#line 2200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new SwitchConnectionRecordImplActor(connRecord, self));
															#line 10125 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 2241 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

Reference<IClusterConnectionRecord> DatabaseContext::getConnectionRecord() {
	if (connectionRecord) {
		return connectionRecord->get();
	}
	return Reference<IClusterConnectionRecord>();
}

Future<Void> DatabaseContext::switchConnectionRecord(Reference<IClusterConnectionRecord> standby) {
	ASSERT(switchable);
	return switchConnectionRecordImpl(standby, this);
}

Future<Void> DatabaseContext::connectionFileChanged() {
	return connectionFileChangedTrigger.onTrigger();
}

void DatabaseContext::expireThrottles() {
	for (auto& priorityItr : throttledTags) {
		for (auto tagItr = priorityItr.second.begin(); tagItr != priorityItr.second.end();) {
			if (tagItr->second.expired()) {
				CODE_PROBE(true, "Expiring client throttle");
				tagItr = priorityItr.second.erase(tagItr);
			} else {
				++tagItr;
			}
		}
	}
}

// Initialize tracing for FDB client
//
// connRecord is necessary for determining the local IP, which is then included in the trace
// file name, and also used to annotate all trace events.
//
// If trace_initialize_on_setup is not set, tracing is initialized when opening a database.
// In that case we can immediately determine the IP. Thus, we can use the IP in the
// trace file name and annotate all events with it.
//
// If trace_initialize_on_setup network option is set, tracing is at first initialized without
// connRecord and thus without the local IP. In that case we cannot use the local IP in the
// trace file names. The IP is then provided by a repeated call to initializeClientTracing
// when opening a database. All tracing events from this point are annotated with the local IP
//
// If tracing initialization is completed, further calls to initializeClientTracing are ignored
void initializeClientTracing(Reference<IClusterConnectionRecord> connRecord, Optional<int> apiVersion) {
	if (!networkOptions.traceDirectory.present()) {
		return;
	}

	bool initialized = traceFileIsOpen();
	if (initialized && (isTraceLocalAddressSet() || !connRecord)) {
		// Tracing initialization is completed
		return;
	}

	// Network must be created before initializing tracing
	ASSERT(g_network);

	Optional<NetworkAddress> localAddress;
	if (connRecord) {
		auto publicIP = connRecord->getConnectionString().determineLocalSourceIP();
		localAddress = NetworkAddress(publicIP, ::getpid());
	}
	platform::ImageInfo imageInfo = platform::getImageInfo();

	if (initialized) {
		// Tracing already initialized, just need to update the IP address
		setTraceLocalAddress(localAddress.get());
		TraceEvent("ClientStart")
		    .detail("SourceVersion", getSourceVersion())
		    .detail("Version", FDB_VT_VERSION)
		    .detail("PackageName", FDB_VT_PACKAGE_NAME)
		    .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(nullptr))
		    .detail("ApiVersion", apiVersion)
		    .detail("ClientLibrary", imageInfo.fileName)
		    .detailf("ImageOffset", "%p", imageInfo.offset)
		    .detail("Primary", networkOptions.primaryClient)
		    .trackLatest("ClientStart");
	} else {
		// Initialize tracing
		selectTraceFormatter(networkOptions.traceFormat);
		selectTraceClockSource(networkOptions.traceClockSource);
		addUniversalTraceField("ClientDescription",
		                       format("%s-%s-%" PRIu64,
		                              networkOptions.primaryClient ? "primary" : "external",
		                              FDB_VT_VERSION,
		                              deterministicRandom()->randomUInt64()));

		std::string identifier = networkOptions.traceFileIdentifier;
		openTraceFile(localAddress,
		              networkOptions.traceRollSize,
		              networkOptions.traceMaxLogsSize,
		              networkOptions.traceDirectory.get(),
		              "trace",
		              networkOptions.traceLogGroup,
		              identifier,
		              networkOptions.tracePartialFileSuffix,
		              InitializeTraceMetrics::True);

		TraceEvent("ClientStart")
		    .detail("SourceVersion", getSourceVersion())
		    .detail("Version", FDB_VT_VERSION)
		    .detail("PackageName", FDB_VT_PACKAGE_NAME)
		    .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(nullptr))
		    .detail("ApiVersion", apiVersion)
		    .detail("ClientLibrary", imageInfo.fileName)
		    .detailf("ImageOffset", "%p", imageInfo.offset)
		    .detail("Primary", networkOptions.primaryClient)
		    .trackLatest("ClientStart");

		g_network->initMetrics();
		FlowTransport::transport().initMetrics();
	}

	// Initialize system monitoring once the local IP is available
	if (localAddress.present()) {
		initializeSystemMonitorMachineState(SystemMonitorMachineState(IPAddress(localAddress.get().ip)));
		systemMonitor();
		uncancellable(recurring(&systemMonitor, CLIENT_KNOBS->SYSTEM_MONITOR_INTERVAL, TaskPriority::FlushTrace));
	}
}

// Creates a database object that represents a connection to a cluster
// This constructor uses a preallocated DatabaseContext that may have been created
// on another thread
Database Database::createDatabase(Reference<IClusterConnectionRecord> connRecord,
                                  int apiVersion,
                                  IsInternal internal,
                                  LocalityData const& clientLocality,
                                  DatabaseContext* preallocatedDb) {
	if (!g_network)
		throw network_not_setup();

	ASSERT(TraceEvent::isNetworkThread());

	initializeClientTracing(connRecord, apiVersion);

	g_network->initTLS();

	auto clientInfo = makeReference<AsyncVar<ClientDBInfo>>();
	auto coordinator = makeReference<AsyncVar<Optional<ClientLeaderRegInterface>>>();
	auto connectionRecord = makeReference<AsyncVar<Reference<IClusterConnectionRecord>>>();
	connectionRecord->set(connRecord);
	Future<Void> clientInfoMonitor = monitorProxies(connectionRecord,
	                                                clientInfo,
	                                                coordinator,
	                                                networkOptions.supportedVersions,
	                                                StringRef(networkOptions.traceLogGroup),
	                                                internal);

	DatabaseContext* db;
	if (preallocatedDb) {
		db = new (preallocatedDb) DatabaseContext(connectionRecord,
		                                          clientInfo,
		                                          coordinator,
		                                          clientInfoMonitor,
		                                          TaskPriority::DefaultEndpoint,
		                                          clientLocality,
		                                          EnableLocalityLoadBalance::True,
		                                          LockAware::False,
		                                          internal,
		                                          apiVersion,
		                                          IsSwitchable::True);
	} else {
		db = new DatabaseContext(connectionRecord,
		                         clientInfo,
		                         coordinator,
		                         clientInfoMonitor,
		                         TaskPriority::DefaultEndpoint,
		                         clientLocality,
		                         EnableLocalityLoadBalance::True,
		                         LockAware::False,
		                         internal,
		                         apiVersion,
		                         IsSwitchable::True);
	}

	auto database = Database(db);
	database->globalConfig->init(Reference<AsyncVar<ClientDBInfo> const>(clientInfo),
	                             std::addressof(clientInfo->get()));
	database->globalConfig->trigger(samplingFrequency, samplingProfilerUpdateFrequency);
	database->globalConfig->trigger(samplingWindow, samplingProfilerUpdateWindow);

	TraceEvent("ConnectToDatabase", database->dbId)
	    .detail("Version", FDB_VT_VERSION)
	    .detail("ClusterFile", connRecord ? connRecord->toString() : "None")
	    .detail("ConnectionString", connRecord ? connRecord->getConnectionString().toString() : "None")
	    .detail("ClientLibrary", platform::getImageInfo().fileName)
	    .detail("Primary", networkOptions.primaryClient)
	    .detail("Internal", internal)
	    .trackLatest(database->connectToDatabaseEventCacheHolder.trackingKey);

	return database;
}

Database Database::createDatabase(std::string connFileName,
                                  int apiVersion,
                                  IsInternal internal,
                                  LocalityData const& clientLocality) {
	Reference<IClusterConnectionRecord> rccr = ClusterConnectionFile::openOrDefault(connFileName);
	return Database::createDatabase(rccr, apiVersion, internal, clientLocality);
}

Database Database::createSimulatedExtraDatabase(std::string connectionString, Optional<TenantName> defaultTenant) {
	auto extraFile = makeReference<ClusterConnectionMemoryRecord>(ClusterConnectionString(connectionString));
	Database db = Database::createDatabase(extraFile, ApiVersion::LATEST_VERSION);
	db->defaultTenant = defaultTenant;
	return db;
}

const UniqueOrderedOptionList<FDBTransactionOptions>& Database::getTransactionDefaults() const {
	ASSERT(db);
	return db->transactionDefaults;
}

void setNetworkOption(FDBNetworkOptions::Option option, Optional<StringRef> value) {
	std::regex identifierRegex("^[a-zA-Z0-9_]*$");
	switch (option) {
	// SOMEDAY: If the network is already started, should these five throw an error?
	case FDBNetworkOptions::TRACE_ENABLE:
		networkOptions.traceDirectory = value.present() ? value.get().toString() : "";
		break;
	case FDBNetworkOptions::TRACE_ROLL_SIZE:
		validateOptionValuePresent(value);
		networkOptions.traceRollSize = extractIntOption(value, 0, std::numeric_limits<int64_t>::max());
		break;
	case FDBNetworkOptions::TRACE_MAX_LOGS_SIZE:
		validateOptionValuePresent(value);
		networkOptions.traceMaxLogsSize = extractIntOption(value, 0, std::numeric_limits<int64_t>::max());
		break;
	case FDBNetworkOptions::TRACE_FORMAT:
		validateOptionValuePresent(value);
		networkOptions.traceFormat = value.get().toString();
		if (!validateTraceFormat(networkOptions.traceFormat)) {
			fprintf(stderr, "Unrecognized trace format: `%s'\n", networkOptions.traceFormat.c_str());
			throw invalid_option_value();
		}
		break;
	case FDBNetworkOptions::TRACE_FILE_IDENTIFIER:
		validateOptionValuePresent(value);
		networkOptions.traceFileIdentifier = value.get().toString();
		if (networkOptions.traceFileIdentifier.length() > CLIENT_KNOBS->TRACE_LOG_FILE_IDENTIFIER_MAX_LENGTH) {
			fprintf(stderr, "Trace file identifier provided is too long.\n");
			throw invalid_option_value();
		} else if (!std::regex_match(networkOptions.traceFileIdentifier, identifierRegex)) {
			fprintf(stderr, "Trace file identifier should only contain alphanumerics and underscores.\n");
			throw invalid_option_value();
		}
		break;

	case FDBNetworkOptions::TRACE_LOG_GROUP:
		if (value.present()) {
			if (traceFileIsOpen()) {
				setTraceLogGroup(value.get().toString());
			} else {
				networkOptions.traceLogGroup = value.get().toString();
			}
		}
		break;
	case FDBNetworkOptions::TRACE_CLOCK_SOURCE:
		validateOptionValuePresent(value);
		networkOptions.traceClockSource = value.get().toString();
		if (!validateTraceClockSource(networkOptions.traceClockSource)) {
			fprintf(stderr, "Unrecognized trace clock source: `%s'\n", networkOptions.traceClockSource.c_str());
			throw invalid_option_value();
		}
		break;
	case FDBNetworkOptions::TRACE_PARTIAL_FILE_SUFFIX:
		validateOptionValuePresent(value);
		networkOptions.tracePartialFileSuffix = value.get().toString();
		break;
	case FDBNetworkOptions::TRACE_INITIALIZE_ON_SETUP:
		networkOptions.traceInitializeOnSetup = true;
		break;
	case FDBNetworkOptions::KNOB: {
		validateOptionValuePresent(value);

		std::string optionValue = value.get().toString();
		TraceEvent("SetKnob").detail("KnobString", optionValue);

		size_t eq = optionValue.find_first_of('=');
		if (eq == optionValue.npos) {
			TraceEvent(SevWarnAlways, "InvalidKnobString").detail("KnobString", optionValue);
			throw invalid_option_value();
		}

		std::string knobName = optionValue.substr(0, eq);
		std::string knobValueString = optionValue.substr(eq + 1);

		try {
			auto knobValue = IKnobCollection::parseKnobValue(knobName, knobValueString, IKnobCollection::Type::CLIENT);
			if (g_network) {
				IKnobCollection::getMutableGlobalKnobCollection().setKnob(knobName, knobValue);
			} else {
				networkOptions.knobs[knobName] = knobValue;
			}
		} catch (Error& e) {
			TraceEvent(SevWarnAlways, "UnrecognizedKnob").detail("Knob", knobName.c_str());
			fprintf(stderr, "FoundationDB client ignoring unrecognized knob option '%s'\n", knobName.c_str());
		}
		break;
	}
	case FDBNetworkOptions::TLS_PLUGIN:
		validateOptionValuePresent(value);
		break;
	case FDBNetworkOptions::TLS_CERT_PATH:
		validateOptionValuePresent(value);
		tlsConfig.setCertificatePath(value.get().toString());
		break;
	case FDBNetworkOptions::TLS_CERT_BYTES: {
		validateOptionValuePresent(value);
		tlsConfig.setCertificateBytes(value.get().toString());
		break;
	}
	case FDBNetworkOptions::TLS_CA_PATH: {
		validateOptionValuePresent(value);
		tlsConfig.setCAPath(value.get().toString());
		break;
	}
	case FDBNetworkOptions::TLS_CA_BYTES: {
		validateOptionValuePresent(value);
		tlsConfig.setCABytes(value.get().toString());
		break;
	}
	case FDBNetworkOptions::TLS_PASSWORD:
		validateOptionValuePresent(value);
		tlsConfig.setPassword(value.get().toString());
		break;
	case FDBNetworkOptions::TLS_KEY_PATH:
		validateOptionValuePresent(value);
		tlsConfig.setKeyPath(value.get().toString());
		break;
	case FDBNetworkOptions::TLS_KEY_BYTES: {
		validateOptionValuePresent(value);
		tlsConfig.setKeyBytes(value.get().toString());
		break;
	}
	case FDBNetworkOptions::TLS_VERIFY_PEERS:
		validateOptionValuePresent(value);
		tlsConfig.clearVerifyPeers();
		tlsConfig.addVerifyPeers(value.get().toString());
		break;
	case FDBNetworkOptions::TLS_DISABLE_PLAINTEXT_CONNECTION:
		tlsConfig.setDisablePlainTextConnection(true);
		break;
	case FDBNetworkOptions::CLIENT_BUGGIFY_ENABLE:
		enableClientBuggify();
		break;
	case FDBNetworkOptions::CLIENT_BUGGIFY_DISABLE:
		disableClientBuggify();
		break;
	case FDBNetworkOptions::CLIENT_BUGGIFY_SECTION_ACTIVATED_PROBABILITY:
		validateOptionValuePresent(value);
		clearClientBuggifySections();
		P_CLIENT_BUGGIFIED_SECTION_ACTIVATED = double(extractIntOption(value, 0, 100)) / 100.0;
		break;
	case FDBNetworkOptions::CLIENT_BUGGIFY_SECTION_FIRED_PROBABILITY:
		validateOptionValuePresent(value);
		P_CLIENT_BUGGIFIED_SECTION_FIRES = double(extractIntOption(value, 0, 100)) / 100.0;
		break;
	case FDBNetworkOptions::DISABLE_CLIENT_STATISTICS_LOGGING:
		validateOptionValueNotPresent(value);
		networkOptions.logClientInfo = false;
		break;
	case FDBNetworkOptions::SUPPORTED_CLIENT_VERSIONS: {
		// The multi-version API should be providing us these guarantees
		ASSERT(g_network);
		ASSERT(value.present());

		Standalone<VectorRef<ClientVersionRef>> supportedVersions;
		std::vector<StringRef> supportedVersionsStrings = value.get().splitAny(";"_sr);
		for (StringRef versionString : supportedVersionsStrings) {
#ifdef ADDRESS_SANITIZER
			__lsan_disable();
#endif
			// LSAN reports that we leak this allocation in client
			// tests, but I cannot seem to figure out why. AFAICT
			// it's not actually leaking. If it is a leak, it's only a few bytes.
			supportedVersions.push_back_deep(supportedVersions.arena(), ClientVersionRef(versionString));
#ifdef ADDRESS_SANITIZER
			__lsan_enable();
#endif
		}

		ASSERT(supportedVersions.size() > 0);
		networkOptions.supportedVersions->set(supportedVersions);

		break;
	}
	case FDBNetworkOptions::ENABLE_RUN_LOOP_PROFILING: // Same as ENABLE_SLOW_TASK_PROFILING
		validateOptionValueNotPresent(value);
		networkOptions.runLoopProfilingEnabled = true;
		break;
	case FDBNetworkOptions::DISTRIBUTED_CLIENT_TRACER: {
		validateOptionValuePresent(value);
		std::string tracer = value.get().toString();
		if (tracer == "none" || tracer == "disabled") {
			openTracer(TracerType::DISABLED);
		} else if (tracer == "logfile" || tracer == "file" || tracer == "log_file") {
			openTracer(TracerType::LOG_FILE);
		} else if (tracer == "network_lossy") {
			openTracer(TracerType::NETWORK_LOSSY);
		} else {
			fprintf(stderr, "ERROR: Unknown or unsupported tracer: `%s'", tracer.c_str());
			throw invalid_option_value();
		}
		break;
	}
	case FDBNetworkOptions::EXTERNAL_CLIENT:
		networkOptions.primaryClient = false;
		break;
	default:
		break;
	}
}

// update the network busyness on a 1s cadence
															#line 10547 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorNetworkBusyness()
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class MonitorNetworkBusynessActor>
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class MonitorNetworkBusynessActorState {
															#line 10554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	MonitorNetworkBusynessActorState() 
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 2660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : prevTime(now())
															#line 10561 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this));

	}
	~MonitorNetworkBusynessActorState() 
	{
		fdb_probe_actor_destroy("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 10576 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorNetworkBusynessActorState();
		static_cast<MonitorNetworkBusynessActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(CLIENT_KNOBS->NETWORK_BUSYNESS_MONITOR_INTERVAL, TaskPriority::FlushTrace);
															#line 2662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<MonitorNetworkBusynessActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MonitorNetworkBusynessActor*>(this)->actor_wait_state = 1;
															#line 2662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorNetworkBusynessActor, 0, Void >*>(static_cast<MonitorNetworkBusynessActor*>(this)));
															#line 10613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 2663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double elapsed = now() - prevTime;
															#line 2664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		prevTime = now();
															#line 2665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		struct NetworkMetrics::PriorityStats& tracker = g_network->networkInfo.metrics.starvationTrackerNetworkBusyness;
															#line 2667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (tracker.active)
															#line 10628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracker.duration += now() - tracker.windowedTimer;
															#line 2669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracker.maxDuration = std::max(tracker.maxDuration, now() - tracker.timer);
															#line 2670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracker.windowedTimer = now();
															#line 10636 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 2673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double busyFraction = std::min(elapsed, tracker.duration) / elapsed;
															#line 2681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double burstiness = 0;
															#line 2682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD >= 0 && CLIENT_KNOBS->BUSYNESS_SPIKE_SATURATED_THRESHOLD >= CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD)
															#line 10644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			burstiness = std::min(1.0, std::max(0.0, tracker.maxDuration - CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD) / std::max(1e-6, CLIENT_KNOBS->BUSYNESS_SPIKE_SATURATED_THRESHOLD - CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD));
															#line 10648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 2691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		g_network->networkInfo.metrics.networkBusyness = std::max(busyFraction, burstiness);
															#line 2693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tracker.duration = 0;
															#line 2694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tracker.maxDuration = 0;
															#line 10656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 2663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double elapsed = now() - prevTime;
															#line 2664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		prevTime = now();
															#line 2665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		struct NetworkMetrics::PriorityStats& tracker = g_network->networkInfo.metrics.starvationTrackerNetworkBusyness;
															#line 2667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (tracker.active)
															#line 10671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracker.duration += now() - tracker.windowedTimer;
															#line 2669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracker.maxDuration = std::max(tracker.maxDuration, now() - tracker.timer);
															#line 2670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tracker.windowedTimer = now();
															#line 10679 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 2673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double busyFraction = std::min(elapsed, tracker.duration) / elapsed;
															#line 2681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double burstiness = 0;
															#line 2682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD >= 0 && CLIENT_KNOBS->BUSYNESS_SPIKE_SATURATED_THRESHOLD >= CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD)
															#line 10687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			burstiness = std::min(1.0, std::max(0.0, tracker.maxDuration - CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD) / std::max(1e-6, CLIENT_KNOBS->BUSYNESS_SPIKE_SATURATED_THRESHOLD - CLIENT_KNOBS->BUSYNESS_SPIKE_START_THRESHOLD));
															#line 10691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 2691 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		g_network->networkInfo.metrics.networkBusyness = std::max(busyFraction, burstiness);
															#line 2693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tracker.duration = 0;
															#line 2694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		tracker.maxDuration = 0;
															#line 10699 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorNetworkBusynessActor*>(this)->actor_wait_state > 0) static_cast<MonitorNetworkBusynessActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorNetworkBusynessActor*>(this)->ActorCallback< MonitorNetworkBusynessActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorNetworkBusynessActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorNetworkBusynessActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorNetworkBusynessActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorNetworkBusynessActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorNetworkBusynessActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorNetworkBusynessActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double prevTime;
															#line 10781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via monitorNetworkBusyness()
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class MonitorNetworkBusynessActor final : public Actor<Void>, public ActorCallback< MonitorNetworkBusynessActor, 0, Void >, public FastAllocated<MonitorNetworkBusynessActor>, public MonitorNetworkBusynessActorState<MonitorNetworkBusynessActor> {
															#line 10786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<MonitorNetworkBusynessActor>::operator new;
	using FastAllocated<MonitorNetworkBusynessActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7925040699012886528UL, 3188485831084249088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorNetworkBusynessActor, 0, Void >;
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	MonitorNetworkBusynessActor() 
															#line 10803 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorNetworkBusynessActorState<MonitorNetworkBusynessActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9706183877542601984UL, 10068029726573410048UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorNetworkBusynessActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorNetworkBusyness");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorNetworkBusyness", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorNetworkBusynessActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> monitorNetworkBusyness(  ) {
															#line 2659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new MonitorNetworkBusynessActor());
															#line 10836 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 2697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

static void setupGlobalKnobs() {
	IKnobCollection::setGlobalKnobCollection(IKnobCollection::Type::CLIENT, Randomize::False, IsSimulated::False);
	for (const auto& [knobName, knobValue] : networkOptions.knobs) {
		IKnobCollection::getMutableGlobalKnobCollection().setKnob(knobName, knobValue);
	}
}

// Setup g_network and start monitoring for network busyness
void setupNetwork(uint64_t transportId, UseMetrics useMetrics) {
	if (g_network)
		throw network_already_setup();

	if (!networkOptions.logClientInfo.present())
		networkOptions.logClientInfo = true;

	setupGlobalKnobs();
	g_network = newNet2(tlsConfig, false, useMetrics || networkOptions.traceDirectory.present());
	g_network->addStopCallback(Net2FileSystem::stop);
	FlowTransport::createInstance(true, transportId, WLTOKEN_RESERVED_COUNT);
	Net2FileSystem::newFileSystem();

	if (networkOptions.traceInitializeOnSetup) {
		::initializeClientTracing({}, {});
	}

	uncancellable(monitorNetworkBusyness());
}

void runNetwork() {
	if (!g_network) {
		throw network_not_setup();
	}

	if (!g_network->checkRunnable()) {
		throw network_cannot_be_restarted();
	}

	if (networkOptions.traceDirectory.present() && networkOptions.runLoopProfilingEnabled) {
		setupRunLoopProfiler();
	}

	g_network->run();

	if (networkOptions.traceDirectory.present())
		systemMonitor();
}

void stopNetwork() {
	if (!g_network)
		throw network_not_setup();

	TraceEvent("ClientStopNetwork").log();

	if (networkOptions.traceDirectory.present() && networkOptions.runLoopProfilingEnabled) {
		stopRunLoopProfiler();
	}

	g_network->stop();
}

void DatabaseContext::updateProxies() {
	if (proxiesLastChange == clientInfo->get().id)
		return;
	proxiesLastChange = clientInfo->get().id;
	commitProxies.clear();
	grvProxies.clear();
	bool commitProxyProvisional = false, grvProxyProvisional = false;
	if (clientInfo->get().commitProxies.size()) {
		commitProxies = makeReference<CommitProxyInfo>(clientInfo->get().commitProxies);
		commitProxyProvisional = clientInfo->get().commitProxies[0].provisional;
	}
	if (clientInfo->get().grvProxies.size()) {
		grvProxies = makeReference<GrvProxyInfo>(clientInfo->get().grvProxies, BalanceOnRequests::True);
		grvProxyProvisional = clientInfo->get().grvProxies[0].provisional;
	}
	if (clientInfo->get().commitProxies.size() && clientInfo->get().grvProxies.size()) {
		ASSERT(commitProxyProvisional == grvProxyProvisional);
		proxyProvisional = commitProxyProvisional;
	}
}

Reference<CommitProxyInfo> DatabaseContext::getCommitProxies(UseProvisionalProxies useProvisionalProxies) {
	updateProxies();
	if (proxyProvisional && !useProvisionalProxies) {
		return Reference<CommitProxyInfo>();
	}
	return commitProxies;
}

Reference<GrvProxyInfo> DatabaseContext::getGrvProxies(UseProvisionalProxies useProvisionalProxies) {
	updateProxies();
	if (proxyProvisional && !useProvisionalProxies) {
		return Reference<GrvProxyInfo>();
	}
	return grvProxies;
}

bool DatabaseContext::isCurrentGrvProxy(UID proxyId) const {
	for (const auto& proxy : clientInfo->get().grvProxies) {
		if (proxy.id() == proxyId)
			return true;
	}
	CODE_PROBE(true, "stale GRV proxy detected", probe::decoration::rare);
	return false;
}

// Actor which will wait until the MultiInterface<CommitProxyInterface> returned by the DatabaseContext cx is not
// nullptr
															#line 10949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getCommitProxiesFuture()
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetCommitProxiesFutureActor>
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetCommitProxiesFutureActorState {
															#line 10956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetCommitProxiesFutureActorState(DatabaseContext* const& cx,UseProvisionalProxies const& useProvisionalProxies) 
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   useProvisionalProxies(useProvisionalProxies)
															#line 10965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this));

	}
	~GetCommitProxiesFutureActorState() 
	{
		fdb_probe_actor_destroy("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 10980 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetCommitProxiesFutureActorState();
		static_cast<GetCommitProxiesFutureActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		Reference<CommitProxyInfo> commitProxies = cx->getCommitProxies(useProvisionalProxies);
															#line 2810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (commitProxies)
															#line 11012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<GetCommitProxiesFutureActor*>(this)->SAV<Reference<CommitProxyInfo>>::futures) { (void)(commitProxies); this->~GetCommitProxiesFutureActorState(); static_cast<GetCommitProxiesFutureActor*>(this)->destroy(); return 0; }
															#line 11016 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<GetCommitProxiesFutureActor*>(this)->SAV< Reference<CommitProxyInfo> >::value()) Reference<CommitProxyInfo>(commitProxies);
			this->~GetCommitProxiesFutureActorState();
			static_cast<GetCommitProxiesFutureActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = cx->onProxiesChanged();
															#line 2812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetCommitProxiesFutureActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11026 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetCommitProxiesFutureActor*>(this)->actor_wait_state = 1;
															#line 2812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetCommitProxiesFutureActor, 0, Void >*>(static_cast<GetCommitProxiesFutureActor*>(this)));
															#line 11031 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetCommitProxiesFutureActor*>(this)->actor_wait_state > 0) static_cast<GetCommitProxiesFutureActor*>(this)->actor_wait_state = 0;
		static_cast<GetCommitProxiesFutureActor*>(this)->ActorCallback< GetCommitProxiesFutureActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetCommitProxiesFutureActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCommitProxiesFutureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetCommitProxiesFutureActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCommitProxiesFutureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetCommitProxiesFutureActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetCommitProxiesFutureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UseProvisionalProxies useProvisionalProxies;
															#line 11127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getCommitProxiesFuture()
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetCommitProxiesFutureActor final : public Actor<Reference<CommitProxyInfo>>, public ActorCallback< GetCommitProxiesFutureActor, 0, Void >, public FastAllocated<GetCommitProxiesFutureActor>, public GetCommitProxiesFutureActorState<GetCommitProxiesFutureActor> {
															#line 11132 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetCommitProxiesFutureActor>::operator new;
	using FastAllocated<GetCommitProxiesFutureActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10327499880685981696UL, 2458955293967973888UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Reference<CommitProxyInfo>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetCommitProxiesFutureActor, 0, Void >;
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetCommitProxiesFutureActor(DatabaseContext* const& cx,UseProvisionalProxies const& useProvisionalProxies) 
															#line 11149 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Reference<CommitProxyInfo>>(),
		   GetCommitProxiesFutureActorState<GetCommitProxiesFutureActor>(cx, useProvisionalProxies),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5140351585656034048UL, 11319334858847052544UL);
		ActorExecutionContextHelper __helper(static_cast<GetCommitProxiesFutureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getCommitProxiesFuture");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getCommitProxiesFuture", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetCommitProxiesFutureActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Reference<CommitProxyInfo>> getCommitProxiesFuture( DatabaseContext* const& cx, UseProvisionalProxies const& useProvisionalProxies ) {
															#line 2806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Reference<CommitProxyInfo>>(new GetCommitProxiesFutureActor(cx, useProvisionalProxies));
															#line 11182 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 2815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// Returns a future which will not be set until the CommitProxyInfo of this DatabaseContext is not nullptr
Future<Reference<CommitProxyInfo>> DatabaseContext::getCommitProxiesFuture(
    UseProvisionalProxies useProvisionalProxies) {
	return ::getCommitProxiesFuture(this, useProvisionalProxies);
}

void GetRangeLimits::decrement(VectorRef<KeyValueRef> const& data) {
	if (rows != GetRangeLimits::ROW_LIMIT_UNLIMITED) {
		ASSERT(data.size() <= rows);
		rows -= data.size();
	}

	minRows = std::max(0, minRows - data.size());

	if (bytes != GetRangeLimits::BYTE_LIMIT_UNLIMITED)
		bytes = std::max(0, bytes - (int)data.expectedSize() - (8 - (int)sizeof(KeyValueRef)) * data.size());
}

void GetRangeLimits::decrement(KeyValueRef const& data) {
	minRows = std::max(0, minRows - 1);
	if (rows != GetRangeLimits::ROW_LIMIT_UNLIMITED)
		rows--;
	if (bytes != GetRangeLimits::BYTE_LIMIT_UNLIMITED)
		bytes = std::max(0, bytes - (int)8 - (int)data.expectedSize());
}

void GetRangeLimits::decrement(VectorRef<MappedKeyValueRef> const& data) {
	if (rows != GetRangeLimits::ROW_LIMIT_UNLIMITED) {
		ASSERT(data.size() <= rows);
		rows -= data.size();
	}

	minRows = std::max(0, minRows - data.size());

	// TODO: For now, expectedSize only considers the size of the original key values, but not the underlying queries or
	// results. Also, double check it is correct when dealing with sizeof(MappedKeyValueRef).
	if (bytes != GetRangeLimits::BYTE_LIMIT_UNLIMITED)
		bytes = std::max(0, bytes - (int)data.expectedSize() - (8 - (int)sizeof(MappedKeyValueRef)) * data.size());
}

void GetRangeLimits::decrement(MappedKeyValueRef const& data) {
	minRows = std::max(0, minRows - 1);
	if (rows != GetRangeLimits::ROW_LIMIT_UNLIMITED)
		rows--;
	// TODO: For now, expectedSize only considers the size of the original key values, but not the underlying queries or
	// results. Also, double check it is correct when dealing with sizeof(MappedKeyValueRef).
	if (bytes != GetRangeLimits::BYTE_LIMIT_UNLIMITED)
		bytes = std::max(0, bytes - (int)8 - (int)data.expectedSize());
}

// True if either the row or byte limit has been reached
bool GetRangeLimits::isReached() const {
	return rows == 0 || (bytes == 0 && minRows == 0);
}

// True if data would cause the row or byte limit to be reached
bool GetRangeLimits::reachedBy(VectorRef<KeyValueRef> const& data) const {
	return (rows != GetRangeLimits::ROW_LIMIT_UNLIMITED && data.size() >= rows) ||
	       (bytes != GetRangeLimits::BYTE_LIMIT_UNLIMITED &&
	        (int)data.expectedSize() + (8 - (int)sizeof(KeyValueRef)) * data.size() >= bytes && data.size() >= minRows);
}

bool GetRangeLimits::hasByteLimit() const {
	return bytes != GetRangeLimits::BYTE_LIMIT_UNLIMITED;
}

bool GetRangeLimits::hasRowLimit() const {
	return rows != GetRangeLimits::ROW_LIMIT_UNLIMITED;
}

bool GetRangeLimits::hasSatisfiedMinRows() const {
	return hasByteLimit() && minRows == 0;
}

AddressExclusion AddressExclusion::parse(StringRef const& key) {
	// Must not change: serialized to the database!
	auto parsedIp = IPAddress::parse(key.toString());
	if (parsedIp.present()) {
		return AddressExclusion(parsedIp.get());
	}

	// Not a whole machine, includes `port'.
	try {
		auto addr = NetworkAddress::parse(key.toString());
		if (addr.isTLS()) {
			TraceEvent(SevWarnAlways, "AddressExclusionParseError")
			    .detail("String", key)
			    .detail("Description", "Address inclusion string should not include `:tls' suffix.");
			return AddressExclusion();
		}
		return AddressExclusion(addr.ip, addr.port);
	} catch (Error&) {
		TraceEvent(SevWarnAlways, "AddressExclusionParseError").detail("String", key);
		return AddressExclusion();
	}
}

Tenant::Tenant(Database cx, TenantName name) : idFuture(cx->lookupTenant(name)), name(name) {}
Tenant::Tenant(int64_t id) : idFuture(id) {}
Tenant::Tenant(Future<int64_t> id, Optional<TenantName> name) : idFuture(id), name(name) {}

int64_t Tenant::id() const {
	ASSERT(idFuture.isReady());
	return idFuture.get();
}

Future<int64_t> Tenant::getIdFuture() const {
	return idFuture;
}

KeyRef Tenant::prefix() const {
	ASSERT(idFuture.isReady());
	if (bigEndianId == -1) {
		bigEndianId = bigEndian64(idFuture.get());
	}
	return StringRef(reinterpret_cast<const uint8_t*>(&bigEndianId), TenantAPI::PREFIX_SIZE);
}

std::string Tenant::description() const {
	StringRef nameStr = name.castTo<TenantNameRef>().orDefault("<unspecified>"_sr);
	if (idFuture.canGet()) {
		return format("%.*s (%lld)", nameStr.size(), nameStr.begin(), idFuture.get());
	} else {
		return format("%.*s", nameStr.size(), nameStr.begin());
	}
}

Future<Optional<Value>> getValue(Reference<TransactionState> const& trState,
                                 Key const& key,
                                 UseTenant const& useTenant = UseTenant::True,
                                 TransactionRecordLogInfo const& recordLogInfo = TransactionRecordLogInfo::True);

Future<RangeResult> getRange(Reference<TransactionState> const& trState,
                             KeySelector const& begin,
                             KeySelector const& end,
                             GetRangeLimits const& limits,
                             Reverse const& reverse,
                             UseTenant const& useTenant);

															#line 11326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchServerInterface()
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class FetchServerInterfaceActor>
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class FetchServerInterfaceActorState {
															#line 11333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	FetchServerInterfaceActorState(Reference<TransactionState> const& trState,UID const& id) 
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState),
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   id(id)
															#line 11342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchServerInterface", reinterpret_cast<unsigned long>(this));

	}
	~FetchServerInterfaceActorState() 
	{
		fdb_probe_actor_destroy("fetchServerInterface", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = getValue(trState, serverListKeyFor(id), UseTenant::False, TransactionRecordLogInfo::False);
															#line 2956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<FetchServerInterfaceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FetchServerInterfaceActor*>(this)->actor_wait_state = 1;
															#line 2956 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >*>(static_cast<FetchServerInterfaceActor*>(this)));
															#line 11364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchServerInterfaceActorState();
		static_cast<FetchServerInterfaceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& val,int loopDepth) 
	{
															#line 2959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!val.present())
															#line 11387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<FetchServerInterfaceActor*>(this)->SAV<Optional<StorageServerInterface>>::futures) { (void)(Optional<StorageServerInterface>()); this->~FetchServerInterfaceActorState(); static_cast<FetchServerInterfaceActor*>(this)->destroy(); return 0; }
															#line 11391 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<FetchServerInterfaceActor*>(this)->SAV< Optional<StorageServerInterface> >::value()) Optional<StorageServerInterface>(Optional<StorageServerInterface>());
			this->~FetchServerInterfaceActorState();
			static_cast<FetchServerInterfaceActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<FetchServerInterfaceActor*>(this)->SAV<Optional<StorageServerInterface>>::futures) { (void)(decodeServerListValue(val.get())); this->~FetchServerInterfaceActorState(); static_cast<FetchServerInterfaceActor*>(this)->destroy(); return 0; }
															#line 11399 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<FetchServerInterfaceActor*>(this)->SAV< Optional<StorageServerInterface> >::value()) Optional<StorageServerInterface>(decodeServerListValue(val.get()));
		this->~FetchServerInterfaceActorState();
		static_cast<FetchServerInterfaceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && val,int loopDepth) 
	{
															#line 2959 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!val.present())
															#line 11411 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 2961 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<FetchServerInterfaceActor*>(this)->SAV<Optional<StorageServerInterface>>::futures) { (void)(Optional<StorageServerInterface>()); this->~FetchServerInterfaceActorState(); static_cast<FetchServerInterfaceActor*>(this)->destroy(); return 0; }
															#line 11415 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<FetchServerInterfaceActor*>(this)->SAV< Optional<StorageServerInterface> >::value()) Optional<StorageServerInterface>(Optional<StorageServerInterface>());
			this->~FetchServerInterfaceActorState();
			static_cast<FetchServerInterfaceActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<FetchServerInterfaceActor*>(this)->SAV<Optional<StorageServerInterface>>::futures) { (void)(decodeServerListValue(val.get())); this->~FetchServerInterfaceActorState(); static_cast<FetchServerInterfaceActor*>(this)->destroy(); return 0; }
															#line 11423 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<FetchServerInterfaceActor*>(this)->SAV< Optional<StorageServerInterface> >::value()) Optional<StorageServerInterface>(decodeServerListValue(val.get()));
		this->~FetchServerInterfaceActorState();
		static_cast<FetchServerInterfaceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont1(val, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchServerInterfaceActor*>(this)->actor_wait_state > 0) static_cast<FetchServerInterfaceActor*>(this)->actor_wait_state = 0;
		static_cast<FetchServerInterfaceActor*>(this)->ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("fetchServerInterface", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchServerInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchServerInterface", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("fetchServerInterface", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchServerInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchServerInterface", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("fetchServerInterface", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchServerInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchServerInterface", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UID id;
															#line 11510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via fetchServerInterface()
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class FetchServerInterfaceActor final : public Actor<Optional<StorageServerInterface>>, public ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >, public FastAllocated<FetchServerInterfaceActor>, public FetchServerInterfaceActorState<FetchServerInterfaceActor> {
															#line 11515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<FetchServerInterfaceActor>::operator new;
	using FastAllocated<FetchServerInterfaceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6306903960321807872UL, 14778102724244047360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<StorageServerInterface>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >;
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	FetchServerInterfaceActor(Reference<TransactionState> const& trState,UID const& id) 
															#line 11532 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Optional<StorageServerInterface>>(),
		   FetchServerInterfaceActorState<FetchServerInterfaceActor>(trState, id),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchServerInterface", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14331509920591601920UL, 209697105602724096UL);
		ActorExecutionContextHelper __helper(static_cast<FetchServerInterfaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchServerInterface");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchServerInterface", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchServerInterfaceActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Optional<StorageServerInterface>> fetchServerInterface( Reference<TransactionState> const& trState, UID const& id ) {
															#line 2955 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Optional<StorageServerInterface>>(new FetchServerInterfaceActor(trState, id));
															#line 11565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 2966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 11570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via transactionalGetServerInterfaces()
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class TransactionalGetServerInterfacesActor>
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class TransactionalGetServerInterfacesActorState {
															#line 11577 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TransactionalGetServerInterfacesActorState(Reference<TransactionState> const& trState,std::vector<UID> const& ids) 
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState),
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   ids(ids),
															#line 2970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   serverListEntries()
															#line 11588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this));

	}
	~TransactionalGetServerInterfacesActorState() 
	{
		fdb_probe_actor_destroy("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			serverListEntries.reserve(ids.size());
															#line 2972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			for(int s = 0;s < ids.size();s++) {
															#line 2973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				serverListEntries.push_back(fetchServerInterface(trState, ids[s]));
															#line 11607 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 2976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<std::vector<Optional<StorageServerInterface>>> __when_expr_0 = getAll(serverListEntries);
															#line 2976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<TransactionalGetServerInterfacesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TransactionalGetServerInterfacesActor*>(this)->actor_wait_state = 1;
															#line 2976 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >*>(static_cast<TransactionalGetServerInterfacesActor*>(this)));
															#line 11618 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TransactionalGetServerInterfacesActorState();
		static_cast<TransactionalGetServerInterfacesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<Optional<StorageServerInterface>> const& serverListValues,int loopDepth) 
	{
															#line 2977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::vector<StorageServerInterface> serverInterfaces;
															#line 2978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for(int s = 0;s < serverListValues.size();s++) {
															#line 2979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!serverListValues[s].present())
															#line 11645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 2981 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (!static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV<Optional<std::vector<StorageServerInterface>>>::futures) { (void)(Optional<std::vector<StorageServerInterface>>()); this->~TransactionalGetServerInterfacesActorState(); static_cast<TransactionalGetServerInterfacesActor*>(this)->destroy(); return 0; }
															#line 11649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				new (&static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV< Optional<std::vector<StorageServerInterface>> >::value()) Optional<std::vector<StorageServerInterface>>(Optional<std::vector<StorageServerInterface>>());
				this->~TransactionalGetServerInterfacesActorState();
				static_cast<TransactionalGetServerInterfacesActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			serverInterfaces.push_back(serverListValues[s].get());
															#line 11657 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 2985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV<Optional<std::vector<StorageServerInterface>>>::futures) { (void)(serverInterfaces); this->~TransactionalGetServerInterfacesActorState(); static_cast<TransactionalGetServerInterfacesActor*>(this)->destroy(); return 0; }
															#line 11661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV< Optional<std::vector<StorageServerInterface>> >::value()) Optional<std::vector<StorageServerInterface>>(serverInterfaces);
		this->~TransactionalGetServerInterfacesActorState();
		static_cast<TransactionalGetServerInterfacesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<Optional<StorageServerInterface>> && serverListValues,int loopDepth) 
	{
															#line 2977 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		std::vector<StorageServerInterface> serverInterfaces;
															#line 2978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for(int s = 0;s < serverListValues.size();s++) {
															#line 2979 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!serverListValues[s].present())
															#line 11677 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 2981 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (!static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV<Optional<std::vector<StorageServerInterface>>>::futures) { (void)(Optional<std::vector<StorageServerInterface>>()); this->~TransactionalGetServerInterfacesActorState(); static_cast<TransactionalGetServerInterfacesActor*>(this)->destroy(); return 0; }
															#line 11681 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				new (&static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV< Optional<std::vector<StorageServerInterface>> >::value()) Optional<std::vector<StorageServerInterface>>(Optional<std::vector<StorageServerInterface>>());
				this->~TransactionalGetServerInterfacesActorState();
				static_cast<TransactionalGetServerInterfacesActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			serverInterfaces.push_back(serverListValues[s].get());
															#line 11689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 2985 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV<Optional<std::vector<StorageServerInterface>>>::futures) { (void)(serverInterfaces); this->~TransactionalGetServerInterfacesActorState(); static_cast<TransactionalGetServerInterfacesActor*>(this)->destroy(); return 0; }
															#line 11693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<TransactionalGetServerInterfacesActor*>(this)->SAV< Optional<std::vector<StorageServerInterface>> >::value()) Optional<std::vector<StorageServerInterface>>(serverInterfaces);
		this->~TransactionalGetServerInterfacesActorState();
		static_cast<TransactionalGetServerInterfacesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<Optional<StorageServerInterface>> const& serverListValues,int loopDepth) 
	{
		loopDepth = a_body1cont1(serverListValues, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<Optional<StorageServerInterface>> && serverListValues,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(serverListValues), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TransactionalGetServerInterfacesActor*>(this)->actor_wait_state > 0) static_cast<TransactionalGetServerInterfacesActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionalGetServerInterfacesActor*>(this)->ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >::remove();

	}
	void a_callback_fire(ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >*,std::vector<Optional<StorageServerInterface>> const& value) 
	{
		fdb_probe_actor_enter("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionalGetServerInterfacesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >*,std::vector<Optional<StorageServerInterface>> && value) 
	{
		fdb_probe_actor_enter("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionalGetServerInterfacesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >*,Error err) 
	{
		fdb_probe_actor_enter("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionalGetServerInterfacesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<UID> ids;
															#line 2970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<Future<Optional<StorageServerInterface>>> serverListEntries;
															#line 11782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via transactionalGetServerInterfaces()
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class TransactionalGetServerInterfacesActor final : public Actor<Optional<std::vector<StorageServerInterface>>>, public ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >, public FastAllocated<TransactionalGetServerInterfacesActor>, public TransactionalGetServerInterfacesActorState<TransactionalGetServerInterfacesActor> {
															#line 11787 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<TransactionalGetServerInterfacesActor>::operator new;
	using FastAllocated<TransactionalGetServerInterfacesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13924699364048429312UL, 3188752299979864832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<std::vector<StorageServerInterface>>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >;
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TransactionalGetServerInterfacesActor(Reference<TransactionState> const& trState,std::vector<UID> const& ids) 
															#line 11804 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Optional<std::vector<StorageServerInterface>>>(),
		   TransactionalGetServerInterfacesActorState<TransactionalGetServerInterfacesActor>(trState, ids),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8960874033469204480UL, 15207940511002765056UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionalGetServerInterfacesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("transactionalGetServerInterfaces");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("transactionalGetServerInterfaces", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TransactionalGetServerInterfacesActor, 0, std::vector<Optional<StorageServerInterface>> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Optional<std::vector<StorageServerInterface>>> transactionalGetServerInterfaces( Reference<TransactionState> const& trState, std::vector<UID> const& ids ) {
															#line 2967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Optional<std::vector<StorageServerInterface>>>(new TransactionalGetServerInterfacesActor(trState, ids));
															#line 11837 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 2987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

void updateTssMappings(Database cx, const GetKeyServerLocationsReply& reply) {
	// Since a ss -> tss mapping is included in resultsTssMapping iff that SS is in results and has a tss pair,
	// all SS in results that do not have a mapping present must not have a tss pair.
	std::unordered_map<UID, const StorageServerInterface*> ssiById;
	for (const auto& [_, shard] : reply.results) {
		for (auto& ssi : shard) {
			ssiById[ssi.id()] = &ssi;
		}
	}

	for (const auto& mapping : reply.resultsTssMapping) {
		auto ssi = ssiById.find(mapping.first);
		ASSERT(ssi != ssiById.end());
		cx->addTssMapping(*ssi->second, mapping.second);
		ssiById.erase(mapping.first);
	}

	// if SS didn't have a mapping above, it's still in the ssiById map, so remove its tss mapping
	for (const auto& it : ssiById) {
		cx->removeTssMapping(*it.second);
	}
}

void updateTagMappings(Database cx, const GetKeyServerLocationsReply& reply) {
	for (const auto& mapping : reply.resultsTagMapping) {
		cx->addSSIdTagMapping(mapping.first, mapping.second);
	}
}

// If isBackward == true, returns the shard containing the key before 'key' (an infinitely long, inexpressible key).
// Otherwise returns the shard containing key
															#line 11873 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getKeyLocation_internal()
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetKeyLocation_internalActor>
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetKeyLocation_internalActorState {
															#line 11880 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyLocation_internalActorState(Database const& cx,TenantInfo const& tenant,Key const& key,SpanContext const& spanContext,Optional<UID> const& debugID,UseProvisionalProxies const& useProvisionalProxies,Reverse const& isBackward,Version const& version) 
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tenant(tenant),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   key(key),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   spanContext(spanContext),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   debugID(debugID),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   useProvisionalProxies(useProvisionalProxies),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   isBackward(isBackward),
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   version(version),
															#line 3028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   span("NAPI:getKeyLocation"_loc, spanContext)
															#line 11903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getKeyLocation_internal", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyLocation_internalActorState() 
	{
		fdb_probe_actor_destroy("getKeyLocation_internal", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (isBackward)
															#line 11918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3030 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				ASSERT(key != allKeys.begin && key <= allKeys.end);
															#line 11922 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 3032 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				ASSERT(key < allKeys.end);
															#line 11928 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3035 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (debugID.present())
															#line 11932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3036 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", debugID.get().first(), "NativeAPI.getKeyLocation.Before");
															#line 11936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 11940 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyLocation_internalActorState();
		static_cast<GetKeyLocation_internalActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 3040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = cx->getBackoff();
															#line 3040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state = 1;
															#line 3040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyLocation_internalActor, 0, Void >*>(static_cast<GetKeyLocation_internalActor*>(this)));
															#line 11978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3066 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_commit_proxy_memory_limit_exceeded)
															#line 12000 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3068 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				TraceEvent(SevWarnAlways, "CommitProxyOverloadedForKeyLocation").suppressFor(5);
															#line 3069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->updateBackoff(e);
															#line 12006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				return a_body1loopHead1(loopDepth); // continue
			}
															#line 3073 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 12011 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 3041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequests;
															#line 3043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = cx->onProxiesChanged();
															#line 3042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12029 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetKeyServerLocationsReply> __when_expr_2 = basicLoadBalance( cx->getCommitProxies(useProvisionalProxies), &CommitProxyInterface::getKeyServersLocations, GetKeyServerLocationsRequest( span.context, tenant, key, Optional<KeyRef>(), 100, isBackward, version, key.arena()), TaskPriority::DefaultPromiseEndpoint);
															#line 12033 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when2(__when_expr_2.get(), loopDepth); };
		static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state = 2;
															#line 3043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyLocation_internalActor, 1, Void >*>(static_cast<GetKeyLocation_internalActor*>(this)));
															#line 3044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >*>(static_cast<GetKeyLocation_internalActor*>(this)));
															#line 12040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 3041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequests;
															#line 3043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = cx->onProxiesChanged();
															#line 3042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetKeyServerLocationsReply> __when_expr_2 = basicLoadBalance( cx->getCommitProxies(useProvisionalProxies), &CommitProxyInterface::getKeyServersLocations, GetKeyServerLocationsRequest( span.context, tenant, key, Optional<KeyRef>(), 100, isBackward, version, key.arena()), TaskPriority::DefaultPromiseEndpoint);
															#line 12057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when2(__when_expr_2.get(), loopDepth); };
		static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state = 2;
															#line 3043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyLocation_internalActor, 1, Void >*>(static_cast<GetKeyLocation_internalActor*>(this)));
															#line 3044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >*>(static_cast<GetKeyLocation_internalActor*>(this)));
															#line 12064 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state > 0) static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyLocation_internalActor*>(this)->ActorCallback< GetKeyLocation_internalActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyLocation_internalActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyLocation_internalActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyLocation_internalActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2(GetKeyServerLocationsReply const& rep,int loopDepth) 
	{
															#line 3050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequestsCompleted;
															#line 3051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (debugID.present())
															#line 12168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "NativeAPI.getKeyLocation.After");
															#line 12172 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(rep.results.size() == 1);
															#line 3056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		auto locationInfo = cx->setCachedLocation(rep.results[0].first, rep.results[0].second);
															#line 3057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		updateTssMappings(cx, rep);
															#line 3058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		updateTagMappings(cx, rep);
															#line 3060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->updateBackoff(success());
															#line 3061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetKeyLocation_internalActor*>(this)->SAV<KeyRangeLocationInfo>::futures) { (void)(KeyRangeLocationInfo( KeyRange(toPrefixRelativeRange(rep.results[0].first, tenant.prefix), rep.arena), locationInfo)); this->~GetKeyLocation_internalActorState(); static_cast<GetKeyLocation_internalActor*>(this)->destroy(); return 0; }
															#line 12186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetKeyLocation_internalActor*>(this)->SAV< KeyRangeLocationInfo >::value()) KeyRangeLocationInfo(KeyRangeLocationInfo( KeyRange(toPrefixRelativeRange(rep.results[0].first, tenant.prefix), rep.arena), locationInfo));
		this->~GetKeyLocation_internalActorState();
		static_cast<GetKeyLocation_internalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when2(GetKeyServerLocationsReply && rep,int loopDepth) 
	{
															#line 3050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequestsCompleted;
															#line 3051 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (debugID.present())
															#line 12200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "NativeAPI.getKeyLocation.After");
															#line 12204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3054 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(rep.results.size() == 1);
															#line 3056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		auto locationInfo = cx->setCachedLocation(rep.results[0].first, rep.results[0].second);
															#line 3057 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		updateTssMappings(cx, rep);
															#line 3058 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		updateTagMappings(cx, rep);
															#line 3060 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->updateBackoff(success());
															#line 3061 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetKeyLocation_internalActor*>(this)->SAV<KeyRangeLocationInfo>::futures) { (void)(KeyRangeLocationInfo( KeyRange(toPrefixRelativeRange(rep.results[0].first, tenant.prefix), rep.arena), locationInfo)); this->~GetKeyLocation_internalActorState(); static_cast<GetKeyLocation_internalActor*>(this)->destroy(); return 0; }
															#line 12218 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetKeyLocation_internalActor*>(this)->SAV< KeyRangeLocationInfo >::value()) KeyRangeLocationInfo(KeyRangeLocationInfo( KeyRange(toPrefixRelativeRange(rep.results[0].first, tenant.prefix), rep.arena), locationInfo));
		this->~GetKeyLocation_internalActorState();
		static_cast<GetKeyLocation_internalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state > 0) static_cast<GetKeyLocation_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyLocation_internalActor*>(this)->ActorCallback< GetKeyLocation_internalActor, 1, Void >::remove();
		static_cast<GetKeyLocation_internalActor*>(this)->ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyLocation_internalActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyLocation_internalActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyLocation_internalActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply const& value) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply && value) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Database cx;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TenantInfo tenant;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Key key;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	SpanContext spanContext;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<UID> debugID;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UseProvisionalProxies useProvisionalProxies;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reverse isBackward;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Version version;
															#line 3028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 12378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getKeyLocation_internal()
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetKeyLocation_internalActor final : public Actor<KeyRangeLocationInfo>, public ActorCallback< GetKeyLocation_internalActor, 0, Void >, public ActorCallback< GetKeyLocation_internalActor, 1, Void >, public ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >, public FastAllocated<GetKeyLocation_internalActor>, public GetKeyLocation_internalActorState<GetKeyLocation_internalActor> {
															#line 12383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetKeyLocation_internalActor>::operator new;
	using FastAllocated<GetKeyLocation_internalActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10330905200804568576UL, 14536883190458859008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<KeyRangeLocationInfo>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyLocation_internalActor, 0, Void >;
friend struct ActorCallback< GetKeyLocation_internalActor, 1, Void >;
friend struct ActorCallback< GetKeyLocation_internalActor, 2, GetKeyServerLocationsReply >;
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyLocation_internalActor(Database const& cx,TenantInfo const& tenant,Key const& key,SpanContext const& spanContext,Optional<UID> const& debugID,UseProvisionalProxies const& useProvisionalProxies,Reverse const& isBackward,Version const& version) 
															#line 12402 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<KeyRangeLocationInfo>(),
		   GetKeyLocation_internalActorState<GetKeyLocation_internalActor>(cx, tenant, key, spanContext, debugID, useProvisionalProxies, isBackward, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1277245642794616832UL, 6556686229128886016UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyLocation_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKeyLocation_internal");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKeyLocation_internal", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyLocation_internalActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyLocation_internalActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<KeyRangeLocationInfo> getKeyLocation_internal( Database const& cx, TenantInfo const& tenant, Key const& key, SpanContext const& spanContext, Optional<UID> const& debugID, UseProvisionalProxies const& useProvisionalProxies, Reverse const& isBackward, Version const& version ) {
															#line 3019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<KeyRangeLocationInfo>(new GetKeyLocation_internalActor(cx, tenant, key, spanContext, debugID, useProvisionalProxies, isBackward, version));
															#line 12436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3077 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// Checks if `endpoint` is failed on a healthy server or not. Returns true if we need to refresh the location cache for
// the endpoint.
bool checkOnlyEndpointFailed(const Database& cx, const Endpoint& endpoint) {
	if (IFailureMonitor::failureMonitor().onlyEndpointFailed(endpoint)) {
		// This endpoint is failed, but the server is still healthy. There are two cases this can happen:
		//    - There is a recent bounce in the cluster where the endpoints in SSes get updated.
		//    - The SS is failed and terminated on a server, but the server is kept running.
		// To account for the first case, we invalidate the cache and issue GetKeyLocation requests to the proxy to
		// update the cache with the new SS points. However, if the failure is caused by the second case, the
		// requested key location will continue to be the failed endpoint until the data movement is finished. But
		// every read will generate a GetKeyLocation request to the proxies (and still getting the failed endpoint
		// back), which may overload the proxy and affect data movement speed. Therefore, we only refresh the
		// location cache for short period of time, and after the initial grace period that we keep retrying
		// resolving key location, we will slow it down to resolve it only once every
		// `LOCATION_CACHE_FAILED_ENDPOINT_RETRY_INTERVAL`.
		cx->setFailedEndpointOnHealthyServer(endpoint);
		const auto& failureInfo = cx->getEndpointFailureInfo(endpoint);
		ASSERT(failureInfo.present());
		if (now() - failureInfo.get().startTime < CLIENT_KNOBS->LOCATION_CACHE_ENDPOINT_FAILURE_GRACE_PERIOD ||
		    now() - failureInfo.get().lastRefreshTime > CLIENT_KNOBS->LOCATION_CACHE_FAILED_ENDPOINT_RETRY_INTERVAL) {
			cx->updateFailedEndpointRefreshTime(endpoint);
			return true;
		}
	} else {
		cx->clearFailedEndpointOnHealthyServer(endpoint);
	}
	return false;
}

template <class F>
Future<KeyRangeLocationInfo> getKeyLocation(Database const& cx,
                                            TenantInfo const& tenant,
                                            Key const& key,
                                            F StorageServerInterface::* member,
                                            SpanContext spanContext,
                                            Optional<UID> debugID,
                                            UseProvisionalProxies useProvisionalProxies,
                                            Reverse isBackward,
                                            Version version) {
	// we first check whether this range is cached
	Optional<KeyRangeLocationInfo> locationInfo = cx->getCachedLocation(tenant, key, isBackward);
	if (!locationInfo.present()) {
		return getKeyLocation_internal(
		    cx, tenant, key, spanContext, debugID, useProvisionalProxies, isBackward, version);
	}

	bool onlyEndpointFailedAndNeedRefresh = false;
	for (int i = 0; i < locationInfo.get().locations->size(); i++) {
		if (checkOnlyEndpointFailed(cx, locationInfo.get().locations->get(i, member).getEndpoint())) {
			onlyEndpointFailedAndNeedRefresh = true;
		}
	}

	if (onlyEndpointFailedAndNeedRefresh) {
		cx->invalidateCache(tenant.prefix, key);

		// Refresh the cache with a new getKeyLocations made to proxies.
		return getKeyLocation_internal(
		    cx, tenant, key, spanContext, debugID, useProvisionalProxies, isBackward, version);
	}

	return locationInfo.get();
}

template <class F>
Future<KeyRangeLocationInfo> getKeyLocation(Reference<TransactionState> trState,
                                            Key const& key,
                                            F StorageServerInterface::* member,
                                            Reverse isBackward,
                                            UseTenant useTenant) {
	CODE_PROBE(!useTenant, "Get key location ignoring tenant");
	return getKeyLocation(trState->cx,
	                      useTenant ? trState->getTenantInfo() : TenantInfo(),
	                      key,
	                      member,
	                      trState->spanContext,
	                      trState->readOptions.present() ? trState->readOptions.get().debugID : Optional<UID>(),
	                      trState->useProvisionalProxies,
	                      isBackward,
	                      trState->readVersionFuture.isValid() && trState->readVersionFuture.isReady()
	                          ? trState->readVersion()
	                          : latestVersion);
}

void DatabaseContext::updateBackoff(const Error& err) {
	switch (err.code()) {
	case error_code_success:
		backoffDelay = backoffDelay / CLIENT_KNOBS->BACKOFF_GROWTH_RATE;
		if (backoffDelay < CLIENT_KNOBS->DEFAULT_BACKOFF) {
			backoffDelay = 0.0;
		}
		break;

	case error_code_commit_proxy_memory_limit_exceeded:
		++transactionsResourceConstrained;
		if (backoffDelay == 0.0) {
			backoffDelay = CLIENT_KNOBS->DEFAULT_BACKOFF;
		} else {
			backoffDelay = std::min(backoffDelay * CLIENT_KNOBS->BACKOFF_GROWTH_RATE,
			                        CLIENT_KNOBS->RESOURCE_CONSTRAINED_MAX_BACKOFF);
		}
		break;

	default:
		ASSERT_WE_THINK(false);
	}
}

															#line 12549 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getKeyRangeLocations_internal()
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetKeyRangeLocations_internalActor>
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetKeyRangeLocations_internalActorState {
															#line 12556 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyRangeLocations_internalActorState(Database const& cx,TenantInfo const& tenant,KeyRange const& keys,int const& limit,Reverse const& reverse,SpanContext const& spanContext,Optional<UID> const& debugID,UseProvisionalProxies const& useProvisionalProxies,Version const& version) 
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tenant(tenant),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   keys(keys),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   limit(limit),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   reverse(reverse),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   spanContext(spanContext),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   debugID(debugID),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   useProvisionalProxies(useProvisionalProxies),
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   version(version),
															#line 3196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   span("NAPI:getKeyRangeLocations"_loc, spanContext)
															#line 12581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyRangeLocations_internalActorState() 
	{
		fdb_probe_actor_destroy("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3197 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (debugID.present())
															#line 12596 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3198 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", debugID.get().first(), "NativeAPI.getKeyLocations.Before");
															#line 12600 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 12604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyRangeLocations_internalActorState();
		static_cast<GetKeyRangeLocations_internalActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 3202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = cx->getBackoff();
															#line 3202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 1;
															#line 3202 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >*>(static_cast<GetKeyRangeLocations_internalActor*>(this)));
															#line 12642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3237 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_commit_proxy_memory_limit_exceeded)
															#line 12664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3239 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				TraceEvent(SevWarnAlways, "CommitProxyOverloadedForRangeLocation").suppressFor(5);
															#line 3240 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->updateBackoff(e);
															#line 12670 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				return a_body1loopHead1(loopDepth); // continue
			}
															#line 3244 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 12675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 3203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequests;
															#line 3205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = cx->onProxiesChanged();
															#line 3204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetKeyServerLocationsReply> __when_expr_2 = basicLoadBalance( cx->getCommitProxies(useProvisionalProxies), &CommitProxyInterface::getKeyServersLocations, GetKeyServerLocationsRequest( span.context, tenant, keys.begin, keys.end, limit, reverse, version, keys.arena()), TaskPriority::DefaultPromiseEndpoint);
															#line 12697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when2(__when_expr_2.get(), loopDepth); };
		static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 2;
															#line 3205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >*>(static_cast<GetKeyRangeLocations_internalActor*>(this)));
															#line 3206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >*>(static_cast<GetKeyRangeLocations_internalActor*>(this)));
															#line 12704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 3203 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequests;
															#line 3205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = cx->onProxiesChanged();
															#line 3204 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetKeyServerLocationsReply> __when_expr_2 = basicLoadBalance( cx->getCommitProxies(useProvisionalProxies), &CommitProxyInterface::getKeyServersLocations, GetKeyServerLocationsRequest( span.context, tenant, keys.begin, keys.end, limit, reverse, version, keys.arena()), TaskPriority::DefaultPromiseEndpoint);
															#line 12721 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when2(__when_expr_2.get(), loopDepth); };
		static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 2;
															#line 3205 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >*>(static_cast<GetKeyRangeLocations_internalActor*>(this)));
															#line 3206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >*>(static_cast<GetKeyRangeLocations_internalActor*>(this)));
															#line 12728 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state > 0) static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyRangeLocations_internalActor*>(this)->ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2(GetKeyServerLocationsReply const& _rep,int loopDepth) 
	{
															#line 3212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequestsCompleted;
															#line 3213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		rep = _rep;
															#line 3214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (debugID.present())
															#line 12834 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "NativeAPI.getKeyLocations.After");
															#line 12838 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(rep.results.size());
															#line 3219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		results = std::vector<KeyRangeLocationInfo>();
															#line 3220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		shard = 0;
															#line 3221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 12848 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2when2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2(GetKeyServerLocationsReply && _rep,int loopDepth) 
	{
															#line 3212 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionKeyServerLocationRequestsCompleted;
															#line 3213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		rep = _rep;
															#line 3214 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (debugID.present())
															#line 12861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3215 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "NativeAPI.getKeyLocations.After");
															#line 12865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3217 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(rep.results.size());
															#line 3219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		results = std::vector<KeyRangeLocationInfo>();
															#line 3220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		shard = 0;
															#line 3221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 12875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2when2loopHead1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state > 0) static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyRangeLocations_internalActor*>(this)->ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >::remove();
		static_cast<GetKeyRangeLocations_internalActor*>(this)->ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont2when2cont1(int loopDepth) 
	{
															#line 3229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		updateTssMappings(cx, rep);
															#line 3230 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		updateTagMappings(cx, rep);
															#line 3232 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->updateBackoff(success());
															#line 3233 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetKeyRangeLocations_internalActor*>(this)->SAV<std::vector<KeyRangeLocationInfo>>::futures) { (void)(results); this->~GetKeyRangeLocations_internalActorState(); static_cast<GetKeyRangeLocations_internalActor*>(this)->destroy(); return 0; }
															#line 12954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetKeyRangeLocations_internalActor*>(this)->SAV< std::vector<KeyRangeLocationInfo> >::value()) std::vector<KeyRangeLocationInfo>(std::move(results)); // state_var_RVO
		this->~GetKeyRangeLocations_internalActorState();
		static_cast<GetKeyRangeLocations_internalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2when2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2loopBody1(int loopDepth) 
	{
															#line 3221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!(shard < rep.results.size()))
															#line 12973 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			return a_body1loopBody1cont2when2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		results.emplace_back( (toPrefixRelativeRange(rep.results[shard].first, tenant.prefix) & keys), cx->setCachedLocation(rep.results[shard].first, rep.results[shard].second));
															#line 3227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = yield();
															#line 3227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when2loopBody1when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 3;
															#line 3227 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >*>(static_cast<GetKeyRangeLocations_internalActor*>(this)));
															#line 12988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2when2cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 3221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		shard++;
															#line 13010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2when2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 3221 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		shard++;
															#line 13019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont2when2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2when2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2when2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state > 0) static_cast<GetKeyRangeLocations_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyRangeLocations_internalActor*>(this)->ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply const& value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >*,GetKeyServerLocationsReply && value) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Database cx;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TenantInfo tenant;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyRange keys;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int limit;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reverse reverse;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	SpanContext spanContext;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<UID> debugID;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UseProvisionalProxies useProvisionalProxies;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Version version;
															#line 3196 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 3213 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyServerLocationsReply rep;
															#line 3219 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<KeyRangeLocationInfo> results;
															#line 3220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int shard;
															#line 13195 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getKeyRangeLocations_internal()
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetKeyRangeLocations_internalActor final : public Actor<std::vector<KeyRangeLocationInfo>>, public ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >, public ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >, public ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >, public ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >, public FastAllocated<GetKeyRangeLocations_internalActor>, public GetKeyRangeLocations_internalActorState<GetKeyRangeLocations_internalActor> {
															#line 13200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetKeyRangeLocations_internalActor>::operator new;
	using FastAllocated<GetKeyRangeLocations_internalActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3280208419576113664UL, 4180770802477432320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<KeyRangeLocationInfo>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >;
friend struct ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >;
friend struct ActorCallback< GetKeyRangeLocations_internalActor, 2, GetKeyServerLocationsReply >;
friend struct ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >;
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyRangeLocations_internalActor(Database const& cx,TenantInfo const& tenant,KeyRange const& keys,int const& limit,Reverse const& reverse,SpanContext const& spanContext,Optional<UID> const& debugID,UseProvisionalProxies const& useProvisionalProxies,Version const& version) 
															#line 13220 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<std::vector<KeyRangeLocationInfo>>(),
		   GetKeyRangeLocations_internalActorState<GetKeyRangeLocations_internalActor>(cx, tenant, keys, limit, reverse, spanContext, debugID, useProvisionalProxies, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7661675249109580544UL, 8773323650101319936UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyRangeLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKeyRangeLocations_internal");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKeyRangeLocations_internal", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyRangeLocations_internalActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyRangeLocations_internalActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetKeyRangeLocations_internalActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<std::vector<KeyRangeLocationInfo>> getKeyRangeLocations_internal( Database const& cx, TenantInfo const& tenant, KeyRange const& keys, int const& limit, Reverse const& reverse, SpanContext const& spanContext, Optional<UID> const& debugID, UseProvisionalProxies const& useProvisionalProxies, Version const& version ) {
															#line 3186 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<std::vector<KeyRangeLocationInfo>>(new GetKeyRangeLocations_internalActor(cx, tenant, keys, limit, reverse, spanContext, debugID, useProvisionalProxies, version));
															#line 13255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3248 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// Get the SS locations for each shard in the 'keys' key-range;
// Returned vector size is the number of shards in the input keys key-range.
// Returned vector element is <ShardRange, storage server location info> pairs, where
// ShardRange is the whole shard key-range, not a part of the given key range.
// Example: If query the function with  key range (b, d), the returned list of pairs could be something like:
// [([a, b1), locationInfo), ([b1, c), locationInfo), ([c, d1), locationInfo)].
template <class F>
Future<std::vector<KeyRangeLocationInfo>> getKeyRangeLocations(Database const& cx,
                                                               TenantInfo const& tenant,
                                                               KeyRange const& keys,
                                                               int limit,
                                                               Reverse reverse,
                                                               F StorageServerInterface::* member,
                                                               SpanContext const& spanContext,
                                                               Optional<UID> const& debugID,
                                                               UseProvisionalProxies useProvisionalProxies,
                                                               Version version) {

	ASSERT(!keys.empty());

	std::vector<KeyRangeLocationInfo> locations;
	if (!cx->getCachedLocations(tenant, keys, locations, limit, reverse)) {
		return getKeyRangeLocations_internal(
		    cx, tenant, keys, limit, reverse, spanContext, debugID, useProvisionalProxies, version);
	}

	bool foundFailed = false;
	for (const auto& locationInfo : locations) {
		bool onlyEndpointFailedAndNeedRefresh = false;
		for (int i = 0; i < locationInfo.locations->size(); i++) {
			if (checkOnlyEndpointFailed(cx, locationInfo.locations->get(i, member).getEndpoint())) {
				onlyEndpointFailedAndNeedRefresh = true;
			}
		}

		if (onlyEndpointFailedAndNeedRefresh) {
			cx->invalidateCache(tenant.prefix, locationInfo.range.begin);
			foundFailed = true;
		}
	}

	if (foundFailed) {
		// Refresh the cache with a new getKeyRangeLocations made to proxies.
		return getKeyRangeLocations_internal(
		    cx, tenant, keys, limit, reverse, spanContext, debugID, useProvisionalProxies, version);
	}

	return locations;
}

template <class F>
Future<std::vector<KeyRangeLocationInfo>> getKeyRangeLocations(Reference<TransactionState> trState,
                                                               KeyRange const& keys,
                                                               int limit,
                                                               Reverse reverse,
                                                               F StorageServerInterface::* member,
                                                               UseTenant useTenant) {
	CODE_PROBE(!useTenant, "Get key range locations ignoring tenant");
	return getKeyRangeLocations(trState->cx,
	                            useTenant ? trState->getTenantInfo(AllowInvalidTenantID::True) : TenantInfo(),
	                            keys,
	                            limit,
	                            reverse,
	                            member,
	                            trState->spanContext,
	                            trState->readOptions.present() ? trState->readOptions.get().debugID : Optional<UID>(),
	                            trState->useProvisionalProxies,
	                            trState->readVersionFuture.isValid() && trState->readVersionFuture.isReady()
	                                ? trState->readVersion()
	                                : latestVersion);
}

															#line 13332 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getBlobGranuleLocations_internal()
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetBlobGranuleLocations_internalActor>
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetBlobGranuleLocations_internalActorState {
															#line 13339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetBlobGranuleLocations_internalActorState(Database const& cx,TenantInfo const& tenant,KeyRange const& keys,int const& limit,Reverse const& reverse,JustGranules const& justGranules,SpanContext const& spanContext,Optional<UID> const& debugID,UseProvisionalProxies const& useProvisionalProxies,Version const& version,bool* const& more) 
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tenant(tenant),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   keys(keys),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   limit(limit),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   reverse(reverse),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   justGranules(justGranules),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   spanContext(spanContext),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   debugID(debugID),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   useProvisionalProxies(useProvisionalProxies),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   version(version),
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   more(more),
															#line 3333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   span("NAPI:getBlobGranuleLocations"_loc, spanContext)
															#line 13368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this));

	}
	~GetBlobGranuleLocations_internalActorState() 
	{
		fdb_probe_actor_destroy("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3334 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (debugID.present())
															#line 13383 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3335 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", debugID.get().first(), "NativeAPI.getBlobGranuleLocations.Before");
															#line 13387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3337 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(tenant.hasTenant(), "NativeAPI getBlobGranuleLocations has tenant");
															#line 3339 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 13393 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetBlobGranuleLocations_internalActorState();
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3340 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionBlobGranuleLocationRequests;
															#line 3342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = cx->onProxiesChanged();
															#line 3341 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13427 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 3343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetBlobGranuleLocationsReply> __when_expr_1 = basicLoadBalance(cx->getCommitProxies(useProvisionalProxies), &CommitProxyInterface::getBlobGranuleLocations, GetBlobGranuleLocationsRequest(span.context, tenant, keys.begin, keys.end, limit, reverse, justGranules, version, keys.arena()), TaskPriority::DefaultPromiseEndpoint);
															#line 13431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state = 1;
															#line 3342 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >*>(static_cast<GetBlobGranuleLocations_internalActor*>(this)));
															#line 3343 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >*>(static_cast<GetBlobGranuleLocations_internalActor*>(this)));
															#line 13438 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(GetBlobGranuleLocationsReply const& _rep,int loopDepth) 
	{
															#line 3356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionBlobGranuleLocationRequestsCompleted;
															#line 3357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		rep = _rep;
															#line 3358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (debugID.present())
															#line 13469 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "NativeAPI.getBlobGranuleLocations.After");
															#line 13473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(justGranules || rep.results.size());
															#line 3363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(!rep.more || !rep.results.empty());
															#line 3364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		*more = rep.more;
															#line 3366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		results = std::vector<std::pair<KeyRange, UID>>();
															#line 3367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		granule = 0;
															#line 3368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( auto& bwInterf : rep.bwInterfs ) {
															#line 3369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->blobWorker_interf.insert({ bwInterf.id(), bwInterf });
															#line 13489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 13493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1when2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(GetBlobGranuleLocationsReply && _rep,int loopDepth) 
	{
															#line 3356 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionBlobGranuleLocationRequestsCompleted;
															#line 3357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		rep = _rep;
															#line 3358 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (debugID.present())
															#line 13506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3359 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", debugID.get().first(), "NativeAPI.getBlobGranuleLocations.After");
															#line 13510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3362 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(justGranules || rep.results.size());
															#line 3363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ASSERT(!rep.more || !rep.results.empty());
															#line 3364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		*more = rep.more;
															#line 3366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		results = std::vector<std::pair<KeyRange, UID>>();
															#line 3367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		granule = 0;
															#line 3368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		for( auto& bwInterf : rep.bwInterfs ) {
															#line 3369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->blobWorker_interf.insert({ bwInterf.id(), bwInterf });
															#line 13526 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 13530 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1when2loopHead1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state > 0) static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >::remove();
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1when2cont1(int loopDepth) 
	{
															#line 3381 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetBlobGranuleLocations_internalActor*>(this)->SAV<std::vector<std::pair<KeyRange, UID>>>::futures) { (void)(results); this->~GetBlobGranuleLocations_internalActorState(); static_cast<GetBlobGranuleLocations_internalActor*>(this)->destroy(); return 0; }
															#line 13603 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetBlobGranuleLocations_internalActor*>(this)->SAV< std::vector<std::pair<KeyRange, UID>> >::value()) std::vector<std::pair<KeyRange, UID>>(std::move(results)); // state_var_RVO
		this->~GetBlobGranuleLocations_internalActorState();
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1when2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2loopBody1(int loopDepth) 
	{
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!(granule < rep.results.size()))
															#line 13622 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			return a_body1loopBody1when2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3373 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		KeyRange range(toPrefixRelativeRange(rep.results[granule].first, tenant.prefix));
															#line 3374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!justGranules)
															#line 13630 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			range = range & keys;
															#line 13634 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3377 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		results.emplace_back(range, rep.results[granule].second);
															#line 3378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield();
															#line 3378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 13642 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1when2loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state = 2;
															#line 3378 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >*>(static_cast<GetBlobGranuleLocations_internalActor*>(this)));
															#line 13647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1when2cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1when2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		granule++;
															#line 13669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1when2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 3371 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		granule++;
															#line 13678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1when2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1when2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1when2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state > 0) static_cast<GetBlobGranuleLocations_internalActor*>(this)->actor_wait_state = 0;
		static_cast<GetBlobGranuleLocations_internalActor*>(this)->ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >*,GetBlobGranuleLocationsReply const& value) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >*,GetBlobGranuleLocationsReply && value) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >*,Error err) 
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Database cx;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TenantInfo tenant;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyRange keys;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int limit;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reverse reverse;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	JustGranules justGranules;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	SpanContext spanContext;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<UID> debugID;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UseProvisionalProxies useProvisionalProxies;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Version version;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	bool* more;
															#line 3333 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 3357 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetBlobGranuleLocationsReply rep;
															#line 3366 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	std::vector<std::pair<KeyRange, UID>> results;
															#line 3367 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int granule;
															#line 13845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getBlobGranuleLocations_internal()
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetBlobGranuleLocations_internalActor final : public Actor<std::vector<std::pair<KeyRange, UID>>>, public ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >, public ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >, public ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >, public FastAllocated<GetBlobGranuleLocations_internalActor>, public GetBlobGranuleLocations_internalActorState<GetBlobGranuleLocations_internalActor> {
															#line 13850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetBlobGranuleLocations_internalActor>::operator new;
	using FastAllocated<GetBlobGranuleLocations_internalActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15888833822515069184UL, 8782194837760805120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<std::pair<KeyRange, UID>>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >;
friend struct ActorCallback< GetBlobGranuleLocations_internalActor, 1, GetBlobGranuleLocationsReply >;
friend struct ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >;
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetBlobGranuleLocations_internalActor(Database const& cx,TenantInfo const& tenant,KeyRange const& keys,int const& limit,Reverse const& reverse,JustGranules const& justGranules,SpanContext const& spanContext,Optional<UID> const& debugID,UseProvisionalProxies const& useProvisionalProxies,Version const& version,bool* const& more) 
															#line 13869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<std::vector<std::pair<KeyRange, UID>>>(),
		   GetBlobGranuleLocations_internalActorState<GetBlobGranuleLocations_internalActor>(cx, tenant, keys, limit, reverse, justGranules, spanContext, debugID, useProvisionalProxies, version, more),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1026975397208984064UL, 13817401766710445312UL);
		ActorExecutionContextHelper __helper(static_cast<GetBlobGranuleLocations_internalActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getBlobGranuleLocations_internal");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getBlobGranuleLocations_internal", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetBlobGranuleLocations_internalActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetBlobGranuleLocations_internalActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<std::vector<std::pair<KeyRange, UID>>> getBlobGranuleLocations_internal( Database const& cx, TenantInfo const& tenant, KeyRange const& keys, int const& limit, Reverse const& reverse, JustGranules const& justGranules, SpanContext const& spanContext, Optional<UID> const& debugID, UseProvisionalProxies const& useProvisionalProxies, Version const& version, bool* const& more ) {
															#line 3321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<std::vector<std::pair<KeyRange, UID>>>(new GetBlobGranuleLocations_internalActor(cx, tenant, keys, limit, reverse, justGranules, spanContext, debugID, useProvisionalProxies, version, more));
															#line 13903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3386 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

// Get the Blob Worker locations for each granule in the 'keys' key-range, similar to getKeyRangeLocations
Future<std::vector<std::pair<KeyRange, UID>>> getBlobGranuleLocations(Database const& cx,
                                                                      TenantInfo const& tenant,
                                                                      KeyRange const& keys,
                                                                      int limit,
                                                                      Reverse reverse,
                                                                      JustGranules justGranules,
                                                                      SpanContext const& spanContext,
                                                                      Optional<UID> const& debugID,
                                                                      UseProvisionalProxies useProvisionalProxies,
                                                                      Version version,
                                                                      bool* more) {

	ASSERT(!keys.empty());

	// FIXME: wrap this with location caching for blob workers like getKeyRangeLocations has
	return getBlobGranuleLocations_internal(
	    cx, tenant, keys, limit, reverse, justGranules, spanContext, debugID, useProvisionalProxies, version, more);
}

Future<std::vector<std::pair<KeyRange, UID>>> getBlobGranuleLocations(Reference<TransactionState> trState,
                                                                      KeyRange const& keys,
                                                                      int limit,
                                                                      Reverse reverse,
                                                                      UseTenant useTenant,
                                                                      JustGranules justGranules,
                                                                      bool* more) {
	return getBlobGranuleLocations(
	    trState->cx,
	    useTenant ? trState->getTenantInfo(AllowInvalidTenantID::True) : TenantInfo(),
	    keys,
	    limit,
	    reverse,
	    justGranules,
	    trState->spanContext,
	    trState->readOptions.present() ? trState->readOptions.get().debugID : Optional<UID>(),
	    trState->useProvisionalProxies,
	    trState->readVersionFuture.isValid() && trState->readVersionFuture.isReady() ? trState->readVersion()
	                                                                                 : latestVersion,
	    more);
}

															#line 13950 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via warmRange_impl()
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class WarmRange_implActor>
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class WarmRange_implActorState {
															#line 13957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	WarmRange_implActorState(Reference<TransactionState> const& trState,KeyRange const& keys) 
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState),
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   keys(keys),
															#line 3430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   totalRanges(0),
															#line 3431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   totalRequests(0)
															#line 13970 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("warmRange_impl", reinterpret_cast<unsigned long>(this));

	}
	~WarmRange_implActorState() 
	{
		fdb_probe_actor_destroy("warmRange_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = trState->startTransaction();
															#line 3433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<WarmRange_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13987 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WarmRange_implActor*>(this)->actor_wait_state = 1;
															#line 3433 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WarmRange_implActor, 0, Void >*>(static_cast<WarmRange_implActor*>(this)));
															#line 13992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WarmRange_implActorState();
		static_cast<WarmRange_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 14015 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3435 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 14024 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WarmRange_implActor*>(this)->actor_wait_state > 0) static_cast<WarmRange_implActor*>(this)->actor_wait_state = 0;
		static_cast<WarmRange_implActor*>(this)->ActorCallback< WarmRange_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WarmRange_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 3470 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<WarmRange_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WarmRange_implActorState(); static_cast<WarmRange_implActor*>(this)->destroy(); return 0; }
															#line 14108 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<WarmRange_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WarmRange_implActorState();
		static_cast<WarmRange_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 3436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<std::vector<KeyRangeLocationInfo>> __when_expr_1 = getKeyRangeLocations_internal( trState->cx, trState->getTenantInfo(), keys, CLIENT_KNOBS->WARM_RANGE_SHARD_LIMIT, Reverse::False, trState->spanContext, trState->readOptions.present() ? trState->readOptions.get().debugID : Optional<UID>(), trState->useProvisionalProxies, trState->readVersion());
															#line 3436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WarmRange_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14129 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WarmRange_implActor*>(this)->actor_wait_state = 2;
															#line 3436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >*>(static_cast<WarmRange_implActor*>(this)));
															#line 14134 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(std::vector<KeyRangeLocationInfo> const& locations,int loopDepth) 
	{
															#line 3446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		totalRanges += CLIENT_KNOBS->WARM_RANGE_SHARD_LIMIT;
															#line 3447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		totalRequests++;
															#line 3448 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (locations.size() == 0 || totalRanges >= trState->cx->locationCacheSize || locations[locations.size() - 1].range.end >= keys.end)
															#line 14160 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		keys = KeyRangeRef(locations[locations.size() - 1].range.end, keys.end);
															#line 3454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (totalRequests % 20 == 0)
															#line 14168 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr = Transaction(trState->cx, trState->tenant());
															#line 3457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 14174 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(std::vector<KeyRangeLocationInfo> && locations,int loopDepth) 
	{
															#line 3446 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		totalRanges += CLIENT_KNOBS->WARM_RANGE_SHARD_LIMIT;
															#line 3447 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		totalRequests++;
															#line 3448 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (locations.size() == 0 || totalRanges >= trState->cx->locationCacheSize || locations[locations.size() - 1].range.end >= keys.end)
															#line 14192 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		keys = KeyRangeRef(locations[locations.size() - 1].range.end, keys.end);
															#line 3454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (totalRequests % 20 == 0)
															#line 14200 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr = Transaction(trState->cx, trState->tenant());
															#line 3457 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 14206 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(std::vector<KeyRangeLocationInfo> const& locations,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(locations, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(std::vector<KeyRangeLocationInfo> && locations,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(locations), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WarmRange_implActor*>(this)->actor_wait_state > 0) static_cast<WarmRange_implActor*>(this)->actor_wait_state = 0;
		static_cast<WarmRange_implActor*>(this)->ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >::remove();

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >*,std::vector<KeyRangeLocationInfo> const& value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >*,std::vector<KeyRangeLocationInfo> && value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >*,Error err) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 3459 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 3460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			tr.setOption(FDBTransactionOptions::CAUSAL_READ_RISKY);
															#line 3461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = success(tr.getReadVersion());
															#line 3461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<WarmRange_implActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14321 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<WarmRange_implActor*>(this)->actor_wait_state = 3;
															#line 3461 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WarmRange_implActor, 2, Void >*>(static_cast<WarmRange_implActor*>(this)));
															#line 14326 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 3464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<WarmRange_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 14363 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1cont1loopBody1cont1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WarmRange_implActor*>(this)->actor_wait_state = 4;
															#line 3464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WarmRange_implActor, 3, Void >*>(static_cast<WarmRange_implActor*>(this)));
															#line 14368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1cont1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1cont1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WarmRange_implActor*>(this)->actor_wait_state > 0) static_cast<WarmRange_implActor*>(this)->actor_wait_state = 0;
		static_cast<WarmRange_implActor*>(this)->ActorCallback< WarmRange_implActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WarmRange_implActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WarmRange_implActor*>(this)->actor_wait_state > 0) static_cast<WarmRange_implActor*>(this)->actor_wait_state = 0;
		static_cast<WarmRange_implActor*>(this)->ActorCallback< WarmRange_implActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WarmRange_implActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WarmRange_implActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyRange keys;
															#line 3430 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int totalRanges;
															#line 3431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	int totalRequests;
															#line 3456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Transaction tr;
															#line 14563 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via warmRange_impl()
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class WarmRange_implActor final : public Actor<Void>, public ActorCallback< WarmRange_implActor, 0, Void >, public ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >, public ActorCallback< WarmRange_implActor, 2, Void >, public ActorCallback< WarmRange_implActor, 3, Void >, public FastAllocated<WarmRange_implActor>, public WarmRange_implActorState<WarmRange_implActor> {
															#line 14568 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<WarmRange_implActor>::operator new;
	using FastAllocated<WarmRange_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13565239261179497984UL, 1056175794688107264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WarmRange_implActor, 0, Void >;
friend struct ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >;
friend struct ActorCallback< WarmRange_implActor, 2, Void >;
friend struct ActorCallback< WarmRange_implActor, 3, Void >;
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	WarmRange_implActor(Reference<TransactionState> const& trState,KeyRange const& keys) 
															#line 14588 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   WarmRange_implActorState<WarmRange_implActor>(trState, keys),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("warmRange_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8548379736654756608UL, 17106069152090307840UL);
		ActorExecutionContextHelper __helper(static_cast<WarmRange_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("warmRange_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("warmRange_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WarmRange_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WarmRange_implActor, 1, std::vector<KeyRangeLocationInfo> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WarmRange_implActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WarmRange_implActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> warmRange_impl( Reference<TransactionState> const& trState, KeyRange const& keys ) {
															#line 3429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new WarmRange_implActor(trState, keys));
															#line 14624 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3472 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

SpanContext generateSpanID(bool transactionTracingSample, SpanContext parentContext = SpanContext()) {
	if (parentContext.isValid()) {
		return SpanContext(parentContext.traceID, deterministicRandom()->randomUInt64(), parentContext.m_Flags);
	}
	if (transactionTracingSample) {
		return SpanContext(deterministicRandom()->randomUniqueID(),
		                   deterministicRandom()->randomUInt64(),
		                   deterministicRandom()->random01() <= FLOW_KNOBS->TRACING_SAMPLE_RATE
		                       ? TraceFlags::sampled
		                       : TraceFlags::unsampled);
	}
	return SpanContext(
	    deterministicRandom()->randomUniqueID(), deterministicRandom()->randomUInt64(), TraceFlags::unsampled);
}

															#line 14644 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via lookupTenantImpl()
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class LookupTenantImplActor>
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class LookupTenantImplActorState {
															#line 14651 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	LookupTenantImplActorState(DatabaseContext* const& cx,TenantName const& tenant) 
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   tenant(tenant)
															#line 14660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("lookupTenantImpl", reinterpret_cast<unsigned long>(this));

	}
	~LookupTenantImplActorState() 
	{
		fdb_probe_actor_destroy("lookupTenantImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3489 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 14675 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LookupTenantImplActorState();
		static_cast<LookupTenantImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 3491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = cx->getBackoff();
															#line 3491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<LookupTenantImplActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LookupTenantImplActor*>(this)->actor_wait_state = 1;
															#line 3491 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LookupTenantImplActor, 0, Void >*>(static_cast<LookupTenantImplActor*>(this)));
															#line 14713 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3506 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_commit_proxy_memory_limit_exceeded)
															#line 14735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3507 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				CODE_PROBE(true, "Lookup tenant memory limit exceeded");
															#line 3508 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				TraceEvent(SevWarnAlways, "CommitProxyOverloadedForTenant").suppressFor(5);
															#line 3510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->updateBackoff(e);
															#line 14743 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				return a_body1loopHead1(loopDepth); // continue
			}
															#line 3514 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 14748 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 3493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionTenantLookupRequests;
															#line 3495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = cx->onProxiesChanged();
															#line 3494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<LookupTenantImplActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14766 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetTenantIdReply> __when_expr_2 = basicLoadBalance(cx->getCommitProxies(UseProvisionalProxies::False), &CommitProxyInterface::getTenantId, GetTenantIdRequest(tenant, latestVersion), TaskPriority::DefaultPromiseEndpoint);
															#line 14770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when2(__when_expr_2.get(), loopDepth); };
		static_cast<LookupTenantImplActor*>(this)->actor_wait_state = 2;
															#line 3495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LookupTenantImplActor, 1, Void >*>(static_cast<LookupTenantImplActor*>(this)));
															#line 3496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >*>(static_cast<LookupTenantImplActor*>(this)));
															#line 14777 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 3493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionTenantLookupRequests;
															#line 3495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_1 = cx->onProxiesChanged();
															#line 3494 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<LookupTenantImplActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14790 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
															#line 3496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetTenantIdReply> __when_expr_2 = basicLoadBalance(cx->getCommitProxies(UseProvisionalProxies::False), &CommitProxyInterface::getTenantId, GetTenantIdRequest(tenant, latestVersion), TaskPriority::DefaultPromiseEndpoint);
															#line 14794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when2(__when_expr_2.get(), loopDepth); };
		static_cast<LookupTenantImplActor*>(this)->actor_wait_state = 2;
															#line 3495 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LookupTenantImplActor, 1, Void >*>(static_cast<LookupTenantImplActor*>(this)));
															#line 3496 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >*>(static_cast<LookupTenantImplActor*>(this)));
															#line 14801 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LookupTenantImplActor*>(this)->actor_wait_state > 0) static_cast<LookupTenantImplActor*>(this)->actor_wait_state = 0;
		static_cast<LookupTenantImplActor*>(this)->ActorCallback< LookupTenantImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LookupTenantImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LookupTenantImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LookupTenantImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when2(GetTenantIdReply const& rep,int loopDepth) 
	{
															#line 3500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionTenantLookupRequestsCompleted;
															#line 3501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->updateBackoff(success());
															#line 3502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<LookupTenantImplActor*>(this)->SAV<int64_t>::futures) { (void)(rep.tenantId); this->~LookupTenantImplActorState(); static_cast<LookupTenantImplActor*>(this)->destroy(); return 0; }
															#line 14907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<LookupTenantImplActor*>(this)->SAV< int64_t >::value()) int64_t(rep.tenantId);
		this->~LookupTenantImplActorState();
		static_cast<LookupTenantImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when2(GetTenantIdReply && rep,int loopDepth) 
	{
															#line 3500 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++cx->transactionTenantLookupRequestsCompleted;
															#line 3501 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->updateBackoff(success());
															#line 3502 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<LookupTenantImplActor*>(this)->SAV<int64_t>::futures) { (void)(rep.tenantId); this->~LookupTenantImplActorState(); static_cast<LookupTenantImplActor*>(this)->destroy(); return 0; }
															#line 14923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<LookupTenantImplActor*>(this)->SAV< int64_t >::value()) int64_t(rep.tenantId);
		this->~LookupTenantImplActorState();
		static_cast<LookupTenantImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LookupTenantImplActor*>(this)->actor_wait_state > 0) static_cast<LookupTenantImplActor*>(this)->actor_wait_state = 0;
		static_cast<LookupTenantImplActor*>(this)->ActorCallback< LookupTenantImplActor, 1, Void >::remove();
		static_cast<LookupTenantImplActor*>(this)->ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >::remove();

	}
	void a_callback_fire(ActorCallback< LookupTenantImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LookupTenantImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LookupTenantImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >*,GetTenantIdReply const& value) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >*,GetTenantIdReply && value) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >*,Error err) 
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	DatabaseContext* cx;
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TenantName tenant;
															#line 15069 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via lookupTenantImpl()
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class LookupTenantImplActor final : public Actor<int64_t>, public ActorCallback< LookupTenantImplActor, 0, Void >, public ActorCallback< LookupTenantImplActor, 1, Void >, public ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >, public FastAllocated<LookupTenantImplActor>, public LookupTenantImplActorState<LookupTenantImplActor> {
															#line 15074 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<LookupTenantImplActor>::operator new;
	using FastAllocated<LookupTenantImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12124660469276543232UL, 8725177287925019392UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int64_t>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LookupTenantImplActor, 0, Void >;
friend struct ActorCallback< LookupTenantImplActor, 1, Void >;
friend struct ActorCallback< LookupTenantImplActor, 2, GetTenantIdReply >;
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	LookupTenantImplActor(DatabaseContext* const& cx,TenantName const& tenant) 
															#line 15093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<int64_t>(),
		   LookupTenantImplActorState<LookupTenantImplActor>(cx, tenant),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("lookupTenantImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11585582208618864384UL, 12455856959616988672UL);
		ActorExecutionContextHelper __helper(static_cast<LookupTenantImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("lookupTenantImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("lookupTenantImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LookupTenantImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LookupTenantImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<int64_t> lookupTenantImpl( DatabaseContext* const& cx, TenantName const& tenant ) {
															#line 3488 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<int64_t>(new LookupTenantImplActor(cx, tenant));
															#line 15127 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3518 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

Future<int64_t> DatabaseContext::lookupTenant(TenantName tenant) {
	return lookupTenantImpl(this, tenant);
}

TransactionState::TransactionState(Database cx,
                                   Optional<Reference<Tenant>> tenant,
                                   TaskPriority taskID,
                                   SpanContext spanContext,
                                   Reference<TransactionLogInfo> trLogInfo)
  : cx(cx), trLogInfo(trLogInfo), options(cx), taskID(taskID), spanContext(spanContext),
    readVersionObtainedFromGrvProxy(true), tenant_(tenant), tenantSet(tenant.present()) {}

Reference<TransactionState> TransactionState::cloneAndReset(Reference<TransactionLogInfo> newTrLogInfo,
                                                            bool generateNewSpan) const {

	SpanContext newSpanContext = generateNewSpan ? generateSpanID(cx->transactionTracingSample) : spanContext;
	Reference<TransactionState> newState =
	    makeReference<TransactionState>(cx, tenant_, cx->taskID, newSpanContext, newTrLogInfo);

	if (!cx->apiVersionAtLeast(16)) {
		newState->options = options;
	}

	newState->readVersionFuture = Future<Version>();
	newState->metadataVersion = Promise<Optional<Key>>();
	newState->numErrors = numErrors;
	newState->startTime = startTime;
	newState->committedVersion = committedVersion;
	newState->conflictingKeys = conflictingKeys;
	newState->tenantSet = tenantSet;

	return newState;
}

TenantInfo TransactionState::getTenantInfo(AllowInvalidTenantID allowInvalidTenantId /* = false */) {
	Optional<Reference<Tenant>> const& t = tenant();

	if (options.rawAccess) {
		CODE_PROBE(true, "Get tenant info raw access transaction");
		return TenantInfo();
	} else if (!cx->internal && cx->clientInfo->get().clusterType == ClusterType::METACLUSTER_MANAGEMENT) {
		CODE_PROBE(true, "Get tenant info invalid management cluster access", probe::decoration::rare);
		throw management_cluster_invalid_access();
	} else if (!cx->internal && cx->clientInfo->get().tenantMode == TenantMode::REQUIRED && !t.present()) {
		CODE_PROBE(true, "Get tenant info tenant name required", probe::decoration::rare);
		throw tenant_name_required();
	} else if (!t.present()) {
		CODE_PROBE(true, "Get tenant info without tenant");
		return TenantInfo();
	} else if (cx->clientInfo->get().tenantMode == TenantMode::DISABLED && t.present()) {
		// If we are running provisional proxies, we allow a tenant request to go through since we don't know the tenant
		// mode. Such a transaction would not be allowed to commit without enabling provisional commits because either
		// the commit proxies will be provisional or the read version will be too old.
		if (!cx->clientInfo->get().grvProxies.empty() && !cx->clientInfo->get().grvProxies[0].provisional) {
			CODE_PROBE(true, "Get tenant info use tenant in disabled tenant mode", probe::decoration::rare);
			throw tenants_disabled();
		} else {
			CODE_PROBE(true, "Get tenant info provisional proxies");
			ASSERT(!useProvisionalProxies);
		}
	}

	ASSERT(t.present() && (allowInvalidTenantId || t.get()->id() != TenantInfo::INVALID_TENANT));
	return TenantInfo(
	    (allowInvalidTenantId && !t.get()->ready().isReady()) ? TenantInfo::INVALID_TENANT : t.get()->id(), authToken);
}

// Returns the tenant used in this transaction. If the tenant is unset and raw access isn't specified, then the default
// tenant from DatabaseContext is applied to this transaction (note: the default tenant is typically unset, but in
// simulation could be something different).
//
// This function should not be called in the transaction constructor or in the setOption function to allow a user the
// opportunity to set raw access.
Optional<Reference<Tenant>> const& TransactionState::tenant() {
	hasTenant(ResolveDefaultTenant::True);
	return tenant_;
}

// Returns true if the tenant has been set, but does not cause default tenant resolution. This is useful in setOption
// (where we do not want to call tenant()) if we want to enforce that an option not be set on a Tenant transaction (e.g.
// for raw access).
bool TransactionState::hasTenant(ResolveDefaultTenant resolveDefaultTenant) {
	if (!tenantSet && resolveDefaultTenant) {
		if (!options.rawAccess && cx->defaultTenant.present()) {
			tenant_ = makeReference<Tenant>(cx->lookupTenant(cx->defaultTenant.get()), cx->defaultTenant);
		}
		tenantSet = true;
	}

	return tenant_.present();
}

															#line 15224 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via startTransaction()
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class StartTransactionActor>
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class StartTransactionActorState {
															#line 15231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	StartTransactionActorState(Reference<TransactionState> const& trState) 
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState)
															#line 15238 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("startTransaction", reinterpret_cast<unsigned long>(this));

	}
	~StartTransactionActorState() 
	{
		fdb_probe_actor_destroy("startTransaction", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(trState->readVersionFuture);
															#line 3612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<StartTransactionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<StartTransactionActor*>(this)->actor_wait_state = 1;
															#line 3612 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartTransactionActor, 0, Void >*>(static_cast<StartTransactionActor*>(this)));
															#line 15260 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartTransactionActorState();
		static_cast<StartTransactionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->hasTenant())
															#line 15283 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = trState->tenant().get()->ready();
															#line 3614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<StartTransactionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15289 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<StartTransactionActor*>(this)->actor_wait_state = 2;
															#line 3614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StartTransactionActor, 1, Void >*>(static_cast<StartTransactionActor*>(this)));
															#line 15294 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3613 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->hasTenant())
															#line 15308 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_1 = trState->tenant().get()->ready();
															#line 3614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<StartTransactionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15314 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<StartTransactionActor*>(this)->actor_wait_state = 2;
															#line 3614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StartTransactionActor, 1, Void >*>(static_cast<StartTransactionActor*>(this)));
															#line 15319 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartTransactionActor*>(this)->actor_wait_state > 0) static_cast<StartTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<StartTransactionActor*>(this)->ActorCallback< StartTransactionActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartTransactionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartTransactionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartTransactionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 3617 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<StartTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartTransactionActorState(); static_cast<StartTransactionActor*>(this)->destroy(); return 0; }
															#line 15408 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<StartTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartTransactionActorState();
		static_cast<StartTransactionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StartTransactionActor*>(this)->actor_wait_state > 0) static_cast<StartTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<StartTransactionActor*>(this)->ActorCallback< StartTransactionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartTransactionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StartTransactionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StartTransactionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 15505 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via startTransaction()
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class StartTransactionActor final : public Actor<Void>, public ActorCallback< StartTransactionActor, 0, Void >, public ActorCallback< StartTransactionActor, 1, Void >, public FastAllocated<StartTransactionActor>, public StartTransactionActorState<StartTransactionActor> {
															#line 15510 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<StartTransactionActor>::operator new;
	using FastAllocated<StartTransactionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17189972259744483584UL, 8571875239189864192UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartTransactionActor, 0, Void >;
friend struct ActorCallback< StartTransactionActor, 1, Void >;
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	StartTransactionActor(Reference<TransactionState> const& trState) 
															#line 15528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Void>(),
		   StartTransactionActorState<StartTransactionActor>(trState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startTransaction", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7927092769933293824UL, 16612754116029991424UL);
		ActorExecutionContextHelper __helper(static_cast<StartTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startTransaction");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startTransaction", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartTransactionActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StartTransactionActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Void> startTransaction( Reference<TransactionState> const& trState ) {
															#line 3611 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Void>(new StartTransactionActor(trState));
															#line 15562 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

Future<Void> TransactionState::startTransaction(uint32_t readVersionFlags) {
	if (!startFuture.isValid()) {
		if (!readVersionFuture.isValid()) {
			readVersionFuture = getReadVersion(readVersionFlags);
		}
		if (readVersionFuture.isReady() && (!hasTenant() || tenant().get()->ready().isReady())) {
			startFuture = Void();
		} else {
			startFuture = ::startTransaction(Reference<TransactionState>::addRef(this));
		}
	}

	return startFuture;
}

Future<Void> Transaction::warmRange(KeyRange keys) {
	return warmRange_impl(trState, keys);
}

															#line 15586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getValue()
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetValueActor>
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetValueActorState {
															#line 15593 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetValueActorState(Reference<TransactionState> const& trState,Key const& key,UseTenant const& useTenant,TransactionRecordLogInfo const& recordLogInfo) 
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState),
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   key(key),
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   useTenant(useTenant),
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   recordLogInfo(recordLogInfo)
															#line 15606 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getValue", reinterpret_cast<unsigned long>(this));

	}
	~GetValueActorState() 
	{
		fdb_probe_actor_destroy("getValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = trState->startTransaction();
															#line 3643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<GetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15623 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetValueActor*>(this)->actor_wait_state = 1;
															#line 3643 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetValueActor, 0, Void >*>(static_cast<GetValueActor*>(this)));
															#line 15628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetValueActorState();
		static_cast<GetValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		CODE_PROBE(trState->hasTenant(), "NativeAPI getValue has tenant");
															#line 3647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		span = Span("NAPI:getValue"_loc, trState->spanContext);
															#line 3648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (useTenant && trState->hasTenant())
															#line 15655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			span.addAttribute("tenant"_sr, trState->tenant().get()->name.castTo<TenantNameRef>().orDefault("<unspecified>"_sr));
															#line 15659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->validateVersion(trState->readVersion());
															#line 3655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 15665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		CODE_PROBE(trState->hasTenant(), "NativeAPI getValue has tenant");
															#line 3647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		span = Span("NAPI:getValue"_loc, trState->spanContext);
															#line 3648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (useTenant && trState->hasTenant())
															#line 15678 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			span.addAttribute("tenant"_sr, trState->tenant().get()->name.castTo<TenantNameRef>().orDefault("<unspecified>"_sr));
															#line 15682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->validateVersion(trState->readVersion());
															#line 3655 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 15688 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetValueActor*>(this)->actor_wait_state > 0) static_cast<GetValueActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueActor*>(this)->ActorCallback< GetValueActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetValueActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetValueActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 3656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<KeyRangeLocationInfo> __when_expr_1 = getKeyLocation(trState, key, &StorageServerInterface::getValue, Reverse::False, useTenant);
															#line 3656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueActor*>(this)->actor_wait_state = 2;
															#line 3656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >*>(static_cast<GetValueActor*>(this)));
															#line 15786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 3659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		getValueID = Optional<UID>();
															#line 3660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		startTime = uint64_t();
															#line 3661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		startTimeD = double();
															#line 3662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ssLatestCommitVersions = VersionVector();
															#line 3663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		readOptions = trState->readOptions;
															#line 3665 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->getLatestCommitVersions(locationInfo.locations, trState, ssLatestCommitVersions);
															#line 15805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		try {
															#line 3667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (trState->readOptions.present() && trState->readOptions.get().debugID.present())
															#line 15809 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3668 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				getValueID = nondeterministicRandom()->randomUniqueID();
															#line 3669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				readOptions.get().debugID = getValueID;
															#line 3671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addAttach( "GetValueAttachID", trState->readOptions.get().debugID.get().first(), getValueID.get().first());
															#line 3673 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("GetValueDebug", getValueID.get().first(), "NativeAPI.getValue.Before");
															#line 15819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3682 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			++trState->cx->getValueSubmitted;
															#line 3683 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			startTime = timer_int();
															#line 3684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			startTimeD = now();
															#line 3685 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			++trState->cx->transactionPhysicalReads;
															#line 3687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			reply = GetValueReply();
															#line 15831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			try {
															#line 3689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (CLIENT_BUGGIFY_WITH_PROB(.01))
															#line 15835 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 3690 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					return a_body1cont1loopBody1cont1Catch2(deterministicRandom()->randomChoice( std::vector<Error>{ transaction_too_old(), future_version() }), loopDepth);
															#line 15839 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
															#line 3694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_2 = trState->cx->connectionFileChanged();
															#line 3693 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<GetValueActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch2(actor_cancelled(), loopDepth);
															#line 15845 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
															#line 3697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<GetValueReply> __when_expr_3 = loadBalance(trState->cx.getPtr(), locationInfo.locations, &StorageServerInterface::getValue, GetValueRequest(span.context, useTenant ? trState->getTenantInfo() : TenantInfo(), key, trState->readVersion(), trState->cx->sampleReadTags() ? trState->options.readTags : Optional<TagSet>(), readOptions, ssLatestCommitVersions), TaskPriority::DefaultPromiseEndpoint, AtMostOnce::False, trState->cx->enableLocalityLoadBalance ? &trState->cx->queueModel : nullptr, trState->options.enableReplicaConsistencyCheck, trState->options.requiredReplicas);
															#line 15849 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont1when2(__when_expr_3.get(), loopDepth); };
				static_cast<GetValueActor*>(this)->actor_wait_state = 3;
															#line 3694 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetValueActor, 2, Void >*>(static_cast<GetValueActor*>(this)));
															#line 3697 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetValueActor, 3, GetValueReply >*>(static_cast<GetValueActor*>(this)));
															#line 15856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont1loopBody1cont1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont1loopBody1cont1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(KeyRangeLocationInfo const& __locationInfo,int loopDepth) 
	{
															#line 3656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		locationInfo = __locationInfo;
															#line 15877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(KeyRangeLocationInfo && __locationInfo,int loopDepth) 
	{
		locationInfo = std::move(__locationInfo);
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetValueActor*>(this)->actor_wait_state > 0) static_cast<GetValueActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueActor*>(this)->ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >*,KeyRangeLocationInfo const& value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >*,KeyRangeLocationInfo && value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >*,Error err) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3753 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			trState->cx->getValueCompleted->latency = timer_int() - startTime;
															#line 3754 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			trState->cx->getValueCompleted->log();
															#line 3755 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (getValueID.present())
															#line 15967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("GetValueDebug", getValueID.get().first(), "NativeAPI.getValue.Error");
															#line 15971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3764 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_wrong_shard_server || e.code() == error_code_all_alternatives_failed)
															#line 15975 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3765 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				trState->cx->invalidateCache(useTenant ? trState->tenant().mapRef(&Tenant::prefix) : Optional<KeyRef>(), key);
															#line 3767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_4 = delay(CLIENT_KNOBS->WRONG_SHARD_SERVER_DELAY, trState->taskID);
															#line 3767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<GetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1Catch1when1(__when_expr_4.get(), loopDepth); };
				static_cast<GetValueActor*>(this)->actor_wait_state = 4;
															#line 3767 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GetValueActor, 4, Void >*>(static_cast<GetValueActor*>(this)));
															#line 15988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 3769 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (trState->trLogInfo && recordLogInfo)
															#line 15995 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 3770 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					trState->trLogInfo->addLog( FdbClientLogEvents::EventGetError(startTimeD, trState->cx->clientLocality.dcId(), static_cast<int>(e.code()), key, trState->tenant().flatMapRef(&Tenant::name)));
															#line 15999 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				}
															#line 3776 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 16003 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(int loopDepth) 
	{
															#line 3723 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		double latency = now() - startTimeD;
															#line 3724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->readLatencies.addSample(latency);
															#line 3725 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->trLogInfo && recordLogInfo)
															#line 16022 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3726 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			int valueSize = reply.value.present() ? reply.value.get().size() : 0;
															#line 3727 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			trState->trLogInfo->addLog(FdbClientLogEvents::EventGet(startTimeD, trState->cx->clientLocality.dcId(), latency, valueSize, key, trState->tenant().flatMapRef(&Tenant::name)));
															#line 16028 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3734 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->getValueCompleted->latency = timer_int() - startTime;
															#line 3735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->getValueCompleted->log();
															#line 3736 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->totalCost += getReadOperationCost(key.size() + (reply.value.present() ? reply.value.get().size() : 0));
															#line 3739 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (getValueID.present())
															#line 16038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3740 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", getValueID.get().first(), "NativeAPI.getValue.After");
															#line 16042 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3749 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->transactionBytesRead += reply.value.present() ? reply.value.get().size() : 0;
															#line 3750 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++trState->cx->transactionKeysRead;
															#line 3751 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetValueActor*>(this)->SAV<Optional<Value>>::futures) { (void)(reply.value); this->~GetValueActorState(); static_cast<GetValueActor*>(this)->destroy(); return 0; }
															#line 16050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetValueActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(reply.value);
		this->~GetValueActorState();
		static_cast<GetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 3719 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			++trState->cx->transactionPhysicalReadsCompleted;
															#line 3720 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1cont1loopBody1cont1Catch1(__current_error, loopDepth);
															#line 16065 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
															#line 3717 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++trState->cx->transactionPhysicalReadsCompleted;
															#line 16079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
															#line 3695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		return a_body1cont1loopBody1cont1Catch2(transaction_too_old(), loopDepth);
															#line 16088 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
															#line 3695 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		return a_body1cont1loopBody1cont1Catch2(transaction_too_old(), loopDepth);
															#line 16096 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(GetValueReply const& _reply,int loopDepth) 
	{
															#line 3714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		reply = _reply;
															#line 16104 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(GetValueReply && _reply,int loopDepth) 
	{
															#line 3714 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		reply = _reply;
															#line 16113 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetValueActor*>(this)->actor_wait_state > 0) static_cast<GetValueActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueActor*>(this)->ActorCallback< GetValueActor, 2, Void >::remove();
		static_cast<GetValueActor*>(this)->ActorCallback< GetValueActor, 3, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetValueActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetValueActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetValueActor, 3, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetValueActor, 3, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetValueActor, 3, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetValueActor*>(this)->actor_wait_state > 0) static_cast<GetValueActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueActor*>(this)->ActorCallback< GetValueActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GetValueActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GetValueActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Key key;
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UseTenant useTenant;
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	TransactionRecordLogInfo recordLogInfo;
															#line 3647 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 3656 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyRangeLocationInfo locationInfo;
															#line 3659 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<UID> getValueID;
															#line 3660 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	uint64_t startTime;
															#line 3661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	double startTimeD;
															#line 3662 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	VersionVector ssLatestCommitVersions;
															#line 3663 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<ReadOptions> readOptions;
															#line 3687 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetValueReply reply;
															#line 16369 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getValue()
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetValueActor final : public Actor<Optional<Value>>, public ActorCallback< GetValueActor, 0, Void >, public ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >, public ActorCallback< GetValueActor, 2, Void >, public ActorCallback< GetValueActor, 3, GetValueReply >, public ActorCallback< GetValueActor, 4, Void >, public FastAllocated<GetValueActor>, public GetValueActorState<GetValueActor> {
															#line 16374 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetValueActor>::operator new;
	using FastAllocated<GetValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3982525477193542912UL, 11302255394474690048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Value>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetValueActor, 0, Void >;
friend struct ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >;
friend struct ActorCallback< GetValueActor, 2, Void >;
friend struct ActorCallback< GetValueActor, 3, GetValueReply >;
friend struct ActorCallback< GetValueActor, 4, Void >;
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetValueActor(Reference<TransactionState> const& trState,Key const& key,UseTenant const& useTenant,TransactionRecordLogInfo const& recordLogInfo) 
															#line 16395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Optional<Value>>(),
		   GetValueActorState<GetValueActor>(trState, key, useTenant, recordLogInfo),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4269006456623515904UL, 18410873568439542016UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetValueActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetValueActor, 1, KeyRangeLocationInfo >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetValueActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetValueActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Optional<Value>> getValue( Reference<TransactionState> const& trState, Key const& key, UseTenant const& useTenant, TransactionRecordLogInfo const& recordLogInfo ) {
															#line 3639 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Optional<Value>>(new GetValueActor(trState, key, useTenant, recordLogInfo));
															#line 16431 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3781 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 16436 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getKey()
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetKeyActor>
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetKeyActorState {
															#line 16443 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyActorState(Reference<TransactionState> const& trState,KeySelector const& k,UseTenant const& useTenant = UseTenant::True) 
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState),
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   k(k),
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   useTenant(useTenant)
															#line 16454 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getKey", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyActorState() 
	{
		fdb_probe_actor_destroy("getKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3783 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(!useTenant, "Get key ignoring tenant");
															#line 3784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = trState->startTransaction();
															#line 3784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 16473 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyActor*>(this)->actor_wait_state = 1;
															#line 3784 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 0, Void >*>(static_cast<GetKeyActor*>(this)));
															#line 16478 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyActorState();
		static_cast<GetKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		CODE_PROBE(trState->hasTenant(), "NativeAPI getKey has tenant");
															#line 3788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		getKeyID = Optional<UID>();
															#line 3789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		readOptions = trState->readOptions;
															#line 3791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		span = Span("NAPI:getKey"_loc, trState->spanContext);
															#line 3792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->readOptions.present() && trState->readOptions.get().debugID.present())
															#line 16509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			getKeyID = nondeterministicRandom()->randomUniqueID();
															#line 3794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			readOptions.get().debugID = getKeyID;
															#line 3796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addAttach( "GetKeyAttachID", trState->readOptions.get().debugID.get().first(), getKeyID.get().first());
															#line 3798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "GetKeyDebug", getKeyID.get().first(), "NativeAPI.getKey.AfterVersion");
															#line 16519 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 16523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3786 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		CODE_PROBE(trState->hasTenant(), "NativeAPI getKey has tenant");
															#line 3788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		getKeyID = Optional<UID>();
															#line 3789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		readOptions = trState->readOptions;
															#line 3791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		span = Span("NAPI:getKey"_loc, trState->spanContext);
															#line 3792 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->readOptions.present() && trState->readOptions.get().debugID.present())
															#line 16540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3793 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			getKeyID = nondeterministicRandom()->randomUniqueID();
															#line 3794 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			readOptions.get().debugID = getKeyID;
															#line 3796 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addAttach( "GetKeyAttachID", trState->readOptions.get().debugID.get().first(), getKeyID.get().first());
															#line 3798 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent( "GetKeyDebug", getKeyID.get().first(), "NativeAPI.getKey.AfterVersion");
															#line 16550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3805 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		;
															#line 16554 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyActor*>(this)->actor_wait_state > 0) static_cast<GetKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 3806 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (k.getKey() == allKeys.end)
															#line 16645 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3807 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (k.offset > 0)
															#line 16649 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3808 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (!static_cast<GetKeyActor*>(this)->SAV<Key>::futures) { (void)(allKeys.end); this->~GetKeyActorState(); static_cast<GetKeyActor*>(this)->destroy(); return 0; }
															#line 16653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				new (&static_cast<GetKeyActor*>(this)->SAV< Key >::value()) Key(allKeys.end);
				this->~GetKeyActorState();
				static_cast<GetKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 3810 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			k.orEqual = false;
															#line 16661 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		else
		{
															#line 3811 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (k.getKey() == allKeys.begin && k.offset <= 0)
															#line 16667 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3812 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (!static_cast<GetKeyActor*>(this)->SAV<Key>::futures) { (void)(Key()); this->~GetKeyActorState(); static_cast<GetKeyActor*>(this)->destroy(); return 0; }
															#line 16671 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				new (&static_cast<GetKeyActor*>(this)->SAV< Key >::value()) Key(Key());
				this->~GetKeyActorState();
				static_cast<GetKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
															#line 3815 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		Key locationKey(k.getKey(), k.arena());
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<KeyRangeLocationInfo> __when_expr_1 = getKeyLocation( trState, locationKey, &StorageServerInterface::getKey, Reverse{ k.isBackward() }, useTenant);
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16684 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyActor*>(this)->actor_wait_state = 2;
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >*>(static_cast<GetKeyActor*>(this)));
															#line 16689 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 3819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		ssLatestCommitVersions = VersionVector();
															#line 3820 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		trState->cx->getLatestCommitVersions(locationInfo.locations, trState, ssLatestCommitVersions);
															#line 16700 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		try {
															#line 3823 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (getKeyID.present())
															#line 16704 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3824 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent( "GetKeyDebug", getKeyID.get().first(), "NativeAPI.getKey.Before");
															#line 16708 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3829 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			++trState->cx->transactionPhysicalReads;
															#line 3831 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			GetKeyRequest req(span.context, useTenant ? trState->getTenantInfo() : TenantInfo(), k, trState->readVersion(), trState->cx->sampleReadTags() ? trState->options.readTags : Optional<TagSet>(), readOptions, ssLatestCommitVersions);
															#line 3838 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			req.arena.dependsOn(k.arena());
															#line 3840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			reply = GetKeyReply();
															#line 16718 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			try {
															#line 3843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_2 = trState->cx->connectionFileChanged();
															#line 3842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont1Catch2(actor_cancelled(), loopDepth);
															#line 16724 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
															#line 3846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<GetKeyReply> __when_expr_3 = loadBalance(trState->cx.getPtr(), locationInfo.locations, &StorageServerInterface::getKey, req, TaskPriority::DefaultPromiseEndpoint, AtMostOnce::False, trState->cx->enableLocalityLoadBalance ? &trState->cx->queueModel : nullptr, trState->options.enableReplicaConsistencyCheck, trState->options.requiredReplicas);
															#line 16728 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1cont1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont1when2(__when_expr_3.get(), loopDepth); };
				static_cast<GetKeyActor*>(this)->actor_wait_state = 3;
															#line 3843 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 2, Void >*>(static_cast<GetKeyActor*>(this)));
															#line 3846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 3, GetKeyReply >*>(static_cast<GetKeyActor*>(this)));
															#line 16735 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont1loopBody1cont1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont1loopBody1cont1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(KeyRangeLocationInfo const& __locationInfo,int loopDepth) 
	{
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		locationInfo = __locationInfo;
															#line 16756 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(KeyRangeLocationInfo && __locationInfo,int loopDepth) 
	{
		locationInfo = std::move(__locationInfo);
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyActor*>(this)->actor_wait_state > 0) static_cast<GetKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >*,KeyRangeLocationInfo const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >*,KeyRangeLocationInfo && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3874 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (getKeyID.present())
															#line 16842 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3875 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("GetKeyDebug", getKeyID.get().first(), "NativeAPI.getKey.Error");
															#line 16846 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3876 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_wrong_shard_server || e.code() == error_code_all_alternatives_failed)
															#line 16850 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3877 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				trState->cx->invalidateCache(useTenant ? trState->tenant().mapRef(&Tenant::prefix) : Optional<KeyRef>(), k.getKey(), Reverse{ k.isBackward() });
															#line 3881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_4 = delay(CLIENT_KNOBS->WRONG_SHARD_SERVER_DELAY, trState->taskID);
															#line 3881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<GetKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16858 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1Catch1when1(__when_expr_4.get(), loopDepth); };
				static_cast<GetKeyActor*>(this)->actor_wait_state = 4;
															#line 3881 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GetKeyActor, 4, Void >*>(static_cast<GetKeyActor*>(this)));
															#line 16863 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 3883 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				TraceEvent(SevInfo, "GetKeyError").error(e).detail("AtKey", k.getKey()).detail("Offset", k.offset);
															#line 3884 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 16872 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(int loopDepth) 
	{
															#line 3864 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (getKeyID.present())
															#line 16887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3865 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent("GetKeyDebug", getKeyID.get().first(), "NativeAPI.getKey.After");
															#line 16891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3869 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		k = reply.sel;
															#line 3870 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!k.offset && k.orEqual)
															#line 16897 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3871 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<GetKeyActor*>(this)->SAV<Key>::futures) { (void)(k.getKey()); this->~GetKeyActorState(); static_cast<GetKeyActor*>(this)->destroy(); return 0; }
															#line 16901 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<GetKeyActor*>(this)->SAV< Key >::value()) Key(k.getKey());
			this->~GetKeyActorState();
			static_cast<GetKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont1loopBody1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 3861 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			++trState->cx->transactionPhysicalReadsCompleted;
															#line 3862 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			return a_body1cont1loopBody1cont1Catch1(__current_error, loopDepth);
															#line 16918 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
															#line 3859 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		++trState->cx->transactionPhysicalReadsCompleted;
															#line 16932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
															#line 3844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		return a_body1cont1loopBody1cont1Catch2(transaction_too_old(), loopDepth);
															#line 16941 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
															#line 3844 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		return a_body1cont1loopBody1cont1Catch2(transaction_too_old(), loopDepth);
															#line 16949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(GetKeyReply const& _reply,int loopDepth) 
	{
															#line 3856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		reply = _reply;
															#line 16957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when2(GetKeyReply && _reply,int loopDepth) 
	{
															#line 3856 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		reply = _reply;
															#line 16966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetKeyActor*>(this)->actor_wait_state > 0) static_cast<GetKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 2, Void >::remove();
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 3, GetKeyReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 3, GetKeyReply >*,GetKeyReply const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 3, GetKeyReply >*,GetKeyReply && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 3, GetKeyReply >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont11(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetKeyActor*>(this)->actor_wait_state > 0) static_cast<GetKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyActor*>(this)->ActorCallback< GetKeyActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GetKeyActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GetKeyActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeySelector k;
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	UseTenant useTenant;
															#line 3788 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<UID> getKeyID;
															#line 3789 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Optional<ReadOptions> readOptions;
															#line 3791 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 3816 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	KeyRangeLocationInfo locationInfo;
															#line 3819 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	VersionVector ssLatestCommitVersions;
															#line 3840 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyReply reply;
															#line 17229 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getKey()
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetKeyActor final : public Actor<Key>, public ActorCallback< GetKeyActor, 0, Void >, public ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >, public ActorCallback< GetKeyActor, 2, Void >, public ActorCallback< GetKeyActor, 3, GetKeyReply >, public ActorCallback< GetKeyActor, 4, Void >, public FastAllocated<GetKeyActor>, public GetKeyActorState<GetKeyActor> {
															#line 17234 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetKeyActor>::operator new;
	using FastAllocated<GetKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3042623995765574656UL, 9107435528990151936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyActor, 0, Void >;
friend struct ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >;
friend struct ActorCallback< GetKeyActor, 2, Void >;
friend struct ActorCallback< GetKeyActor, 3, GetKeyReply >;
friend struct ActorCallback< GetKeyActor, 4, Void >;
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetKeyActor(Reference<TransactionState> const& trState,KeySelector const& k,UseTenant const& useTenant = UseTenant::True) 
															#line 17255 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Key>(),
		   GetKeyActorState<GetKeyActor>(trState, k, useTenant),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3161838746568296704UL, 16159820804706226432UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyActor, 1, KeyRangeLocationInfo >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetKeyActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetKeyActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Key> getKey( Reference<TransactionState> const& trState, KeySelector const& k, UseTenant const& useTenant = UseTenant::True ) {
															#line 3782 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Key>(new GetKeyActor(trState, k, useTenant));
															#line 17291 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3889 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 17296 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForCommittedVersion()
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class WaitForCommittedVersionActor>
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class WaitForCommittedVersionActorState {
															#line 17303 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	WaitForCommittedVersionActorState(Database const& cx,Version const& version,SpanContext const& spanContext) 
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   version(version),
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   spanContext(spanContext),
															#line 3891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   span("NAPI:waitForCommittedVersion"_loc, spanContext)
															#line 17316 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForCommittedVersion", reinterpret_cast<unsigned long>(this));

	}
	~WaitForCommittedVersionActorState() 
	{
		fdb_probe_actor_destroy("waitForCommittedVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3892 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 17331 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForCommittedVersionActorState();
		static_cast<WaitForCommittedVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 3895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_0 = cx->onProxiesChanged();
															#line 3894 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17364 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 3896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<GetReadVersionReply> __when_expr_1 = basicLoadBalance( cx->getGrvProxies(UseProvisionalProxies::False), &GrvProxyInterface::getConsistentReadVersion, GetReadVersionRequest( span.context, 0, TransactionPriority::IMMEDIATE, cx->ssVersionVectorCache.getMaxVersion()), cx->taskID);
															#line 17368 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 1;
															#line 3895 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForCommittedVersionActor, 0, Void >*>(static_cast<WaitForCommittedVersionActor*>(this)));
															#line 3896 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >*>(static_cast<WaitForCommittedVersionActor*>(this)));
															#line 17375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3920 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_batch_transaction_throttled || e.code() == error_code_grv_proxy_memory_limit_exceeded)
															#line 17397 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_3 = delayJittered(CLIENT_KNOBS->GRV_ERROR_RETRY_DELAY);
															#line 3923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
				static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 3;
															#line 3923 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitForCommittedVersionActor, 3, Void >*>(static_cast<WaitForCommittedVersionActor*>(this)));
															#line 17408 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 3925 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				TraceEvent(SevError, "WaitForCommittedVersionError").error(e);
															#line 3926 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 17417 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(GetReadVersionReply const& v,int loopDepth) 
	{
															#line 3902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->minAcceptableReadVersion = std::min(cx->minAcceptableReadVersion, v.version);
															#line 3903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (v.midShardSize > 0)
															#line 17452 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->smoothMidShardSize.setTotal(v.midShardSize);
															#line 17456 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (cx->versionVectorCacheActive(v.ssVersionVectorDelta))
															#line 17460 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (cx->isCurrentGrvProxy(v.proxyId))
															#line 17464 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->ssVersionVectorCache.applyDelta(v.ssVersionVectorDelta);
															#line 17468 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 3909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->ssVersionVectorCache.clear();
															#line 17474 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
															#line 3912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (v.version >= version)
															#line 17479 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<WaitForCommittedVersionActor*>(this)->SAV<Version>::futures) { (void)(v.version); this->~WaitForCommittedVersionActorState(); static_cast<WaitForCommittedVersionActor*>(this)->destroy(); return 0; }
															#line 17483 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<WaitForCommittedVersionActor*>(this)->SAV< Version >::value()) Version(v.version);
			this->~WaitForCommittedVersionActorState();
			static_cast<WaitForCommittedVersionActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(CLIENT_KNOBS->FUTURE_VERSION_RETRY_DELAY, cx->taskID);
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17493 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1when2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 2;
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitForCommittedVersionActor, 2, Void >*>(static_cast<WaitForCommittedVersionActor*>(this)));
															#line 17498 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(GetReadVersionReply && v,int loopDepth) 
	{
															#line 3902 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		cx->minAcceptableReadVersion = std::min(cx->minAcceptableReadVersion, v.version);
															#line 3903 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (v.midShardSize > 0)
															#line 17509 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3904 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->smoothMidShardSize.setTotal(v.midShardSize);
															#line 17513 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 3905 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (cx->versionVectorCacheActive(v.ssVersionVectorDelta))
															#line 17517 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3906 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (cx->isCurrentGrvProxy(v.proxyId))
															#line 17521 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->ssVersionVectorCache.applyDelta(v.ssVersionVectorDelta);
															#line 17525 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 3909 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->ssVersionVectorCache.clear();
															#line 17531 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
															#line 3912 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (v.version >= version)
															#line 17536 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3913 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (!static_cast<WaitForCommittedVersionActor*>(this)->SAV<Version>::futures) { (void)(v.version); this->~WaitForCommittedVersionActorState(); static_cast<WaitForCommittedVersionActor*>(this)->destroy(); return 0; }
															#line 17540 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			new (&static_cast<WaitForCommittedVersionActor*>(this)->SAV< Version >::value()) Version(v.version);
			this->~WaitForCommittedVersionActorState();
			static_cast<WaitForCommittedVersionActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(CLIENT_KNOBS->FUTURE_VERSION_RETRY_DELAY, cx->taskID);
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1when2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 2;
															#line 3916 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitForCommittedVersionActor, 2, Void >*>(static_cast<WaitForCommittedVersionActor*>(this)));
															#line 17555 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForCommittedVersionActor*>(this)->ActorCallback< WaitForCommittedVersionActor, 0, Void >::remove();
		static_cast<WaitForCommittedVersionActor*>(this)->ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForCommittedVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1when2cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1when2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1when2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForCommittedVersionActor*>(this)->ActorCallback< WaitForCommittedVersionActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitForCommittedVersionActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >*,GetReadVersionReply const& value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >*,GetReadVersionReply && value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >*,Error err) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForCommittedVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForCommittedVersionActor*>(this)->ActorCallback< WaitForCommittedVersionActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitForCommittedVersionActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitForCommittedVersionActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Database cx;
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Version version;
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	SpanContext spanContext;
															#line 3891 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 17882 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via waitForCommittedVersion()
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class WaitForCommittedVersionActor final : public Actor<Version>, public ActorCallback< WaitForCommittedVersionActor, 0, Void >, public ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >, public ActorCallback< WaitForCommittedVersionActor, 2, Void >, public ActorCallback< WaitForCommittedVersionActor, 3, Void >, public FastAllocated<WaitForCommittedVersionActor>, public WaitForCommittedVersionActorState<WaitForCommittedVersionActor> {
															#line 17887 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<WaitForCommittedVersionActor>::operator new;
	using FastAllocated<WaitForCommittedVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2493000804877269504UL, 13704428998262777088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForCommittedVersionActor, 0, Void >;
friend struct ActorCallback< WaitForCommittedVersionActor, 1, GetReadVersionReply >;
friend struct ActorCallback< WaitForCommittedVersionActor, 2, Void >;
friend struct ActorCallback< WaitForCommittedVersionActor, 3, Void >;
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	WaitForCommittedVersionActor(Database const& cx,Version const& version,SpanContext const& spanContext) 
															#line 17907 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForCommittedVersionActorState<WaitForCommittedVersionActor>(cx, version, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9303840678792357376UL, 476365445843715840UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForCommittedVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForCommittedVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForCommittedVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForCommittedVersionActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitForCommittedVersionActor, 2, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitForCommittedVersionActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Version> waitForCommittedVersion( Database const& cx, Version const& version, SpanContext const& spanContext ) {
															#line 3890 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Version>(new WaitForCommittedVersionActor(cx, version, spanContext));
															#line 17942 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3931 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 17947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via getRawVersion()
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class GetRawVersionActor>
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetRawVersionActorState {
															#line 17954 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetRawVersionActorState(Reference<TransactionState> const& trState) 
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : trState(trState),
															#line 3933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   span("NAPI:getRawVersion"_loc, trState->spanContext)
															#line 17963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("getRawVersion", reinterpret_cast<unsigned long>(this));

	}
	~GetRawVersionActorState() 
	{
		fdb_probe_actor_destroy("getRawVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3934 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 17978 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRawVersionActorState();
		static_cast<GetRawVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_0 = trState->cx->onProxiesChanged();
															#line 3935 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<GetRawVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18010 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 3937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<GetReadVersionReply> __when_expr_1 = basicLoadBalance(trState->cx->getGrvProxies(UseProvisionalProxies::False), &GrvProxyInterface::getConsistentReadVersion, GetReadVersionRequest(trState->spanContext, 0, TransactionPriority::IMMEDIATE, trState->cx->ssVersionVectorCache.getMaxVersion()), trState->cx->taskID);
															#line 18014 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<GetRawVersionActor*>(this)->actor_wait_state = 1;
															#line 3936 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRawVersionActor, 0, Void >*>(static_cast<GetRawVersionActor*>(this)));
															#line 3937 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >*>(static_cast<GetRawVersionActor*>(this)));
															#line 18021 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(GetReadVersionReply const& v,int loopDepth) 
	{
															#line 3945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->cx->versionVectorCacheActive(v.ssVersionVectorDelta))
															#line 18048 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (trState->cx->isCurrentGrvProxy(v.proxyId))
															#line 18052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				trState->cx->ssVersionVectorCache.applyDelta(v.ssVersionVectorDelta);
															#line 18056 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 3949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				trState->cx->ssVersionVectorCache.clear();
															#line 18062 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
															#line 3952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetRawVersionActor*>(this)->SAV<Version>::futures) { (void)(v.version); this->~GetRawVersionActorState(); static_cast<GetRawVersionActor*>(this)->destroy(); return 0; }
															#line 18067 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetRawVersionActor*>(this)->SAV< Version >::value()) Version(v.version);
		this->~GetRawVersionActorState();
		static_cast<GetRawVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(GetReadVersionReply && v,int loopDepth) 
	{
															#line 3945 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (trState->cx->versionVectorCacheActive(v.ssVersionVectorDelta))
															#line 18079 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 3946 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (trState->cx->isCurrentGrvProxy(v.proxyId))
															#line 18083 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3947 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				trState->cx->ssVersionVectorCache.applyDelta(v.ssVersionVectorDelta);
															#line 18087 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
			else
			{
															#line 3949 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				trState->cx->ssVersionVectorCache.clear();
															#line 18093 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
		}
															#line 3952 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (!static_cast<GetRawVersionActor*>(this)->SAV<Version>::futures) { (void)(v.version); this->~GetRawVersionActorState(); static_cast<GetRawVersionActor*>(this)->destroy(); return 0; }
															#line 18098 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		new (&static_cast<GetRawVersionActor*>(this)->SAV< Version >::value()) Version(v.version);
		this->~GetRawVersionActorState();
		static_cast<GetRawVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRawVersionActor*>(this)->actor_wait_state > 0) static_cast<GetRawVersionActor*>(this)->actor_wait_state = 0;
		static_cast<GetRawVersionActor*>(this)->ActorCallback< GetRawVersionActor, 0, Void >::remove();
		static_cast<GetRawVersionActor*>(this)->ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetRawVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRawVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRawVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >*,GetReadVersionReply const& value) 
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >*,GetReadVersionReply && value) 
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >*,Error err) 
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Reference<TransactionState> trState;
															#line 3933 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	Span span;
															#line 18231 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
};
// This generated class is to be used only via getRawVersion()
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class GetRawVersionActor final : public Actor<Version>, public ActorCallback< GetRawVersionActor, 0, Void >, public ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >, public FastAllocated<GetRawVersionActor>, public GetRawVersionActorState<GetRawVersionActor> {
															#line 18236 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
	using FastAllocated<GetRawVersionActor>::operator new;
	using FastAllocated<GetRawVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2169049661455160064UL, 5004266191532823296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRawVersionActor, 0, Void >;
friend struct ActorCallback< GetRawVersionActor, 1, GetReadVersionReply >;
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	GetRawVersionActor(Reference<TransactionState> const& trState) 
															#line 18254 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		 : Actor<Version>(),
		   GetRawVersionActorState<GetRawVersionActor>(trState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRawVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6451071362418865408UL, 866680570832095232UL);
		ActorExecutionContextHelper __helper(static_cast<GetRawVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRawVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRawVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRawVersionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
[[nodiscard]] Future<Version> getRawVersion( Reference<TransactionState> const& trState ) {
															#line 3932 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	return Future<Version>(new GetRawVersionActor(trState));
															#line 18287 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
}

#line 3957 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 18292 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
[[nodiscard]] Future<Void> readVersionBatcher( DatabaseContext* const& cx, FutureStream<std::pair<Promise<GetReadVersionReply>, Optional<UID>>> const& versionStream, uint32_t const& flags );

#line 3962 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"

															#line 18297 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
namespace {
// This generated class is to be used only via watchValue()
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
template <class WatchValueActor>
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
class WatchValueActorState {
															#line 18304 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
public:
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
	WatchValueActorState(Database const& cx,Reference<const WatchParameters> const& parameters) 
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		 : cx(cx),
															#line 3963 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   parameters(parameters),
															#line 3964 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   span("NAPI:watchValue"_loc, parameters->spanContext),
															#line 3965 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		   ver(parameters->version)
															#line 18317 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
	{
		fdb_probe_actor_create("watchValue", reinterpret_cast<unsigned long>(this));

	}
	~WatchValueActorState() 
	{
		fdb_probe_actor_destroy("watchValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3966 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			cx->validateVersion(parameters->version);
															#line 3967 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			ASSERT(parameters->version != latestVersion);
															#line 3969 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			CODE_PROBE(parameters->tenant.hasTenant(), "NativeAPI watchValue has tenant");
															#line 3971 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			;
															#line 18338 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchValueActorState();
		static_cast<WatchValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<KeyRangeLocationInfo> __when_expr_0 = getKeyLocation(cx, parameters->tenant, parameters->key, &StorageServerInterface::watchValue, parameters->spanContext, parameters->debugID, parameters->useProvisionalProxies, Reverse::False, parameters->version);
															#line 3972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18370 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<WatchValueActor*>(this)->actor_wait_state = 1;
															#line 3972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 0, KeyRangeLocationInfo >*>(static_cast<WatchValueActor*>(this)));
															#line 18375 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		try {
															#line 3982 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			watchValueID = Optional<UID>();
															#line 3983 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (parameters->debugID.present())
															#line 18387 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 3984 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				watchValueID = nondeterministicRandom()->randomUniqueID();
															#line 3986 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addAttach( "WatchValueAttachID", parameters->debugID.get().first(), watchValueID.get().first());
															#line 3988 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				g_traceBatch.addEvent("WatchValueDebug", watchValueID.get().first(), "NativeAPI.watchValue.Before");
															#line 18395 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			}
															#line 3992 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			resp = WatchValueReply();
															#line 3994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<WatchValueReply> __when_expr_1 = loadBalance(cx.getPtr(), locationInfo.locations, &StorageServerInterface::watchValue, WatchValueRequest(span.context, parameters->tenant, parameters->key, parameters->value, ver, cx->sampleReadTags() ? parameters->tags : Optional<TagSet>(), watchValueID), TaskPriority::DefaultPromiseEndpoint);
															#line 3993 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 18403 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 4008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			StrictFuture<Void> __when_expr_2 = cx->connectionRecord ? cx->connectionRecord->onChange() : Never();
															#line 18407 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont1when2(__when_expr_2.get(), loopDepth); };
			static_cast<WatchValueActor*>(this)->actor_wait_state = 2;
															#line 3994 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 1, WatchValueReply >*>(static_cast<WatchValueActor*>(this)));
															#line 4008 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 2, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18414 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(KeyRangeLocationInfo const& __locationInfo,int loopDepth) 
	{
															#line 3972 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		locationInfo = __locationInfo;
															#line 18429 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(KeyRangeLocationInfo && __locationInfo,int loopDepth) 
	{
		locationInfo = std::move(__locationInfo);
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchValueActor*>(this)->actor_wait_state > 0) static_cast<WatchValueActor*>(this)->actor_wait_state = 0;
		static_cast<WatchValueActor*>(this)->ActorCallback< WatchValueActor, 0, KeyRangeLocationInfo >::remove();

	}
	void a_callback_fire(ActorCallback< WatchValueActor, 0, KeyRangeLocationInfo >*,KeyRangeLocationInfo const& value) 
	{
		fdb_probe_actor_enter("watchValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueActor, 0, KeyRangeLocationInfo >*,KeyRangeLocationInfo && value) 
	{
		fdb_probe_actor_enter("watchValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchValueActor, 0, KeyRangeLocationInfo >*,Error err) 
	{
		fdb_probe_actor_enter("watchValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValue", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4038 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			if (e.code() == error_code_wrong_shard_server || e.code() == error_code_all_alternatives_failed)
															#line 18515 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
			{
															#line 4039 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				cx->invalidateCache(parameters->tenant.prefix, parameters->key);
															#line 4040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				StrictFuture<Void> __when_expr_5 = delay(CLIENT_KNOBS->WRONG_SHARD_SERVER_DELAY, parameters->taskID);
															#line 4040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18523 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when1(__when_expr_5.get(), loopDepth); };
				static_cast<WatchValueActor*>(this)->actor_wait_state = 5;
															#line 4040 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 5, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18528 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 4041 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
				if (e.code() == error_code_watch_cancelled || e.code() == error_code_process_behind)
															#line 18535 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
				{
															#line 4043 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					CODE_PROBE(e.code() == error_code_watch_cancelled, "Too many watches on the storage server, poll for changes instead");
															#line 4044 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					CODE_PROBE(e.code() == error_code_process_behind, "The storage servers are all behind", probe::decoration::rare);
															#line 4046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					StrictFuture<Void> __when_expr_6 = delay(CLIENT_KNOBS->WATCH_POLLING_TIME, parameters->taskID);
															#line 4046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18545 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when2(__when_expr_6.get(), loopDepth); };
					static_cast<WatchValueActor*>(this)->actor_wait_state = 6;
															#line 4046 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 6, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18550 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 4047 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
					if (e.code() == error_code_timed_out)
															#line 18557 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
					{
															#line 4049 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						CODE_PROBE(true, "A watch timed out");
															#line 4050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						StrictFuture<Void> __when_expr_7 = delay(CLIENT_KNOBS->FUTURE_VERSION_RETRY_DELAY, parameters->taskID);
															#line 4050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18565 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
						if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when3(__when_expr_7.get(), loopDepth); };
						static_cast<WatchValueActor*>(this)->actor_wait_state = 7;
															#line 4050 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 7, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18570 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
															#line 4052 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						err = e;
															#line 4053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						StrictFuture<Void> __when_expr_8 = delay(CLIENT_KNOBS->FUTURE_VERSION_RETRY_DELAY, parameters->taskID);
															#line 4053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18581 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
						if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1Catch1when4(__when_expr_8.get(), loopDepth); };
						static_cast<WatchValueActor*>(this)->actor_wait_state = 8;
															#line 4053 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
						__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 8, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18586 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
						loopDepth = 0;
					}
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 4012 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (watchValueID.present())
															#line 18604 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		{
															#line 4013 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", watchValueID.get().first(), "NativeAPI.watchValue.After");
															#line 18608 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		}
															#line 4019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Version> __when_expr_4 = waitForCommittedVersion(cx, resp.version, span.context);
															#line 4019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 18614 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_4.get(), loopDepth); };
		static_cast<WatchValueActor*>(this)->actor_wait_state = 4;
															#line 4019 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 4, Version >*>(static_cast<WatchValueActor*>(this)));
															#line 18619 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(WatchValueReply const& r,int loopDepth) 
	{
															#line 4006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		resp = r;
															#line 18628 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(WatchValueReply && r,int loopDepth) 
	{
															#line 4006 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		resp = r;
															#line 18637 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when2(Void const& _,int loopDepth) 
	{
															#line 4009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = Never();
															#line 4009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 18648 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont1when2when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchValueActor*>(this)->actor_wait_state = 3;
															#line 4009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 3, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18653 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when2(Void && _,int loopDepth) 
	{
															#line 4009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		StrictFuture<Void> __when_expr_3 = Never();
															#line 4009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		if (static_cast<WatchValueActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont1Catch1(actor_cancelled(), loopDepth);
															#line 18664 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont1when2when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchValueActor*>(this)->actor_wait_state = 3;
															#line 4009 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/fdbclient/NativeAPI.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchValueActor, 3, Void >*>(static_cast<WatchValueActor*>(this)));
															#line 18669 "/codebuild/output/src3820056588/src/github.com/apple/foundationdb/build_output/fdbclient/NativeAPI.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WatchValueActor*>(this)->actor_wait_state > 0) static_cast<WatchValueActor*>(this)->actor_wait_state = 0;
		static_cast<WatchValueActor*>(this)->ActorCallback< WatchValueActor, 1, WatchValueReply >::remove();
		static_cast<WatchValueActor*>(this)->ActorCallback< WatchValueActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchValueActor, 1, WatchValueReply >*,WatchValueReply const& value) 
	{
		fdb_probe_actor_enter("watchValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueActor, 1, WatchValueReply >*,WatchValueRe